function ret = plTspTarget(fcn, opt2, opt3, opt4)

currentCircuit = plecs('get', '', 'CurrentCircuit');
aggregationMode = plecs('get', currentCircuit, 'AggregationMode');
isCodeGenSim = strcmp(aggregationMode, 'CodeGen');

% The 'timingTarget' is the top level target structure, which always is the one
% that contains TaskConfigurations and SAMPLE_TIME.
timingTarget = plecs('get', '.', 'CompiledCodeGenTarget');

% In the case of a CodeGen simulation, the 'OriginalTarget' holds the 
% information we care about. We store the info in 'target', which will have
% everything we care about from a TSP perspective.
if isCodeGenSim && ~isempty(timingTarget) && isfield(timingTarget, 'OriginalTarget')
  target = timingTarget.OriginalTarget;
else 
  target = timingTarget;
end
    
switch(fcn)
  case 'targetFound' % returns true if a 'CompiledCodeGenTarget' was found
    % used to detect 'limbo' target blocks not in a subsystem
    ret = ~isempty(target) && ~isempty(fieldnames(target));

  case 'matches' % (targets)
    targets = opt2;
    name = plTspTarget('getName');
    
    if strcmp(name, 'TI2837xS') 
        % handle special case of deprecated target
        name = 'TI2837x';
    end
    ret = any(strcmp(name, targets));

  % Note for 'get' functions, here we will provide minimal validation:
  % If a defaultValue is provided, we will return it and not error.
  % If no defaultValue is provided, we perform minimal validation and 
  % return the value fetched for success and return NaN, or some other safe value.
  % Calling code to should check the return value for NaN and return in that case.

  case 'getFamily'
    if isfield(target, 'Family')
      ret = target.Family;
    else
      ret = 'No Target Family configured.';
    end
 
  case 'getName'
    if isfield(target, 'Name')
      ret = target.Name;
    else
      ret = 'No Target Name Configured.';
    end

  case 'isCodeGenSim'
    ret = isCodeGenSim;

  case 'getVar' % (varNameStr, defaultValue)
    varNameStr = opt2;
    if (nargin < 3)
        defaultValue = NaN;
    else
        defaultValue = opt3;
    end
    ret = defaultValue;

    if isfield(target, 'Variables') && isfield(target.Variables, varNameStr)
        ret = getfield(target.Variables, varNameStr);
        if isnan(ret) % empty field gets the default value
            ret = defaultValue;
        end
    end

  case 'getTimingVar' % (varNameStr, defaultValue)
    varNameStr = opt2;
    if (nargin < 3)
        defaultValue = NaN;
    else
        defaultValue = opt3;
    end
    ret = defaultValue;

    if isfield(timingTarget, 'Variables') && isfield(timingTarget.Variables, varNameStr)
        ret = getfield(timingTarget.Variables, varNameStr);
        if isnan(ret) % empty field gets the default value
            ret = defaultValue;
        end
    end

  case 'getSysClkHz' % (defaultValue)
    if (nargin < 2)
        defaultValue = NaN;
    else
        defaultValue = opt2;
    end

    fMHz = plTspTarget('getVar', 'sysClkMHz'); %% TI uses this name
    if ~isnan(fMHz)
        ret = fMHz * 1e6;
        return;
    end
    fMHz = plTspTarget('getVar', 'SysClkMHz'); %% Other targets use UpperCamelCase
    if ~isnan(fMHz)
        ret = fMHz * 1e6;
        return;
    end
    ret = defaultValue;
  
  case 'getSysClkWidgetName'
    if strcmp(plTspTarget('getFamily'), 'TI C2000')
      ret = 'sysClkMHz';
    else
      ret = 'SysClkMHz';
    end

  case 'getSAMPLE_TIME'
    ret = plTspTarget('getTimingVar', 'SAMPLE_TIME');

  case 'logSAMPLE_TIME_error' % (prefix)
   if (nargin < 2)
        prefix = 'A ';
    else
        prefix = [opt2 ' a '];
    end
    modelPath = plTspTarget('getModelPath');

    if isfield(timingTarget, 'TaskConfigurations')
      plecs('error', [prefix 'positive scalar value is required for the base task sample time in the @param:TaskConfigurations:2: table.'], modelPath);
    else
      plecs('error', [prefix 'positive scalar value is required for the @param:CodeGenSampleTime:2:.'], modelPath);
    end
    ret = true; % return boolean for the test

  case 'getModelPath' % path used for Coder Options widgets
    ret =  plecs('get', '.', 'ParentCodeGenSystem');
  
  case 'getMatlabScriptsVersion'
    ret = 3; % version number of these embedded/shared/TSP_Lib scripts

  otherwise
    plecs('error', ['Invalid first argument to plTspTarget(): ''' fcn '''']);

end

end
