close all;clc;
disp('Loading parameters...')

Load_profile = 2; % Choice of load profile: 1 = constant, 2 = load torque variation
Tsim = 8; % Simulation time span
Tres = 1e-3; % Time resolution of returned simulations results
Nf_a = 4.97e13; % Lifetime parameter a
Nf_b = 4.61; % Lifetime parameter b
plots = 'yes'; plecsblue = [0 164 227]/255; plecsred = [233 61 69]/255; % Show plots (yes/no), define colors
Load_torque_freq = 0.2; % Frequency of the load torque variation

if (Load_profile == 1)
   time_window = 0.1; % Analyze only the last 10% in steady-state
else
   time_window = 1/Tsim/Load_torque_freq; % Analyze only the last periodic load cycle
end

% Set up model variables and solver options for PLECS simulation
OptStruct.ModelVars.Tsim = Tsim;
OptStruct.ModelVars.Load_profile = Load_profile;
OptStruct.ModelVars.Load_torque_freq = Load_torque_freq;
OptStruct.SolverOpts.OutputTimes = 0:Tres:Tsim;

disp('=== Perform simulation of junction temperatures...')

SimOut = sim('rainflow_counting.slx','StopTime',num2str(Tsim));
out.Time = SimOut.results.time(round(end*(1-time_window)):end);
out.Values = SimOut.results.signals.values(round(end*(1-time_window)):end);

disp('=== Find temperature peaks and valleys...')

ids = find_peaks(out.Values);
t_ex = out.Time(ids); % save time points of extrema
Tj_ex = out.Values(ids); % save values of extrema

if (strcmp(plots,'yes'))
   plot(out.Time, out.Values,'Color',plecsblue);hold on
   plot(t_ex, Tj_ex, '*','Color',plecsred); % Plot extrema
   title('Transient junction temperature and peak points')
   xlabel('Time (s)')
   ylabel('Junction temperature (degC)')
end

disp('=== Perform rainflow counting algorithm...')

[cyc_ampl, cyc_freq] = rainflow_algorithm(t_ex,Tj_ex);

disp('=== Generate histogram...')

[hists, bins] = gen_histogram(cyc_ampl, cyc_freq);

if (strcmp(plots,'yes'))
   bins(bins==0) = []; figure; h = bar(bins,hists); set(h, 'facecolor', plecsblue, 'edgecolor', plecsred);
   title('Histogram of temperature cycles - delta Tj (bin width = 5 degC)')
   xlabel('Bins for delta Tj')
   ylabel('Number of cycles that fall into bin')
end

disp('=== Calculate device lifetime...')

lifetime = predict_lifetime(time_window*Tsim, Nf_a, Nf_b, hists, bins);

disp(['=== The lifetime of IGBT module is ' num2str(lifetime) ' year(s).']);