function circuitInfo = connectivity(handle)

visitedConnectors = [];

circuitInfo = struct(...
    'handle', {}, ...
    'ports', struct(...
        'portName', {}, ...
        'potential', {}));

portInfo = iGetPortInfo(handle);

for portIdx = 1:length(portInfo),
    cctIdx = find([circuitInfo.handle] == portInfo(portIdx).circuitHandle, 1);
    if isempty(cctIdx),
        cctIdx = length(circuitInfo) + 1;
        circuitInfo(cctIdx).handle = portInfo(portIdx).circuitHandle;
    end
    circuitInfo(cctIdx).ports(end+1) = struct(...
        'portName', portInfo(portIdx).portName, ...
        'potential', portInfo(portIdx).potential);
end

function result = iIsActive(circuit)
    previous = circuit;
    parent = get_param(circuit, 'Parent');
    while strcmp(get_param(parent, 'Type'), 'block'),
        if strcmp(get_param(parent, 'Variant'), 'on'),
            if ~strcmp(get_param(parent, 'ActiveVariantBlock'), previous)
                result = false;
                return;
            end
        end
        previous = parent;
        parent = get_param(parent, 'Parent');
    end
    
    result = true;
end

function portInfo = iGetPortInfo(handle)

portInfo = struct(...
    'portHandle', {}, ...
    'portName', {}, ...
    'circuitHandle', {}, ...
    'potential', {});

mdl = bdroot(handle);

if verLessThan('matlab', '9.9')
   circuits = find_system(mdl, ...
       'LookUnderMasks', 'all', ...
       'FollowLinks', 'on', ...
       'MaskType', 'PlecsCircuit');
else
   circuits = find_system(mdl, ...
       'MatchFilter', @Simulink.match.activeVariants, ...
       'LookUnderMasks', 'all', ...
       'FollowLinks', 'on', ...
       'MaskType', 'PlecsCircuit');
end

if isempty(circuits),
    return;
end

circuitsToVisit = get_param(circuits, 'Handle');
if iscell(circuitsToVisit),
    circuitsToVisit = cell2mat(circuitsToVisit);
end
visitedCircuits = [];

potential = 1;
while ~isempty(circuitsToVisit),
    % iterate over circuit list in reverse order
    circuit = circuitsToVisit(end);
    circuitsToVisit(end) = [];
    if ~iIsActive(circuit)
        continue
    end
    
    ports = get_param(circuit, 'Ports');
    startIdx = cumsum([1 ports]);
    numPorts = ports(6)+ports(7);
    
    pH = get_param(circuit, 'PortHandles');
    pH = [pH.LConn pH.RConn];
    pC = get_param(circuit, 'PortConnectivity');
    pC = pC(startIdx(6)+(0:(numPorts-1)));
    
    potentialUsed = false;
    
    for srcIdx = 1:numPorts,
        if isempty(pC(srcIdx).DstPort),
            continue;
        end

        [dstBlocks, dstPorts] = iFindDestination(...
            pC(srcIdx).DstBlock, ...
            pC(srcIdx).DstPort);
        
        if isempty(dstPorts),
            continue;
        end
        
        if isempty(find([portInfo.portHandle] == pH(srcIdx), 1)),
            srcPortName = iGetPortName(circuit, pH(srcIdx));
            portInfo(end+1) = struct(...
                'portHandle', pH(srcIdx), ...
                'portName', srcPortName, ...
                'circuitHandle', circuit, ...
                'potential', potential);
            
            potentialUsed = true;
        end
        
        for dstIdx = 1:length(dstPorts),
            if isempty(find([portInfo.portHandle] == dstPorts(dstIdx), 1)),
                dstPortName = iGetPortName(...
                    dstBlocks(dstIdx), ...
                    dstPorts(dstIdx));
                portInfo(end+1) = struct(...
                    'portHandle', dstPorts(dstIdx), ...
                    'portName', dstPortName, ...
                    'circuitHandle', dstBlocks(dstIdx), ...
                    'potential', potential);
                
                potentialUsed = true;
            end
        end
        
        if (potentialUsed),
            potential = potential + 1;
        end
    end
    
    visitedCircuits(end+1) = circuit; 
end

end


function portName = iGetPortName(block, port)
if ~isnumeric(block)
    block = get_param(block, 'Handle');
end
portName = '';
portHandles = get_param(block, 'PortHandles');
idx = find(portHandles.LConn == port, 1);
if ~isempty(idx),
    portBlocks = find_system(block, ...
        'LookUnderMasks', 'all', ...
        'FollowLinks', 'on', ...
        'SearchDepth', 1, ...
        'BlockType', 'PMIOPort', ...
        'Side', 'Left');
    portName = get_param(portBlocks(idx), 'Name');
end
idx = find(portHandles.RConn == port, 1);
if ~isempty(idx),
    portBlocks = find_system(block, ...
        'LookUnderMasks', 'all', ...
        'FollowLinks', 'on', ...
        'SearchDepth', 1, ...
        'BlockType', 'PMIOPort', ...
        'Side', 'Right');
    portName = get_param(portBlocks(idx), 'Name');
end
end


function [dstBlocks, dstPorts] = iFindDestination(inBlocks, inPorts)

dstBlocks = [];
dstPorts = [];

for idx = 1:length(inBlocks),
    block = inBlocks(idx);
    port = inPorts(idx);

    % skip blocks that are commented out
    if isfield(get_param(block, 'ObjectParameters'), 'Commented') ...
       && ~strcmp(get_param(block, 'Commented'), 'off'),
        continue
    end
    
    switch lower(get_param(block, 'BlockType')),
        case 'subsystem'
            switch get_param(block, 'MaskType'),
                case 'PlecsCircuit'
                    dstBlocks(end+1) = block;
                    dstPorts(end+1) = port;
                case 'PlecsConnector'
                    if ~any(visitedConnectors == block),
                        visitedConnectors(end+1) = block;
                        pc = get_param(block, 'PortConnectivity');
                        ph = get_param(block, 'PortHandles');
                        ph = [ph.LConn ph.RConn];
                        for idx = 1:length(pc),
                            if ph(idx) ~= port,
                                [thisDstBlocks, thisDstPorts] = iFindDestination(...
                                    pc(idx).DstBlock, ...
                                    pc(idx).DstPort);
                                dstBlocks = [dstBlocks thisDstBlocks]; %#ok<*AGROW>
                                dstPorts = [dstPorts thisDstPorts];
                            end
                        end
                    end
                otherwise
                    % find peer PMIOPort in SubSystem
                    portHandles = get_param(block, 'PortHandles');
                    portIdx = find(portHandles.LConn == port, 1);
                    if ~isempty(portIdx),
                        pmioPorts = find_system(block, ...
                            'LookUnderMasks', 'all', ...
                            'FollowLinks', 'on', ...
                            'SearchDepth', 1, ...
                            'BlockType', 'PMIOPort', ...
                            'Side', 'Left');
                    else
                        portIdx = find(portHandles.RConn == port, 1);
                        if ~isempty(portIdx),
                            pmioPorts = find_system(block, ...
                                'LookUnderMasks', 'all', ...
                                'FollowLinks', 'on', ...
                                'SearchDepth', 1, ...
                                'BlockType', 'PMIOPort', ...
                                'Side', 'Right');
                        end
                    end
                    port = pmioPorts(portIdx);
                    
                    if strcmp(get_param(block, 'Variant'), 'on'),
                        block = get_param(block, 'ActiveVariantBlock');
                        port = [block '/' get_param(port, 'Name')];
                    end
                    
                    portConn = get_param(port, 'PortConnectivity');
                    [thisDstBlocks, thisDstPorts] = iFindDestination(...
                        portConn.DstBlock, ...
                        portConn.DstPort);
                    dstBlocks = [dstBlocks thisDstBlocks]; %#ok<*AGROW>
                    dstPorts = [dstPorts thisDstPorts];
            end
        case 'pmioport'
            % find peer port of parent subsystem
            parent = get_param(block, 'Parent');
            parent = get_param(parent, 'Handle');
            side = get_param(block, 'Side');
            pmioPorts = find_system(parent, ...
                'LookUnderMasks', 'all', ...
                'FollowLinks', 'on', ...
                'SearchDepth', 1, ...
                'BlockType', 'PMIOPort', ...
                'Side', side);
            portIdx = find(pmioPorts == block);
            if strcmp(side, 'Left'),
                type = ['LConn' num2str(portIdx)];
            else
                type = ['RConn' num2str(portIdx)];
            end
            
            if strcmp(get_param(parent, 'Variant'), 'off') ...
                    && ~isempty(get_param(parent, 'VariantControl')),
                parent = get_param(parent, 'Parent');
            end

            portConn = get_param(parent, 'PortConnectivity');
            portIdx = find(strcmp({portConn.Type}, type), 1);
            if ~isempty(portIdx),
                [thisDstBlocks, thisDstPorts] = iFindDestination(...
                    portConn(portIdx).DstBlock, ...
                    portConn(portIdx).DstPort);
                dstBlocks = [dstBlocks thisDstBlocks]; %#ok<*AGROW>
                dstPorts = [dstPorts thisDstPorts];
            end
        otherwise
            error('Cannot connect to foreign block "%s".', getfullname(block))
    end
    
end

end

end

