%plinstall PLECS Installer.
%
% adds PLECS to the MATLAB path

% Copyright 2011, Plexim GmbH

function plinstall

try
  [matlabVer, count] = sscanf(version, '%i.%i.%i');
  if [1 1e-3 1e-6]*matlabVer(:) < 6.005,
    error('This installer requires MATLAB version 6.5.0 or newer.');
  end

  installDir = fileparts(fileparts(mfilename('fullpath')));
  v = iParseContentsFile(fullfile(installDir, 'Contents.m'));
  myPLECSVer = sscanf(v, '%i.%i.%i');


  minMatlabVer = iDeterminePlatform(matlabVer);
  if ([1 1e-3 1e-6]*matlabVer(:) < [1 1e-3 1e-6]*minMatlabVer(:)),
    error('PLECS requires MATLAB version %i.%i.%i or newer.', minMatlabVer);
  end

  [oldPLECSDir, oldPLECSVer] = iCheckForOldPLECS(installDir);

 
  if isempty(oldPLECSDir),
    iSetMatlabPath(installDir);
  else
    fprintf('Rehashing toolbox cache\n');
    rehash toolboxcache
  end

catch
  err=lasterror;
  waitfor(errordlg(...
    sprintf(['Adding PLECS to the MATLAB path for MATLAB version %i.%i.%i failed: ''%s''.\n\n' ...
    'You must add the directory ''%s'' to the MATLAB path manually.'], ...
    matlabVer, err.message, installDir), ...
    'Installation failed', 'modal'));
  return;
end

fprintf('Done.\n');
fprintf('PLECS version %i.%i.%i installed successfully.\n\n', myPLECSVer);

return;



function minMatlabVer = iDeterminePlatform(matlabVer)

switch mexext,
  case {'dll', 'mexw32'}
    if matlabVer(1) > 7 || (matlabVer(1) == 7 && matlabVer(2) >= 3),
      minMatlabVer = [7, 3, 0];
    else
      minMatlabVer = [6, 5, 0];
    end
  case {'mexw64'}
    minMatlabVer = [7, 4, 0];
  case {'mexglx'}
    minMatlabVer = [7, 4, 0];
  case {'mexa64'}
    minMatlabVer = [7, 4, 0];
  case {'mexmaci'}
    minMatlabVer = [7, 4, 0];
  case {'mexmaci64'}
    minMatlabVer = [7, 9, 0];
  otherwise
    error('Unsupported platform ''%s''.', computer);
end


function [oldPLECSDir, oldPLECSVer] = iCheckForOldPLECS(installPath)

oldPLECSVer = [];

oldPLECSDir = fileparts(which('plecsedit'));
if isempty(oldPLECSDir),
  return;
end
if strcmp(oldPLECSDir, installPath),
  fprintf('This PLECS installation is already on the MATLAB path.\n');
  return;
end

v = iParseContentsFile(fullfile(oldPLECSDir, 'Contents.m'));
oldPLECSVer = sscanf(v, '%i.%i.%i');

fprintf('Found PLECS version %s on your MATLAB path.\n', v);
answer = questdlg(...
  sprintf(['PLECS version %i.%i.%i is already installed at ''%s''. ' ...
    'Do you want to remove this installation from your MATLAB path?'], ...
    oldPLECSVer, oldPLECSDir), ...
  'Overwrite existing PLECS?', ...
  'Yes', 'Cancel', 'Cancel');

if strcmp(answer, 'Cancel'),
  error('PLECS is already installed at ''%s''. Remove first!', oldPLECSDir);
end

[m, mex] = inmem;                                                     %#ok
if (~isempty(strmatch('plecsedit', mex, 'exact'))),
  fprintf('Clearing PLECS module...\n');
  plecsclear;
end

fprintf('Removing PLECS installation at ''%s''.\n', oldPLECSDir);

rmpath(oldPLECSDir);
rmpath(fullfile(oldPLECSDir, 'demos'));
oldPLECSDir = '';

function v = iParseContentsFile(file)

fp = fopen(file,'rt');
if fp>0
  s = fgetl(fp); %#ok
  s = fgetl(fp);
  fclose(fp);
  if ~ischar(s)
    s = '';
  end
else
  s = '';
end

c = strread(s, '%s', 'delimiter', ' ');
if numel(c)<3 || ~strcmpi(c{2}, 'version'),
  error('Error parsing contents file ''%s''.', filename);
end

v = c{3};

function iSetMatlabPath(installDir)

fprintf('Adding PLECS to your MATLAB path...\n');

addpath(installDir);
addpath(fullfile(installDir, 'demos'));

status = 1;
if any(exist('savepath') == [2 3 5 6])
  status = savepath;
elseif any(exist('path2rc') == [2 3 5 6]),
  status = path2rc;
end
if (status),
  msg = 'Unable to save MATLAB path. You will need to manually save the path!';
  fprintf('\n');
  warning(msg); %#ok
  fprintf('\n');
  warndlg(msg, 'Unable to save MATLAB path', 'modal');
end
