import socket
import time
import base64				# deserialise MAT-file contents for JSON-RPC

# The RT Box listens to XML-RPC connections on TCP port 9998
# The following Python code initiates an XML-RPC connection to the RT Box:
    
HOST_NAME       = "rtbox-123.local"
ip              = socket.gethostbyname(HOST_NAME)
HOST_ADDRESS	= "http://" + ip + ":9998/RPC2"
MODEL_NAME		= "rlc_network_scripting"		
METHOD			= "XML"		# choose "XML" or "JSON"

#Initialize server and load executable
if METHOD == "JSON":
    import jsonrpc_requests		# for JSON-RPC
    import collections.abc      # to make jsonrpc_requests usable for Python 3.10+
    collections.Mapping = collections.abc.Mapping
    server = jsonrpc_requests.Server(HOST_ADDRESS)
elif METHOD == "XML":
    import xmlrpc.client        # for XML-RPC
    server = xmlrpc.client.Server(HOST_ADDRESS)
    
with open("rlc_network_scripting_codegen/" + MODEL_NAME + ".elf", "rb") as f:
    print("Uploading executable")
    server.rtbox.load(base64.b64encode(f.read()).decode())
f.closed
    
print("Starting executable")
server.rtbox.start()
print("Real-time simulation running")

#  List of all Programmable Value and Data Capture blocks in the current model
inputblocks = server.rtbox.getProgrammableValueBlocks()
outputblocks = server.rtbox.getDataCaptureBlocks()
print("Available input blocks are:")
print(inputblocks)
print("Available output blocks are:")
print(outputblocks)

# Sending data to the real-time simulation
Vin = 2
print("Setting Vin as %.2f" % Vin + "V")
server.rtbox.setProgrammableValue('Input', [Vin])

# Getting data from the real-time simulation
while server.rtbox.getCaptureTriggerCount('Capture1')==0:
 print("Waiting for data")
 time.sleep(1)
data1 = server.rtbox.getCaptureData('Capture1')
data2 = server.rtbox.getCaptureData('Capture2')

Vm = [row[0] for row in data1['data']]
Am = [row[1] for row in data1['data']]
VmMax = data2['data'][0][0]

print("Stopping executable")
server.rtbox.stop();
print("Real-time simulation stopped")

print("Max value of Vm = %.2f" % VmMax + "V")


# Optionally, the voltmeter and ammeter readings can be plotted

import matplotlib.pyplot as plt
plt.close('all')
x = [i * data1['sampleTime'] for i in range(0, len(data1['data']))]
fig, ax1 = plt.subplots()
ax1.plot(x, Vm, 'b')
ax1.set_xlabel('Time (s)')
ax1.set_ylabel('Vm (V)', color='b')
ax1.tick_params('y', colors='b')
ax2 = ax1.twinx()
ax2.plot(x, Am, 'r')
ax2.set_ylabel('Am (A)', color='r')
ax2.tick_params('y', colors='r')
plt.title("Voltmeter and Ammeter Readings")
plt.show()
