% Power Factor Efficiency Sweep at Rated Current
% for three_phase_t_type_inverter.mdl demo model
 
% Determine sim start time
tic

% Define line frequency and cursor times
F = 60; t1 = 1/F; t2 = 2/F;

% Name of scope used to extract signals
thermalScope = 'three_phase_t_type_inverter/Circuit/Thermal Analysis';

% Clear all previous traces in scope in the current model
plecs('scope', thermalScope , 'ClearTraces');

% Define Cell Array of the different modulation schemes.
modSchemeNames = {'SPWM','SVPWM','THIPWM','THZSPWM','DPWM'};

% Iterate over five PF setpoints and two modulation schemes (SPWM,SVPWM) by default.
results.phi =  linspace(-1,1,5);  % power factor angle setpoint in per unit (phi/pi)
modAnalyzeIdxs = [1,2];

% Uncommment the settings below to reproduce the losses/temperature vs PF sweeping result (Fig. 7 in the demo documentation)
% results.phi =  linspace(-1,1,13); 
% modAnalyzeIdxs = 1:length(modSchemeNames)

% Allocate loss and thermal arrays
condLoss   = zeros(size(results.phi));
swLoss     = zeros(size(results.phi));
totalLoss  = zeros(size(results.phi));
maxTempSA1 = zeros(size(results.phi));
maxTempSA2 = zeros(size(results.phi));
assertSA1  = zeros(size(results.phi));
assertSA2  = zeros(size(results.phi));

% Iterate over the desired modulation schemese
for modIdx = modAnalyzeIdxs
    modName = modSchemeNames{modIdx};
    fprintf('Running simulation with %s modulation scheme: \n', modName) 

    % Iterate over different phase angle set points
    for phiSetIdx = 1:length(results.phi)
        phiPuSet=results.phi(phiSetIdx);

        fprintf('--Sim start %s modulation and %4.2f pu angle. \n', modName, phiPuSet) 
   	  
        %Set modulation type and phase angle setpoint
        modScheme = modIdx;
        phi = phiPuSet*pi;

        % Since there is a strong thermal dependence on efficiency, we should use:
        plsteadystate('three_phase_t_type_inverter/Steady-State Analysis');

		% Store the scope data for analysis
     	scopeData = plecs('scope', thermalScope, 'GetCursorData', [t1 t2], 'mean', 'max'); 

        % Get average device losses over one period
     	condLoss(phiSetIdx)  = scopeData.cursorData{1}{1}.mean;
     	swLoss(phiSetIdx)    = scopeData.cursorData{1}{2}.mean;
     	totalLoss(phiSetIdx) = scopeData.cursorData{1}{3}.mean;
    	% Get peak junction tempeature over one period    	
     	maxTempSA1(phiSetIdx) = scopeData.cursorData{2}{1}.max;
     	maxTempSA2(phiSetIdx) = scopeData.cursorData{2}{2}.max; 	
    	% Determine if any assertions were generated by looking at mean value of trace
        assertSA1(phiSetIdx) = any(cellfun(@(x) x.mean, scopeData.cursorData{3})<1);
        assertSA2(phiSetIdx) = any(cellfun(@(x) x.mean, scopeData.cursorData{4})<1);
   end
   
   % Copy results for each modulation approach into structure.
   results.(modName).condLoss = condLoss;
   results.(modName).swLoss   = swLoss;
   results.(modName).totalLoss = totalLoss;
   results.(modName).maxTempSA1 = maxTempSA1;
   results.(modName).maxTempSA2 = maxTempSA2;
   results.(modName).assertSA1 = assertSA1;
   results.(modName).assertSA2 = assertSA2;
   
end

%Export numeric results for post processing.
save('ThermalSweepResults.mat','results')

%%%%%%%%%%%%%%%%%%%%%%%%
% Begin Plotting
%%%%%%%%%%%%%%%%%%%%%%%%

figure 
colorLegend = 'grbmc'; % Color assigned to each modulation scheme
subplot (2, 1, 1)
legendHandle = [];     % Placeholder for legend
for modIdx = modAnalyzeIdxs
    hold on
    modName=modSchemeNames{modIdx};
    % Plot loss curve.  Retain axis handle for legend.
    hax = plot(results.phi,results.(modName).totalLoss,colorLegend(modIdx));
    legendHandle = [legendHandle,hax];
    % Create curve marking cases with any assertions with a red "X"
    assertFilt = logical(results.(modName).assertSA1 | results.(modName).assertSA2 );
    plot(results.phi(assertFilt),results.(modName).totalLoss(assertFilt),'xr')
end

title ('Total Losses');
xlabel ('Power Factor Angle \phi/\pi (pu)');
ylabel ('Total Loss (W)');
grid on;
legend (legendHandle,modSchemeNames(modAnalyzeIdxs),'location','southoutside','orientation','horizontal');
legend boxoff;

subplot (2, 1, 2)
% Legend name and handles
thermLegendA1 = [];
thermLegendA2 = [];
legendHandleA1 = [];
legendHandleA2 = [];

for modIdx = modAnalyzeIdxs
    hold on
    modName=modSchemeNames{modIdx};
    % Plot temperature curves. Retain axis handle for legend.
    haxA1 = plot(results.phi,results.(modName).maxTempSA1,[colorLegend(modIdx),'-']);
    haxA2 = plot(results.phi,results.(modName).maxTempSA2,[colorLegend(modIdx),'--']);

    % Create separate legend entries for SA1 and SA2
    legendHandleA1 = [legendHandleA1,haxA1];
    legendHandleA2 = [legendHandleA2,haxA2]; 
    thermLegendA1 = [thermLegendA1, strcat({'S1 '},modName)];
    thermLegendA2 = [thermLegendA2, strcat({'S2 '},modName)];
  
    % Create curve marking cases with any assertions with a red "X"
    assertFiltSA1 = logical(results.(modName).assertSA1);
    assertFiltSA2 = logical(results.(modName).assertSA2);
    plot(results.phi(assertFiltSA1),results.(modName).maxTempSA1(assertFiltSA1),'xr')
    plot(results.phi(assertFiltSA2),results.(modName).maxTempSA2(assertFiltSA2),'xr')
end

title ('Junction Temperatures');
xlabel ('Power Factor Angle \phi/\pi (pu)');
ylabel ('Temperature (C)');
grid on;

% Create and align thermal legends for SA1 and SA2
ah1 = gca; 
ah2 = axes('position',get(gca,'position'), 'visible','off');
lA1 = legend(ah1,legendHandleA1,thermLegendA1,'location','southoutside','orientation','horizontal');
lA2 = legend(ah2,legendHandleA2,thermLegendA2,'location','southoutside','orientation','horizontal');
% Shift legend for A1 temps upwards
lA1pos = get (lA1, 'position');
set(lA1,'position', lA1pos + [0 0.04 0 0.0])
set(lA1,'box', 'off')
set(lA2,'box', 'off')

set (0, 'defaultaxesfontname', 'DIN Offc')
set (0, 'defaultaxesfontsize', 12) 

% Save figure as PDF 
print('ThermalSweepResults','-dpdf','-fillpage')

% Determine sim end time
toc
