function b = plTspAssert(fcn, opt2, opt3, opt4) % indexed to align with inputname

modelPath = plTspTarget('getModelPath');

switch(fcn)
  case 'isIntegerInClosedIntervalHex' % (val, minLimit, maxLimit)
    val = opt2;
    minLimit = opt3;
    maxLimit = opt4;
    maskLink = maskLinkFromStr(inputname(2));

    if  ~(  plCheck('integer', val) ...
         && plCheck('inClosedInterval',val, minLimit, maxLimit) )
        plecs('error', [maskLink ' must be a scalar or vector of' ...
                        ' integers in the range ' ...
                        range2StrHex(minLimit, maxLimit) '.']);
        b = false; return;
    end
    b = true;

  case 'targetFamilyIs' %(familyName)
    % This check is intended to enforce the selected 'Target' is correct
    % for the block.
    % Note in CodeGen Simulation mode this check will pass and default values
    % must still be provided for now.
    % This check is pulling double duty to check for limbo blocks that
    % exist outside a CodeGen subsystem (in a model with a CodeGen subsystem),
    % which will have no 'CompiledCodeGenTarget' information.
    desiredFamily = opt2;

    % check for limbo blocks
    if ~plTspTarget('targetFound')
      plecs('error', ...
        ['In a model containing a subsystem with code generation enabled, ' ...
         'all target blocks must be placed in a subsystem configured '...
         'for code generation.']);
      b = false; return;
    end

    % Check for matching Target Family
    if ~strcmp(plTspTarget('getFamily'), desiredFamily)

      plecs('error', ...
        ['This block can only be used in a model or subsystem configured '...
         'to generate code for a @param:CodeGenTarget:2: in '...
         'the ''' desiredFamily ''' family.'], modelPath);
        b = false; return;
    end
    b = true;

  case 'targetMatches' %(targets, targetDescriptionStr)
    % Same as 'targetFamilyIs' above, but checks specific targets, not just
    % the family.
    % The arg `targetDescriptionStr` is used to format the list of options
    % in a human readable way.
    targets = opt2;
    targetDescriptionStr = opt3;

    % check for limbo blocks
    if ~plTspTarget('targetFound')
      plecs('error', ...
        ['In a model containing a subsystem with code generation enabled, ' ...
         'all target blocks must be placed in a subsystem configured '...
         'for code generation.']);
      b = false; return;
    end

    % Check for matching targets
    if ~plTspTarget('matches', targets)

      plecs('error', ...
        ['This block can only be used in a model or subsystem configured '...
         'to generate code for ' targetDescriptionStr ...
         ' @param:CodeGenTarget:2:.'], modelPath);
        b = false; return;
    end
    b = true;

  case 'foundValidSysClkHzForDetermineAutomatically' %(selectionWidget)
    selectionWidget = opt2;
    selectionWidgetLink = maskLinkFromStr(inputname(2), true);
    sysClkHz = plTspTarget('getSysClkHz');


    if ~plCheck('Constraints', sysClkHz, 'positive,integer,scalar')
      sysClkName = plTspTarget('getSysClkWidgetName');
      plecs('error', ...
        ['When ' selectionWidgetLink ' is set to ''Determine automatically'', '...
        'the @param:' sysClkName ':2: must be a positive integer value.'], modelPath);
      b = false; return;
    end
    b = true;

  case 'foundValidSysClkHz'
    sysClkHz = plTspTarget('getSysClkHz');

    if ~plCheck('Constraints', sysClkHz, 'positive,integer,scalar')
      sysClkName = plTspTarget('getSysClkWidgetName');
      plecs('error',...
        ['The @param:' sysClkName ':2: must be a positive integer number of hertz.'], modelPath);
      b = false; return;
    end
    b = true;

  case 'targetVarMeetsConstraints' %('varName', 'constraintsList', 'postfix')
    coderOptionsWidgetNameStr = opt2;
    constraintsListStr = opt3;
    if (nargin < 4)
      postfix = '';
    else
      postfix = opt4;
    end
    modelPath = plTspTarget('getModelPath');

    value = plTspTarget('getVar', coderOptionsWidgetNameStr);
    if ~plCheck('Constraints', value, constraintsListStr)
      if strcmp(postfix, '') % no default provided
        switch(constraintsListStr)
          case 'nonNegative,scalar'
            postfix = 'The value must be a non-negative finite scalar.';
          case 'positive,scalar'
            postfix = 'The value must be a positive finite scalar.';
          case 'nonNegative,integer,scalar'
            postfix = 'The value must be a non-negative integer.';
          case 'positive,integer,scalar'
            postfix = 'The value must be a positive integer.';
          otherwise
            postfix = '';
        end
      end
      plecs('error', ['Invalid value for @param:' coderOptionsWidgetNameStr ':2:. ' postfix], modelPath);
      b = false; return;
    end
    b = true;

  case 'deadTimeNoMoreThanHalfCarrierPeriod' %(deadTime, carrierFreqHz)
    deadTime = opt2;
    carrierFreqHz = opt3;
    deadTimeLink = maskLinkFromStr(inputname(2));
    carrierFreqLink = maskLinkFromStr(inputname(3), true);

    if ~plCheck('Constraints', deadTime, 'nonNegative,scalar')
      plecs('error', ['plTspAssert(''deadTimeNoMoreThanHalfCarrierPeriod'') '...
                       'expects nonNegative scalar value for ' deadTimeLink '.']);
      b = false; return;
    end
    if ~plCheck('Constraints', carrierFreqHz, 'positive,scalar')
      plecs('error', ['plTspAssert(''deadTimeNoMoreThanHalfCarrierPeriod'') '...
                      'expects positive scalar value for ' carrierFreqLink '.']);
      b = false; return;
    end

    halfCarrierPeriod = 1/carrierFreqHz/2;
    if deadTime >= halfCarrierPeriod
        plecs('error', ...
            [deadTimeLink ' should not be larger than half the '...
            'carrier period (' num2str(halfCarrierPeriod) ...
            ' [s]), as determined by ' carrierFreqLink '.'])
        b = false; return;
    end
    b = true;

  case 'discretizationStepSizeDefinedForAutomaticAdcTrigger' %(trigSrc)
    trigSrc = opt2;
    trigSrcLink = maskLinkFromStr(inputname(2), true);
    SAMPLE_TIME = plTspTarget('getSAMPLE_TIME');

    if (trigSrc == 1) % determine automatically
      if ~plCheck('Constraints', SAMPLE_TIME, 'positive,scalar')
        plTspTarget('logSAMPLE_TIME_error', ...
          ['When ' trigSrcLink ' is set to ''Determine automatically'',']);
        b = false; return;
      end
    end
    b = true;

  case 'isValidTriggerDivider' %(triggerDiv)
    % This check is a bit of an abuse of an 'assert' function. For legacy
    % reasons we want to treat empty triggerDividers as a 1. This function
    % will pass through a valid divider, set empty to 1, or return false and
    % log an error if the specified value is invalid. Because of the requirement
    % to continue allowing empty fields, we can't use Constraints in the PLECS file.
    triggerDivVal = opt2;
    triggerDivLink = maskLinkFromStr(inputname(2));
    if isnan(triggerDivVal)
      b = 1; return; % This is the default value for empty widgets
    end
    if ~plCheck('Constraints', triggerDivVal, 'scalar,positive,integer')
      plecs('error', [triggerDivLink ' must be a positive integer value.'])
      b = false; return;
    end
    b = triggerDivVal; return;

  otherwise
    plecs('error', ['Invalid first argument to plTspAssert(): ''' fcn '''']);
    b = false; return;
end

%%%%%%%%%%%%%%%%%%  Helper Functions %%%%%%%%%%%%%%%%%%%%%%%
function linkStr = maskLinkFromStr(varNameStr, useLowerCaseThe)
    % Use this function if you have a string name of the widget
    % to link to (calling from a helper function).
    if (nargin < 2)
        useLowerCaseThe = false;
    end

    if useLowerCaseThe
        linkStr = ['the parameter @param:' varNameStr ':'];
    else
        linkStr = ['The parameter @param:' varNameStr ':'];
    end
end

%%%%%%%%%%%%%%%%%%% String formatting helpers %%%%%%%%%%%%%%%%%%
function s = range2Str(minLimit, maxLimit)
    s =[ '[' num2str(minLimit) ' ... ' num2str(maxLimit) ']'];
end
function s = number2HexStr(val)
    % val should be a non negative number, but an b = assert gives no stack trace
    if val == 0
       s = '0';
    else
        s = ['0x' dec2hex(val)];
    end
end
function s = range2StrHex(minLimit, maxLimit)
    s = [ '[' number2HexStr(minLimit) ' ... ' number2HexStr(maxLimit) ']'];
end
function s = set2Str(setList)
    strElements = arrayfun(@num2str, setList, 'UniformOutput', false); % Convert numbers to strings
    s = strcat('{', strjoin(strElements, ', '), '}'); % Concatenate into the desired format
end

end
