#include "plecsrtw.h"
#include <math.h>

double plecsmod(double a, double b)
{
   if (!b)
   {
      return a;
   }
   else
   {
      double r = fmod(a, b);
      if ((r < 0 && b > 0) || (r > 0 && b < 0))
         r += b;
      return r;
   }
}

size_t findIndex(const double* xVec, size_t numX, double x, double* fraction)
{
   size_t idx = 0;
   size_t top = numX;
   size_t bottom = 0;
   while (top > bottom)
   {
      size_t mid = (bottom + top) / 2;
      if (xVec[mid] < x)
      {
         bottom = mid + 1;
      }
      else
      {
         top = mid;
      }
   }
   idx = bottom;
   if (idx + 2 > numX)
   {
      idx = numX - 2;
   }
   else if (idx > 0 && xVec[idx] > x)
   {
      --idx;
   }
   if (xVec[idx] == xVec[idx + 1])
   {
      if (idx + 2 < numX && xVec[idx] == xVec[idx + 2])
      {
         ++idx;
         *fraction = 0.0;
      }
      else
      {
         if (x > 0)
         {
            *fraction = 0.0;
         }
         else if (x < 0)
         {
            *fraction = 1.0;
         }
         else
         {
            *fraction = 0.5;
         }
      }
   }
   else
   {
      *fraction = (x - xVec[idx]) / (xVec[idx + 1] - xVec[idx]);
   }
   return idx;
}

size_t calcIndex(double x0, double dXInv, size_t numX, double x, double* fraction)
{
   size_t idx = 0;
   double div = (x - x0) * dXInv;
   if (div > 0)
   {
      idx = div;
      if (idx > numX - 2)
         idx = numX - 2;
   }
   *fraction = div - idx;
   return idx;
}

static double plecsinterp(double frac, double val0, double val1)
{
   return (val0 == val1) ? val0 : (1 - frac) * val0 + frac * val1;
}

double plecslu1d(const plecsLUT* lut, double x)
{
   double xFrac = 0.;
   size_t xIdx = findIndex(lut->xVec, lut->numX, x, &xFrac);
   return plecsinterp(xFrac, lut->data[xIdx], lut->data[xIdx + 1]);
}

double plecslu1deven(const plecsLUT* lut, double x)
{
   double xFrac = 0.;
   size_t xIdx = calcIndex(lut->x0, lut->dXInv, lut->numX, x, &xFrac);
   return plecsinterp(xFrac, lut->data[xIdx], lut->data[xIdx + 1]);
}

#define DATA2D(x, y) lut->data[(xIdx + (x)) * lut->numY + yIdx + (y)]

double plecslu2d(const plecsLUT* lut, double x, double y)
{
   double xFrac = 0., yFrac = 0.;
   double outX1, outX2;
   size_t xIdx, yIdx;

   xIdx = findIndex(lut->xVec, lut->numX, x, &xFrac);
   yIdx = findIndex(lut->yVec, lut->numY, y, &yFrac);

   outX1 = plecsinterp(yFrac, DATA2D(0, 0), DATA2D(0, 1));
   outX2 = plecsinterp(yFrac, DATA2D(1, 0), DATA2D(1, 1));

   return plecsinterp(xFrac, outX1, outX2);
}

double plecslu2deven(const plecsLUT* lut, double x, double y)
{
   double xFrac = 0., yFrac = 0.;
   double outX1, outX2;
   size_t xIdx, yIdx;
   xIdx = calcIndex(lut->x0, lut->dXInv, lut->numX, x, &xFrac);
   yIdx = calcIndex(lut->y0, lut->dYInv, lut->numY, y, &yFrac);
   outX1 = plecsinterp(yFrac, DATA2D(0, 0), DATA2D(0, 1));
   outX2 = plecsinterp(yFrac, DATA2D(1, 0), DATA2D(1, 1));
   return plecsinterp(xFrac, outX1, outX2);
}

#define DATA3D(x, y, z) \
   lut->data[((xIdx + (x)) * lut->numY + (yIdx + (y))) * lut->numZ + zIdx + (z)]

double plecslu3d(const plecsLUT* lut, double x, double y, double z)
{
   double xFrac = 0., yFrac = 0., zFrac = 0.;
   double outX1Y1, outX2Y1, outX1Y2, outX2Y2;
   double outX1, outX2;
   size_t xIdx, yIdx, zIdx;

   xIdx = findIndex(lut->xVec, lut->numX, x, &xFrac);
   yIdx = findIndex(lut->yVec, lut->numY, y, &yFrac);
   zIdx = findIndex(lut->zVec, lut->numZ, z, &zFrac);

   outX1Y1 = plecsinterp(zFrac, DATA3D(0, 0, 0), DATA3D(0, 0, 1));
   outX2Y1 = plecsinterp(zFrac, DATA3D(1, 0, 0), DATA3D(1, 0, 1));
   outX1Y2 = plecsinterp(zFrac, DATA3D(0, 1, 0), DATA3D(0, 1, 1));
   outX2Y2 = plecsinterp(zFrac, DATA3D(1, 1, 0), DATA3D(1, 1, 1));

   outX1 = plecsinterp(yFrac, outX1Y1, outX1Y2);
   outX2 = plecsinterp(yFrac, outX2Y1, outX2Y2);

   return plecsinterp(xFrac, outX1, outX2);
}

double plecslu3deven(const plecsLUT* lut, double x, double y, double z)
{
   double xFrac = 0., yFrac = 0., zFrac = 0.;
   double outX1Y1, outX2Y1, outX1Y2, outX2Y2;
   double outX1, outX2;
   size_t xIdx, yIdx, zIdx;
   xIdx = calcIndex(lut->x0, lut->dXInv, lut->numX, x, &xFrac);
   yIdx = calcIndex(lut->y0, lut->dYInv, lut->numY, y, &yFrac);
   zIdx = calcIndex(lut->z0, lut->dZInv, lut->numZ, z, &zFrac);
   outX1Y1 = plecsinterp(zFrac, DATA3D(0, 0, 0), DATA3D(0, 0, 1));
   outX2Y1 = plecsinterp(zFrac, DATA3D(1, 0, 0), DATA3D(1, 0, 1));
   outX1Y2 = plecsinterp(zFrac, DATA3D(0, 1, 0), DATA3D(0, 1, 1));
   outX2Y2 = plecsinterp(zFrac, DATA3D(1, 1, 0), DATA3D(1, 1, 1));
   outX1 = plecsinterp(yFrac, outX1Y1, outX1Y2);
   outX2 = plecsinterp(yFrac, outX2Y1, outX2Y2);
   return plecsinterp(xFrac, outX1, outX2);
}

#define DATA4D(x, y, z, w) lut->data[(((xIdx + (x)) * lut->numY + (yIdx + (y))) * lut->numZ + (zIdx + (z))) * lut->numW + wIdx + (w)]

double plecslu4d(const plecsLUT* lut, double x, double y, double z, double w)
{
   double xFrac = 0., yFrac = 0., zFrac = 0., wFrac = 0.;
   double outX1Y1Z1, outX1Y1Z2, outX1Y2Z1, outX1Y2Z2, outX2Y1Z1, outX2Y1Z2, outX2Y2Z1, outX2Y2Z2;
   double outX1Y1, outX1Y2, outX2Y1, outX2Y2;
   double outX1, outX2;
   size_t xIdx, yIdx, zIdx, wIdx;
   xIdx = findIndex(lut->xVec, lut->numX, x, &xFrac);
   yIdx = findIndex(lut->yVec, lut->numY, y, &yFrac);
   zIdx = findIndex(lut->zVec, lut->numZ, z, &zFrac);
   wIdx = findIndex(lut->wVec, lut->numW, w, &wFrac);
   outX1Y1Z1 = plecsinterp(wFrac, DATA4D(0, 0, 0, 0), DATA4D(0, 0, 0, 1));
   outX1Y1Z2 = plecsinterp(wFrac, DATA4D(0, 0, 1, 0), DATA4D(0, 0, 1, 1));
   outX1Y2Z1 = plecsinterp(wFrac, DATA4D(0, 1, 0, 0), DATA4D(0, 1, 0, 1));
   outX1Y2Z2 = plecsinterp(wFrac, DATA4D(0, 1, 1, 0), DATA4D(0, 1, 1, 1));
   outX2Y1Z1 = plecsinterp(wFrac, DATA4D(1, 0, 0, 0), DATA4D(1, 0, 0, 1));
   outX2Y1Z2 = plecsinterp(wFrac, DATA4D(1, 0, 1, 0), DATA4D(1, 0, 1, 1));
   outX2Y2Z1 = plecsinterp(wFrac, DATA4D(1, 1, 0, 0), DATA4D(1, 1, 0, 1));
   outX2Y2Z2 = plecsinterp(wFrac, DATA4D(1, 1, 1, 0), DATA4D(1, 1, 1, 1));
   outX1Y1 = plecsinterp(zFrac, outX1Y1Z1, outX1Y1Z2);
   outX1Y2 = plecsinterp(zFrac, outX1Y2Z1, outX1Y2Z2);
   outX2Y1 = plecsinterp(zFrac, outX2Y1Z1, outX2Y1Z2);
   outX2Y2 = plecsinterp(zFrac, outX2Y2Z1, outX2Y2Z2);
   outX1 = plecsinterp(yFrac, outX1Y1, outX1Y2);
   outX2 = plecsinterp(yFrac, outX2Y1, outX2Y2);
   return plecsinterp(xFrac, outX1, outX2);
}

double plecslu4deven(const plecsLUT* lut, double x, double y, double z, double w)
{
   double xFrac = 0., yFrac = 0., zFrac = 0., wFrac = 0.;
   double outX1Y1Z1, outX1Y1Z2, outX1Y2Z1, outX1Y2Z2, outX2Y1Z1, outX2Y1Z2, outX2Y2Z1, outX2Y2Z2;
   double outX1Y1, outX1Y2, outX2Y1, outX2Y2;
   double outX1, outX2;
   size_t xIdx, yIdx, zIdx, wIdx;
   xIdx = calcIndex(lut->x0, lut->dXInv, lut->numX, x, &xFrac);
   yIdx = calcIndex(lut->y0, lut->dYInv, lut->numY, y, &yFrac);
   zIdx = calcIndex(lut->z0, lut->dZInv, lut->numZ, z, &zFrac);
   wIdx = calcIndex(lut->w0, lut->dWInv, lut->numW, w, &wFrac);
   outX1Y1Z1 = plecsinterp(wFrac, DATA4D(0, 0, 0, 0), DATA4D(0, 0, 0, 1));
   outX1Y1Z2 = plecsinterp(wFrac, DATA4D(0, 0, 1, 0), DATA4D(0, 0, 1, 1));
   outX1Y2Z1 = plecsinterp(wFrac, DATA4D(0, 1, 0, 0), DATA4D(0, 1, 0, 1));
   outX1Y2Z2 = plecsinterp(wFrac, DATA4D(0, 1, 1, 0), DATA4D(0, 1, 1, 1));
   outX2Y1Z1 = plecsinterp(wFrac, DATA4D(1, 0, 0, 0), DATA4D(1, 0, 0, 1));
   outX2Y1Z2 = plecsinterp(wFrac, DATA4D(1, 0, 1, 0), DATA4D(1, 0, 1, 1));
   outX2Y2Z1 = plecsinterp(wFrac, DATA4D(1, 1, 0, 0), DATA4D(1, 1, 0, 1));
   outX2Y2Z2 = plecsinterp(wFrac, DATA4D(1, 1, 1, 0), DATA4D(1, 1, 1, 1));
   outX1Y1 = plecsinterp(zFrac, outX1Y1Z1, outX1Y1Z2);
   outX1Y2 = plecsinterp(zFrac, outX1Y2Z1, outX1Y2Z2);
   outX2Y1 = plecsinterp(zFrac, outX2Y1Z1, outX2Y1Z2);
   outX2Y2 = plecsinterp(zFrac, outX2Y2Z1, outX2Y2Z2);
   outX1 = plecsinterp(yFrac, outX1Y1, outX1Y2);
   outX2 = plecsinterp(yFrac, outX2Y1, outX2Y2);
   return plecsinterp(xFrac, outX1, outX2);
}
