%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Function rainflow_algorithm()
% input:
%        - t_ex: The time points of the peaks and valleys
%        - Tj_ex: The values of the peak and valley points
% return: 
%        - cyc_ampl: Amplitudes of the different temperature cycles
%        - cyc_fequ: Frequencies (incidence) of the different temperature cycles
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [cyc_ampl, cyc_freq] = rainflow_algorithm(t_ex,Tj_ex)
   % N (an integer) is the length of the list
   N = length(Tj_ex);
   % A (a list) stores the amplitude of each rainflow
   A = zeros(N, 1);
   % i_flow_end (a list) stores end/termination point of each rainflow
   % e.g. i_flow_end[7] = 11 means the rainflow started from #7 extreme point stops at #11 extreme point
   i_flow_end = zeros(N, 1);
   % detect the data list starts with a peak or a valley
   % i_pk_0 and j_vl_0 (integers) are the indices of the first peak and the first valley, respectively
   if (Tj_ex(1) > Tj_ex(2))
      i_pk_0 = 1;
      j_vl_0 = 2;
   else
      i_pk_0 = 2;
      j_vl_0 = 1;
   end

   % rain flow from the first peak
   % i_cur_hi_pk (integer) is the index of the current highest peak
   % i_next_pk (integer) is the index of the next peak higher tahn the current peak
   % end_vl (float) is the value of the flow-end valley
   i_cur_hi_pk = i_pk_0;

   % the current peak is one of the highest peaks of all points
   if ( Tj_ex(i_pk_0) == max(Tj_ex(i_pk_0:N)) )
      end_vl = min(Tj_ex(i_pk_0+2:N));
      A(i_pk_0) = Tj_ex(i_pk_0) - end_vl;
      i_flow_end(i_pk_0) = find(Tj_ex == end_vl);
      
   % the current peak is not one of the highest peaks, find the next higher peak
   else 
      for x = i_pk_0:N
         if ( Tj_ex(x) > Tj_ex(i_pk_0) )
         break 
         end
      end
      end_vl = min(Tj_ex(i_pk_0:x));
      A(i_pk_0) = Tj_ex(i_pk_0) - end_vl;
      i_flow_end(i_pk_0) =  find(Tj_ex == end_vl);
      
   end

   % rain flow from the first valley
   % j_cur_lo_vl (integer) is the index of the current lowest valley
   % i_next_vl (integer) is the index of the next valley lower than the current valley
   % end_pk (float) is the value of the flow-end peak
   j_cur_lo_vl = j_vl_0;

   % the current valley is one of the lowest valleys of all points
   if ( Tj_ex(j_vl_0) == min(Tj_ex(j_vl_0:N)) )
      end_pk = max(Tj_ex(j_vl_0+2:N));
      A(j_vl_0) = -(Tj_ex(j_vl_0) - end_pk);
      i_flow_end(j_vl_0) = find(Tj_ex == end_pk);
      
   % the current valley is not one of the lowest valley, find the next lower valley
   else 
      for y = j_vl_0:N
         if ( Tj_ex(y) < Tj_ex(j_vl_0) )
         break 
         end
      end   
      end_pk = max(Tj_ex(j_vl_0:y));
      A(j_vl_0) = -(Tj_ex(j_vl_0) - end_pk);
      i_flow_end(j_vl_0) = find(Tj_ex == end_pk);
      
   end


   % calculate rainflow started from each peak
   i = i_pk_0 + 2;

   while i < N

      % the current peak higher than the previous peaks
      if ( Tj_ex(i) > Tj_ex(i_cur_hi_pk) )
         i_cur_hi_pk = i;
         
         % case I: and no lower than the next peaks
         if ( Tj_ex(i) == max(Tj_ex(i:N)) )
            end_vl = min(Tj_ex(i:N));
            A(i) = Tj_ex(i) - end_vl;
            i_flow_end(i) = find(Tj_ex == end_vl);
            
         % case II: but lower than one of the next peaks
         else        
            for x = i:N 
               if ( Tj_ex(x) > Tj_ex(i) )
               break
               end
            end % locate i_next_pk
            end_vl = min(Tj_ex(i:x));
            A(i) = Tj_ex(i) - end_vl;
            i_flow_end(i) = find(Tj_ex == end_vl);       
         
         end   
      
      % the current peak no higher than the previous peaks  
      else
         for x = i:-1:1    
            if ( Tj_ex(x) > Tj_ex(i) )
            break
            end
         end % locate i_pre_pk
         pre_vl = min(Tj_ex(x:i));
         
         i_next_pk_mat = [];
         i_next_pk_index = 1;
         for x = i:N 
            if ( Tj_ex(x) > Tj_ex(i) )
               i_next_pk_mat(i_next_pk_index) = x-i;
               i_next_pk_index = i_next_pk_index + 1;
            end         
         end % locate i_next_pk_matrix
         
         % but no lower than the next peaks     
         if ( isempty(i_next_pk_mat) )
            next_min_vl = min(Tj_ex(i:N));
            
            % case III: the next valleys are higher than the lowest valley between the last higher peak and the current peak
            if (next_min_vl > pre_vl)
               A(i) = Tj_ex(i) - next_min_vl;
               i_flow_end(i) = find(Tj_ex == next_min_vl);     
               
            % case IV: one of the next valleys are no higher than the lowest valley between the last higher peak and the current peak
            else
               A(i) = Tj_ex(i) - pre_vl;
               for x = i:N 
                  if ( Tj_ex(x) <= pre_vl )
                  break
                  end
               end 
               i_flow_end(i) = x;
                           
            end
            
         % and lower than one of the next peaks             
         else
            i_next_pk = i_next_pk_mat(1) + i;
            next_min_vl = min(Tj_ex(i:i_next_pk));
            
            % case V: the next valleys are higher than the lowest valley between the last higher peak and the current peak
            if (next_min_vl > pre_vl)
               A(i) = Tj_ex(i) - next_min_vl;
               i_flow_end(i) = find(Tj_ex == next_min_vl);
            
            % case VI: one of the next valleys are no higher than the lowest valley between the last higher peak and the current peak
            else
               A(i) = Tj_ex(i) - pre_vl;
               for x = i:i_next_pk 
                  if ( Tj_ex(x) <= pre_vl )
                  break
                  end
               end      
               i_flow_end(i) = x;      
            
            end         
         
         end   
         
      end
      
      i = i + 2;     
      
   end
      
   % if Tj_ex list ends with a peak
   if (i == N)
      A(i) = 0;
      i_flow_end(i) = i;
   end


   % calculate rainflow started from each valley
   j = j_vl_0 + 2;

   while j < N

      % the current valley lower than the previous valleys
      if ( Tj_ex(j) < Tj_ex(j_cur_lo_vl) )
         j_cur_lo_vl = j;
         
         % case 1: and no higher than the next valleys
         if ( Tj_ex(j) == min(Tj_ex(j:N)) )
            end_pk = max(Tj_ex(j:N));
            A(j) = -(Tj_ex(j) - end_pk);
            i_flow_end(j) = find(Tj_ex == end_pk);
            
         % case 2: but higher than one of the next valleys
         else
            for y = j:N 
               if ( Tj_ex(y) < Tj_ex(j) )
               break
               end
            end % locate i_next_vl
            end_pk = max(Tj_ex(j:y));
            A(j) = -(Tj_ex(j) - end_pk);
            i_flow_end(j) = find(Tj_ex == end_pk);
         
         end
         
      % the current valley no lower than the previous valleys
      else
         for y = j:-1:1    
            if ( Tj_ex(y) < Tj_ex(j) )
            break
            end
         end % locate i_pre_vl   
         pre_pk = max(Tj_ex(y:j));
         
         i_next_vl_mat = [];
         i_next_vl_index = 1;
         for y = j:N 
            if ( Tj_ex(y) < Tj_ex(j) )
               i_next_vl_mat(i_next_vl_index) = y-j;
               i_next_vl_index = i_next_vl_index + 1;
            end         
         end % locate i_next_vl_matrix
         
         % but no higher than the next valleys     
         if ( isempty(i_next_vl_mat) )
            next_max_pk = max(Tj_ex(j:N));
         
            % case 3: the next peaks are lower than the highest peak between the last lower valley and the current valley
            if (next_max_pk < pre_pk)
               A(j) = -(Tj_ex(j) - next_max_pk);
               i_flow_end(j) = find(Tj_ex == next_max_pk);
            
            % case 4: one of the next peaks are no lower than the highest peak between the last lower valley and the current valley
            else
               A(j) = -(Tj_ex(j) - pre_pk);
               for y = j:N 
                  if ( Tj_ex(y) >= pre_pk )
                  break
                  end
               end 
               i_flow_end(j) = y;                     
            
            end
            
         % and higher than one of the next valleys
         else
            i_next_vl = i_next_vl_mat(1) + j;
            next_max_pk = max(Tj_ex(j:i_next_vl));    
            
            % case 5: the next peaks are lower than the highest peak between the last lower valley and the current valley
            if (next_max_pk < pre_pk)
               A(j) = -(Tj_ex(j) - next_max_pk);
               i_flow_end(j) = find(Tj_ex == next_max_pk);

            % case 6: one of the next peaks are no lower than the highest peak between the last lower valley and the current valley
            else
               A(j) = -(Tj_ex(j) - pre_pk);
               for y = j:i_next_vl 
                  if ( Tj_ex(y) >= pre_pk )
                  break
                  end
               end      
               i_flow_end(j) = y;            
            
            end
                           
         end
         
      end
      
      j = j + 2;
      
   end
            
   % if Tj_ex list ends with a valley
   if (j == N)
      A(j) = 0;
      i_flow_end(j) = j;
   end

   %A()  
   %i_flow_end()

   % export rainflow results from each peak and each valley
   % note that the exported files are not needed for lifetime calculation
   % four lists (A, i_flow_end, t_cyc_start, and t_cyc_end) store the amplitude of rainflow, the point 
   % where the each rainflow ends, and starting and ending time of each rainflow, respectively
   t_cyc_start = t_ex;

   for k = 1:N
      t_cyc_end(k) = t_ex(i_flow_end(k));
   end

   %t_cyc_end

   % count number of half cycles with equivalent amplitudes, amplitudes and frequencies of half cycles
   % the unique()function automatically sort counting results in acscending order of cycle amplitudes
   % cyc_ampl() stores the amplitudes of cycles in acscending order
   % cyc_freq() stores the corresponding number of half cycles or occurrences
   [cyc_freq,cyc_ampl] = hist(A,unique(A));
   %convert cyc_ampl in descending order togteher with cyc_freq
   cyc_freq = flip(cyc_freq *0.5); %convert half cycles into full cycles
   cyc_ampl = flip(cyc_ampl);
end