## Copyright (C) 2021 Plexim GmbH. All rights reserved.
##
## This file is part of PLECS.

## -*- texinfo -*-
## @deftypefn  {} {@var{result} = } jsonrpc (@var{hostname}, @var{port}, @var{method}, @var{params})
## Send a JSON-RPC 2.0 request. The decode result is returned or an error raised.
## See https://www.jsonrpc.org/specification.
## @end deftypefn

function result = jsonrpc (hostname, port, method, params)
  narginchk(4, 4);

  persistent idcount = 0;
  json = jsonencode(struct("jsonrpc", "2.0", ...
                           "id", ++idcount, ...
                           "method", method, ...
                           "params", {params}));
  header = sprintf(["POST /RPC2 HTTP/1.1\r\n" ...
                    "Content-Length: %i\r\n" ...
                    "\r\n"], length(json));

  % send request
  client = tcpclient(hostname, port, "ExpectDisconnect", true);
  write(client, uint8([header json]));

  % read response
  while true
    if (client.NumBytesAvailable > 0)
      request_size = client.NumBytesAvailable;
    else
      request_size = 2048;
    endif
    response{end+1} = read(client, request_size, "string");
    if (length(response{end}) ~= request_size)
      break;
    endif
  endwhile

  % combine chunks
  response = char(cell2mat(response));

  % split response into status and content
  statusLineEnd = index(response, "\r\n");
  contentStart = index(response, "\r\n\r\n") + 4;

  status = substr(response, 1, statusLineEnd);
  content = jsondecode(substr(response, contentStart));

  if (isfield(content, "result"))
    result = content.result;
  elseif (isfield(content, "error"))
    error(sprintf("jsonrpc: error %i when calling %s: %s", ...
                  content.error.code, ...
                  method, content.error.message));
  else
    error(sprintf("jsonrpc: connection error %s", status));
  endif
endfunction
