## Copyright (C) 2021 Plexim GmbH. All rights reserved.
##
## This file is part of PLECS.

## -*- texinfo -*-
## @deftypefn  {} {@var{result} = } plecs (@var{command}, ...)
## @deftypefnx {} {} plecs ("simulate" [, @var{options}, @var{callback}])
## @deftypefnx {} {} plecs ("analyze", @var{analysisName} [, @var{options}, @var{callback}])
## @deftypefnx {} {} plecs ("codegen", @var{subsystemPath} [, @var{options}] [, @var{outputDir}])
## @deftypefnx {} {} plecs ("get", @var{componentPath} [, @var{parameterName}])
## @deftypefnx {} {} plecs ("set", @var{componentPath}, @var{parameterName}, @var{parameterValue})
## @deftypefnx {} {} plecs ("scope", @var{scopePath}, @var{scopeCmd} [, ...])
## @deftypefnx {} {} plecs ("warning" [, @var{message}])
## @deftypefnx {} {} plecs ("clc")
## @deftypefnx {} {} plecs ("init", @var{modelname}, @var{hostname} = "localhost", @var{port} = 1080)
## @end deftypefn

function retval = plecs (cmd, varargin)
  mlock();
  persistent hostname = "localhost";
  persistent port = 1080;
  persistent modelname = "";
  persistent scripttoken = "";
  persistent resulttoken = "";

  function checkinit ()
    if (length(modelname) == 0)
      error("plecs: plecs('init') has to be called before calling any other commands.");
    endif
  endfunction

  function checkincallback ()
    if (length(resulttoken) > 0)
      error("plecs: command not allowed in parallel simulation callback.");
    endif
  endfunction

  function ret = replacedotpath (path)
    if (strcmp(path, ".") == 1)
      ret = modelname;
    elseif (strncmp(path, "./", 2) == 1)
      ret = [modelname substr(path, 2)];
    else
      ret = path;
    endif
  endfunction

  function ret = convertResults (inval)
    ret = inval;
    if (isfield(ret, "F"))
      ret.F = (ret.F).';
    elseif (isfield(ret, "Time"))
      ret.Time = (ret.Time).';
    endif
  endfunction

  if (nargin < 1 || !ischar(cmd))
    print_usage();

  % -- plecs("init", modelname [, hostname, port, scripttoken])
  elseif (strcmp(cmd, "init") == 1)
    % Check input arguments
    narginchk(2, 5);

    % Check arguments and save values in persistent variables
    if (!ischar(varargin{1}))
      error("plecs: The first argument for the 'init' command has to be a string.");
    endif
    modelname = varargin{1};

    if (nargin >= 3)
      if (!ischar(varargin{2}))
        error("plecs: The second argument for the 'init' command has to be a string.");
      endif
      hostname = varargin{2};
    endif
    if (nargin >= 4)
      if (!isnumeric(varargin{3}))
        error("plecs: The third argument for the 'init' command has to be a number.");
      endif
      port = varargin{3};
    endif
    if (nargin >= 5)
      if (!ischar(varargin{4}))
        error("plecs: The fourth argument for the 'init' command has to be a string.");
      endif
      scripttoken = varargin{4};
    endif

  % -- plecs("success", result)
  elseif (strcmp(cmd, "success") == 1)
    % Check input arguments
    narginchk(2, 2);
    if (length(scripttoken) == 0)
      error("plecs: 'success' command needs a configured token.");
    endif

    % Send result back to PLECS
    jsonrpc(hostname, port, ...
            "plecs.webserver", { "setWebscriptResult", modelname, ...
            struct("scriptToken", scripttoken, "result", {varargin{1}})});

  % -- plecs("error")
  elseif (strcmp(cmd, "error") == 1)
    % Check input arguments
    narginchk(1, 1);
    if (length(scripttoken) == 0)
      error("plecs: 'error' command needs a configured token.");
    endif

    % Add error message
    errmsg = lasterror.message;
    if (length(lasterror.identifier) > 0)
      errmsg = [errmsg " [" lasterror.identifier "]"];
    endif

    % Add backtrace. Only include backtrace up to the first call from the script
    errmsgtemp = "";

    for i = 1:length(lasterror.stack)
      filename = lasterror.stack(i).file;
      filename = strsplit(strsplit(filename, "/"){end}, "\\"){end};

      if (length(filename) == 24 && strncmp(filename, "_plecsWbsScript_", 16) == 1)
        errmsgtemp = [errmsgtemp "\n  at Script:" num2str(lasterror.stack(i).line) ...
                      ":" num2str(lasterror.stack(i).column)];

        errmsg = [errmsg errmsgtemp];
        errmsgtemp = "";
      else
        errmsgtemp = [errmsgtemp "\n  at " lasterror.stack(i).name " (" ...
                      filename ":" num2str(lasterror.stack(i).line) ":" ...
                      num2str(lasterror.stack(i).column) ")"];
      endif
    endfor

    % Send backtrace back to PLECS
    jsonrpc(hostname, port, ...
            "plecs.webserver", { "setWebscriptResult", modelname, ...
            struct("scriptToken", scripttoken, "errorMsg", errmsg)});

  % -- plecs("simulate" [, opts, callback])
  elseif (strcmp(cmd, "simulate") == 1)
    % Check input arguments
    narginchk(1, 3);
    checkinit();
    checkincallback();
    if (nargin >= 2)
      if (isstruct(varargin{1}))
        if (nargin > 2)
          error("plecs: Invalid number of arguments for non-parallel 'simulate' command.");
        endif
      elseif (iscell(varargin{1}))
        if (rows(varargin{1}) != 1)
          error("plecs: The first argument of the parallel 'simulate' command has to be a cell array with exactly one row.");
        endif
        for i = 1:columns(varargin{1})
          if (!isstruct(varargin{1}{i}))
            error("plecs: The first argument of the paralle 'simulate' command has to be a cell array containing structs.");
          endif
        endfor

        if (nargin > 3)
          error("plecs: Invalid number of arguments for parallel 'simulate' command.");
        endif
        if (nargin == 3 && !(is_function_handle(varargin{2}) || isstruct(varargin{2})))
          error("plecs: The second argument of the parallel 'simulate' command has to be a function handle or a struct.");
        endif
      else
        error("plecs: The first argument for the 'simulate' command has to be a struct or cell array.");
      endif
    endif

    % - non-parallel simulation
    if (nargin == 1 || isstruct(varargin{1}))
      % Create options struct
      if (nargin == 2)
        optStruct = varargin{1};
      else
        optStruct = struct();
      endif

      % Add scriptToken
      optStruct.scriptToken = scripttoken;

      % Start simulation
      jsonrpc(hostname, port, "plecs.webserver", ...
              {"startSimulation", modelname, optStruct});

      % Wait for simulation to end
      while (true)
        status = jsonrpc(hostname, port, "plecs.webserver", ...
                         {"getSimulationState", modelname, struct("scriptToken", scripttoken)});
        if (strcmp(status.state, "running") == 0)
          break;
        endif
        pause(0.01);
      endwhile

      % Return simulation result or throw error
      if (strcmp(status.state, "error") == 1)
        error(status.errorMessage);
      elseif (strcmp(status.state, "finished") == 1)
        if (nargout != 0)
          retval = convertResults(jsonrpc(hostname, port, "plecs.webserver", ...
                                  {"getResults", modelname, struct("scriptToken", scripttoken)}));
        endif
      else
        error(sprintf("plecs: Unknown simulation state %s.", mat2str(status)));
      endif

    % - parallel simulation
    else
      simArgs.runOpts = varargin{1};
      simArgs.simConfig = struct();
      simArgs.scriptToken = scripttoken;

      callback = [];
      if (nargin > 2)
        if (isstruct(varargin{2}))
          if (isfield(varargin{2}, "Callback"))
            if (!is_function_handle(varargin{2}.Callback))
              error("plecs: The 'Callback' struct member of the parallel 'simulate' command has to be a function handle.");
            else
              callback = varargin{2}.Callback;
            endif 
          endif

          simArgs.simConfig = varargin{2};
          for [val, key] = varargin{2}
            if (is_function_handle(val))
              simArgs.simConfig = rmfield(simArgs.simConfig, key);
            endif
          endfor
        else
          callback = varargin{2};
        endif
      endif

      % Start simulation
      jsonrpc(hostname, port, "plecs.webserver", ...
              {"startParallelSimulation", modelname, simArgs});

      parsimresult = {};
      parsimerror = struct();

      % Wait for simulation to end
      resultoken = "";
      unwind_protect
        while (true)
          status = jsonrpc(hostname, port, "plecs.webserver", {"getSimulationState", modelname, ...
                           struct("scriptToken", scripttoken, "resultToken", resulttoken)});
          if (strcmp(status.state, "running") == 0)
            break;
          endif

          % Parallel Simulation has finished --> execute callback and save result
          if (isfield(status, "result"))
            % Get parallel simulation results
            curindex = status.result.index;
            curresult = convertResults(status.result.result);
            resulttoken = status.result.token;

            % Execute callback
            if (!isempty(callback) && numfields(parsimerror) == 0)
              setans = false;
              if (exist("ans", "var"))
                setans = true;
                valans = ans;
                clear ans;
              endif

              try
                callback(curindex, curresult);
                if (exist("ans", "var"))
                  curresult = ans;
                endif
              catch
                parsimerror = lasterror;
                jsonrpc(hostname, port, "plecs.webserver", {"stopSimulation", modelname, ...
                        struct("scriptToken", scripttoken)});
              end_try_catch

              if (setans)
                ans = valans;
              else
                clear ans;
              endif
            endif

            % Save result
            if (nargout != 0)
              parsimresult{curindex} = curresult;
            endif

          % No result to process --> wait for a moment
          else
            resulttoken = "";
            pause(0.01);
          endif
        endwhile
      unwind_protect_cleanup
        resulttoken = "";
      end_unwind_protect

      % Return simulation result or throw error
      if (numfields(parsimerror) > 0)
        rethrow(parsimerror);
      elseif (strcmp(status.state, "error") == 1)
        error(status.errorMessage);
      elseif (strcmp(status.state, "finished") == 1)
        if (nargout != 0)
          retval = parsimresult;
        endif
      else
        error(sprintf("plecs: Unknown simulation state %s.", mat2str(status)));
      endif
    endif

  % -- plecs("analyze", analysisName [, opts, callback])
  elseif (strcmp(cmd, "analyze") == 1)
    % Check input arguments
    narginchk(2, 4);
    checkinit();
    checkincallback();
    if (!ischar(varargin{1}))
      error("plecs: The first argument for the 'analyze' command has to be a string.");
    endif
    if (nargin >= 3)
      if (isstruct(varargin{2}))
        if (nargin > 3)
          error("plecs: Invalid number of arguments for non-parallel 'analyze' command.");
        endif
      elseif (iscell(varargin{2}))
        if (rows(varargin{2}) != 1)
          error("plecs: The second argument of the parallel 'analyze' command has to be a cell array with exactly one row.");
        endif
        for i = 1:columns(varargin{2})
          if (!isstruct(varargin{2}{i}))
            error("plecs: The second argument of the paralle 'analyze' command has to be a cell array containing structs.");
          endif
        endfor

        if (nargin > 4)
          error("plecs: Invalid number of arguments for parallel 'analyze' command.");
        endif
        if (nargin == 4 && !(is_function_handle(varargin{3}) || isstruct(varargin{3})))
          error("plecs: The third argument of the parallel 'analyze' command has to be a function handle or a struct.");
        endif
      else
        error("plecs: The second argument for the 'analyze' command has to be a struct or cell array.");
      endif
    endif

    % - non-parallel analysis
    if (nargin == 2 || isstruct(varargin{2}))
      % Create options struct
      if (nargin == 3)
        optStruct = varargin{2};
      else
        optStruct = struct();
      endif
      optStruct.analysisName = varargin{1};

      % Add scriptToken
      optStruct.scriptToken = scripttoken;

      % Start analysis
      jsonrpc(hostname, port, "plecs.webserver", ...
              {"startAnalysis", modelname, optStruct});

      % Wait for analysis to end
      while (true)
        status = jsonrpc(hostname, port, "plecs.webserver", ...
                         {"getSimulationState", modelname, struct("scriptToken", scripttoken)});
        if (strcmp(status.state, "running") == 0)
          break;
        endif
        pause(0.01);
      endwhile

      % Return analysis result or throw error
      if (strcmp(status.state, "error") == 1)
        error(status.errorMessage);
      elseif (strcmp(status.state, "finished") == 1)
        if (nargout != 0)
          retval = convertResults(jsonrpc(hostname, port, "plecs.webserver", ...
                                  {"getResults", modelname, struct("scriptToken", scripttoken)}));
        endif
      else
        error(sprintf("plecs: Unknown analysis state %s.", mat2str(status)));
      endif

    % - parallel analysis
    else
      simArgs.analysisName = varargin{1};
      simArgs.runOpts = varargin{2};
      simArgs.simConfig = struct();
      simArgs.scriptToken = scripttoken;

      callback = [];
      if (nargin > 3)
        if (isstruct(varargin{3}))
          if (isfield(varargin{3}, "Callback"))
            if (!is_function_handle(varargin{3}.Callback))
              error("plecs: The 'Callback' struct member of the parallel 'analyze' command has to be a function handle.");
            else
              callback = varargin{3}.Callback;
            endif 
          endif

          simArgs.simConfig = varargin{3};
          for [val, key] = varargin{3}
            if (is_function_handle(val))
              simArgs.simConfig = rmfield(simArgs.simConfig, key);
            endif
          endfor
        else
          callback = varargin{3};
        endif
      endif

      % Start analysis
      jsonrpc(hostname, port, "plecs.webserver", ...
              {"startParallelAnalysis", modelname, simArgs});

      parsimresult = {};
      parsimerror = struct();

      % Wait for analysis to end
      resultoken = "";
      unwind_protect
        while (true)
          status = jsonrpc(hostname, port, "plecs.webserver", {"getSimulationState", modelname, ...
                           struct("scriptToken", scripttoken, "resultToken", resulttoken)});
          if (strcmp(status.state, "running") == 0)
            break;
          endif

          % Parallel Analysis has finished --> execute callback and save result
          if (isfield(status, "result"))
            % Get parallel analysis results
            curindex = status.result.index;
            curresult = convertResults(status.result.result);
            resulttoken = status.result.token;

            % Execute callback
            if (!isempty(callback) && numfields(parsimerror) == 0)
              setans = false;
              if (exist("ans", "var"))
                setans = true;
                valans = ans;
                clear ans;
              endif

              try
                callback(curindex, curresult);
                if (exist("ans", "var"))
                  curresult = ans;
                endif
              catch
                parsimerror = lasterror;
                jsonrpc(hostname, port, "plecs.webserver", {"stopSimulation", modelname, ...
                        struct("scriptToken", scripttoken)});
              end_try_catch

              if (setans)
                ans = valans;
              else
                clear ans;
              endif
            endif

            % Save result
            if (nargout != 0)
              parsimresult{curindex} = curresult;
            endif

          % No result to process --> wait for a moment
          else
            resulttoken = "";
            pause(0.01);
          endif
        endwhile
      unwind_protect_cleanup
        resulttoken = "";
      end_unwind_protect

      % Return analysis result or throw error
      if (numfields(parsimerror) > 0)
        rethrow(parsimerror);
      elseif (strcmp(status.state, "error") == 1)
        error(status.errorMessage);
      elseif (strcmp(status.state, "finished") == 1)
        if (nargout != 0)
          retval = parsimresult;
        endif
      else
        error(sprintf("plecs: Unknown analysis state %s.", mat2str(status)));
      endif
    endif

  % -- plecs("codegen", subsystemPath [, optStruct] [, outputDir])
  elseif (strcmp(cmd, "codegen") == 1)
    % Check input arguments
    narginchk(2, 4);
    checkinit();
    checkincallback();
    if (!ischar(varargin{1}))
      error("plecs: The first argument for the 'codegen' command has to be a string.");
    endif
    if (nargin == 3 && !(ischar(varargin{2}) || isstruct(varargin{2})))
      error("plecs: The second argument for the 'codegen' command has to be a string or a struct.");
    endif
    if (nargin == 4 && (!isstruct(varargin{2}) || !ischar(varargin{3})))
      error("plecs: Invalid argument types for 'codegen' command.");
    endif

    % Get subsystemPath and do dot path replacement
    subsystempath = replacedotpath(varargin{1});

    % Get opStruct
    if (nargin >= 3 && isstruct(varargin{2}))
      optstruct = varargin{2};
    else
      optstruct = struct();
    endif

    % Get outputDir
    if (nargin == 4)
      outputdir = varargin{3};
    elseif (nargin == 3 && ischar(varargin{2}))
      outputdir = varargin{2};
    else
      outputdir = "";
    endif

    % Start codegen
    jsonrpc(hostname, port, "plecs.webserver", {"startCodegen", modelname, ...
            struct("scriptToken", scripttoken, "subsystemPath", subsystempath,
                   "options", optstruct, "outputDirectory", outputdir)});

    % Wait for codegen to end
    while (true)
      status = jsonrpc(hostname, port, "plecs.webserver", ...
                       {"getSimulationState", modelname, struct("scriptToken", scripttoken)});
      if (strcmp(status.state, "running") == 0)
        break;
      endif
      pause(0.01);
    endwhile

    % throw error if codegen ended with error
    if (strcmp(status.state, "error") == 1)
      error(status.errorMessage);
    elseif (strcmp(status.state, "finished") != 1)
      error(sprintf("plecs: Unknown codegen state %s.", mat2str(status)));
    endif

  % -- plecs("get", componentPath [, parameterName])
  elseif (strcmp(cmd, "get") == 1)
    % Check input arguments
    narginchk(2, 3);
    checkinit();
    checkincallback();
    if (!ischar(varargin{1}))
      error("plecs: The first argument for the 'get' command has to be a string.");
    endif
    if (nargin == 3 && !ischar(varargin{2}))
      error("plecs: The second argument for the 'get' command has to be a string.");
    endif

    % Get componentPath and do dot path replacement
    componentpath = replacedotpath(varargin{1});

    % Get parameterName
    parametername = "";
    if (nargin == 3)
      parametername = varargin{2};
    endif

    % Execute command
    if (length(componentpath) == 0)
      if (strcmp(parametername, "CurrentCircuit") == 1 || strcmp(parametername, "CurrentComponent") == 1)
        retval = modelname;
      else
        error(sprintf("plecs: Unknown parameter '%s'.", parametername));
      endif
    elseif (strcmp(componentpath, modelname) == 1 && strcmp(parametername, "AggregationMode") == 1)
      retval = "Normal";
    else
      tempretval = jsonrpc(hostname, port, "plecs.get", {componentpath, parametername});
      if (strcmp(parametername, "SystemState") == 1)
        retval = structArr2cellArr(tempretval);
      elseif (strcmp(parametername, "StateSpaceOrder") == 1)
        for [val, key] = tempretval
          if (iscell(val))
            retval.(key) = val;
          else
            retval.(key) = cell();
          endif
        endfor
      elseif (strcmp(parametername, "Topology") == 1)
        for [val, key] = tempretval
          if (iscell(val))
            retval.(key) = [];
          else
            retval.(key) = val;
          endif
        endfor
      else
        retval = tempretval;
      endif
    endif

  % -- plecs("set", componentPath, parameterName, parameterValue)
  elseif (strcmp(cmd, "set") == 1)
    % Check input arguments
    narginchk(4, 4);
    checkinit();
    checkincallback();
    if (!ischar(varargin{1}))
      error("plecs: The first argument for the 'set' command has to be a string.");
    endif
    if (!ischar(varargin{2}))
      error("plecs: The second argument for the 'set' command has to be a string.");
    endif

    % Get componentPath and do dot path replacement
    componentpath = replacedotpath(varargin{1});

    % Execute command
    jsonrpc(hostname, port, "plecs.webserver", {"setParameterValue", modelname, ...
            struct("scriptToken", scripttoken, "componentPath", componentpath,
                   "parameterName", varargin{2}, "parameterValue", varargin{3})});

  % -- plecs("scope", scopePath, scopeCmd [, ...])
  elseif (strcmp(cmd, "scope") == 1)
    % Check input arguments
    narginchk(3, Inf);
    checkinit();
    if (!ischar(varargin{1}))
      error("plecs: The first argument for the 'scope' command has to be a string.");
    endif
    if (!ischar(varargin{2}))
      error("plecs: The second argument for the 'scope' command has to be a string.");
    endif

    % Execute command
    tempretval = jsonrpc(hostname, port, "plecs.webserver", {"scopeCommand", modelname, ...
                         struct("scopePath", varargin{1}, "scopeCmd", varargin{2},
                                "varArgs", {{varargin{3:end}}},
                                "scriptToken", scripttoken, "resultToken", resulttoken)});

    % Convert scalar structs to cell arrays for GetCursorData command
    if (strcmp(varargin{2}, "GetCursorData") == 1)
      retval = structArr2cellArr(tempretval);
    else
      retval = tempretval;
    endif

  % -- plecs("warning" [, message])
  elseif (strcmp(cmd, "warning") == 1)
    narginchk(1, 2);
    checkinit();
    % ignore

  % -- plecs("clc")
  elseif (strcmp(cmd, "clc") == 1)
    narginchk(1, 1);
    checkinit();
    clc;

  % -- Invalid command
  else
    print_usage();
  endif
endfunction

function ret = structArr2cellArr (inval)
  if (iscell(inval))
    for ix = 1:numel(inval)
      ret{ix,1} = structArr2cellArr(inval{ix});
    endfor
  elseif (isstruct(inval))
    len = length(inval);
    % -- Normal struct (size = 1 x 1)
    if (len == 1)
      for [val, key] = inval
        ret.(key) = structArr2cellArr(val);
      endfor
    % -- 1D struct (size = 1 x n)
    elseif (len == numel(inval))
      for ix = 1:len
        ret{ix,1} = structArr2cellArr(inval(ix));
      endfor
    % -- nD struct (size = i x j x k x ...)
    elseif (ndims(inval) > 2)
      reshape_dim = size(inval)(2:end);
      for ix = 1:size(inval, 1)
        ret{ix,1} = structArr2cellArr(reshape(inval(ix,:), reshape_dim));
      endfor
    % -- 2D struct (size = i x j)
    else
      for ix = 1:size(inval, 1)
        ret{ix,1} = structArr2cellArr(inval(ix,:));
      endfor
    endif
  elseif (isvector(inval) && rows(inval) > 1 && columns(inval) == 1)
    ret = inval.';
  else
    ret = inval;
  endif
endfunction
