%   Companion script for PLECS Demo model "Buck Converter with Parameter Sweep"
%
%   Requirements
%   -----------
%   - A JSON-RPC client needs to be available. Plexim provides a suitable
%   client at https://github.com/plexim/matlab-jsonrpc. The jsonrpc.m file
%   needs to be placed in the same directory as this script and the PLECS model,
%   or in a directory available in PATH
%   - The RPC interface has to be enabled in the PLECS preferences before a
%   connection can be established. The same TCP port configured in the
%   PLECS preferences needs to be used in the script. Default is 1080.
%   - RPC connections to PLECS are only allowed from clients running ont he
%   same computer as PLECS. Therefore, the connection should always be
%   initiated using "localhost" in the server URL.
%
%   Description
%   -----------
%   The script uses the RPC interface to control PLECS Standalone and run a
%   parameter sweep in the PLECS model. The sweep is composed by 10
%   independent simulations runs, where the inductance parameter "varL"
%   is modified in every run, ranging from 40 uH to 220 uH.
%
%   The script is configured so the user can choose to run the simulations
%   one by one in a sequential order or in parallel.
%   How many simulations can run in parallel depends on how many CPU
%   core/threads are avaiable. To choose the execution method simply set
%   the "METHOD" variable to "Sequential" or "Parallel".
%

METHOD = 'Sequential'; % The options are 'Sequential' or 'Parallel'
%start RPC client (you need to enable it in PLECS preferences)
proxy = jsonrpc('http://localhost:1080');
%load the model (Start PLECS manualy)
MODEL_NAME = 'buck_converter_with_parameter_sweep';
dir = pwd();
proxy.plecs.load([dir '/' MODEL_NAME '.plecs']);
% Create simStruct with field 'ModelVars'
mdlVars = struct('varL', 50e-6);
simStruct = struct('ModelVars', mdlVars);

% Clear all previous traces in scope 'Scope' in the current model
proxy.plecs.scope([MODEL_NAME '/Scope'], 'ClearTraces');

% Parametric values to be swept
inductorValues = [40:20:220]; % in uH

switch METHOD
case 'Sequential'

  for ix = 1:length(inductorValues)
    % Set value for L1
    simStruct.ModelVars.varL = inductorValues(ix) * 1e-6;
    simStruct.SolverOpts.OutputTimes = 0.001:0.2e-5:0.002;
    if ix == 5
      simStruct.ModelVars.varR = 0;
    else
      simStruct.ModelVars.varR = 1;
    end
    % Start simulation, return probed signal values in 'out'
    try
    out = proxy.plecs.simulate(MODEL_NAME,simStruct); % start AC Sweep analysis
    catch
    disp([' Error in Simulation ', num2str(ix),' for ', num2str(inductorValues(ix)),' μH']);
    end
    % Hold and label trace
    proxy.plecs.scope([MODEL_NAME '/Scope'], 'HoldTrace', ['L=' mat2str(inductorValues(ix)) 'μH']);
    % Find maximum current value and index
    [maxv, maxidx] = max(out.Values(1,:));
    % Output maximum current values to console
    disp(['Max current for L= ', num2str(inductorValues(ix)),'μH: ', num2str(maxv), ' A at ', num2str(out.Time(maxidx)), ' s']);
  end

case 'Parallel'

  % Evaluate simulation results in callback function, the disp in the callback are shown in the PLECS Console
  % In MATLAB use single quotes (') to get a single concatenated character array as in Octave
  callback = ['if ischar(result)' ...
              'disp([''There is a simulation error for the fifth case ('' name '') where varR = 0 is artificially set to zero. Nevertheless the results for the other cases are still calculated'']);' ...
              'plecs(''scope'', ''./Scope'', ''HoldTrace'', name);' ...
              'else ' ...
              'plecs(''scope'', ''./Scope'', ''HoldTrace'', name);' ...
              'disp([''Simulation Nr. '' num2str(index) '' executed'']); '...
              '[maxi, maxidx] = max(result.Values(1,:));' ...
              'maxt = result.Time(maxidx);' ...
              'result = [maxi, maxt];' ...
              'end'];

  % Initialize simStruct as cell array with all values for L1
  for ix = 1:length(inductorValues)
    simStructs{ix}.ModelVars.varL = inductorValues(ix) * 1e-6;
    % Name of 'ModelVars' can be assigned for diagnostic purposes
    simStructs{ix}.Name = ['L=' mat2str(inductorValues(ix)) 'μH'];
    %
    simStructs{ix}.SolverOpts.OutputTimes = 0.001:0.2e-5:0.002;
  end

  % Create a shortcut in simulation 5
  simStructs{5}.ModelVars.varR = 0;
  out = proxy.plecs.simulate(MODEL_NAME, simStructs, callback); % the callback processes the responses from every simulation and returns [maxv, maxt]
  for ix = 1:length(inductorValues)
	% Detect if errors occurred in parallel simulation
   if ischar(out{ix})
    disp([' Error in Simulation ', num2str(ix),' for ', num2str(inductorValues(ix)),' μH']);
	% Output maximum current values to console
   else
    % Find maximum current value and index
    [maxv, maxt] = deal(out{ix}(1), out{ix}(2));
 		disp(['Max current for L= ', num2str(inductorValues(ix)),'μH: ', num2str(maxv), ' A at ', num2str(maxt), ' s']);
	end

end
end
