import xmlrpc.client
import jsonrpc_requests		# for JSON-RPC
import collections.abc      # to make jsonrpc_requests usable for Python 3.10+
import os

HOST_ADDRESS	= "http://localhost:1080/RPC2"
MODEL_NAME  = "buck_converter_with_parameter_sweep"	
METHOD			= "XML"				# "XML", "JSON"
SIMULATION_TYPE = "Parallel"      # "Sequential", "Parallel"

# import RPC module
if METHOD == "JSON":
    collections.Mapping = collections.abc.Mapping
    server = jsonrpc_requests.Server(HOST_ADDRESS)
elif METHOD == "XML":
	server = xmlrpc.client.Server(HOST_ADDRESS)

L_values = [40, 60, 80, 100, 120, 140, 160, 180, 200, 220]

start_time = 0.001
end_time = 0.002
number_of_steps = 500
output_times_vec = [start_time + x*(end_time-start_time)/number_of_steps for x in range(number_of_steps)]

simStructs = []

server.plecs.load(os.path.join(os.getcwd(), MODEL_NAME))
server.plecs.scope(MODEL_NAME + "/Scope", "ClearTraces")

# execute selected simulation type
if SIMULATION_TYPE == "Sequential":

    for i, varL in enumerate(L_values):
        simStructs = {"ModelVars": {"varL": varL * 1e-6},
                            "Name" : f"L = {varL}μH",
                            "SolverOpts": {"OutputTimes": output_times_vec}}
        
        if i == 4:
            simStructs["ModelVars"]["varR"] = 0

        try:
            results = server.plecs.simulate(MODEL_NAME, simStructs)

            maxi = max(results["Values"][0])
            maxidx = list(results["Values"][0]).index(maxi)
            maxidx = results["Time"][maxidx]

            print(f"Max current for L={varL}uH: {maxi:.6f}A at {maxidx:.6f}s")

        except Exception as fault:
            print(f"\nThere is a simulation error for case 5 (L={varL}uH) where varR is artificially set to zero. Nevertheless the results for the other cases are still calculated.\n")
            print(str(fault.faultString))

    server.plecs.close(MODEL_NAME)


elif SIMULATION_TYPE == "Parallel":

    for varL in L_values:
        simStructs.append({"ModelVars": {"varL": varL * 1e-6},
                        "Name" : f"L = {varL}μH",
                        "SolverOpts": {"OutputTimes": output_times_vec}})

    simStructs[4]["ModelVars"]["varR"] = 0

    callback = """
    if ischar(result)
        disp(["There is a simulation error for the fifth case (" name ") where varR = 0 is artificially set to zero. Nevertheless the results for the other cases are still calculated"]);
    else
        plecs('scope', './Scope', 'HoldTrace', name);
        
        [maxi, maxidx] = max(result.Values(1,:));
        maxt = result.Time(maxidx);
        result = [maxi, maxt];
    end
    """

    results = server.plecs.simulate(MODEL_NAME, simStructs, callback)
    server.plecs.close(MODEL_NAME)

    for i, varL in enumerate(L_values):
        if isinstance(results[i], list):
            print(f"Max current for L={varL}uH: {results[i][0][0]:.6f}A at {results[i][0][1]:.6f}s")
        else:
            print(f"\nThere is a simulation error for case 5 (L={varL}uH) where varR is artificially set to zero. Nevertheless the results for the other cases are still calculated.\n")
            print(f"{results[i]}")
