import matplotlib.pyplot as plt

def sorted_range(seq):
    return sorted(range(len(seq)), reverse=True, key=seq.__getitem__)

# this function performs:
# 1. read Tj extrem points from extract_extreme_Tj() output csv file
# 2. calculate rainflow/half cycle started from each peak and valley
# 3. store amplitude, ending point, and time span of each half cycle to a csv file
# 4. count full cycles with equivalent amplitudes and write results to a csv file
# 5. return cycle amplitudes and frequencies
# input: Tj_extreme_filename - extract_extreme_Tj() return, filename of Tj extreme points
# output: 1.'Tj_extreme_rainflow.csv' (rainflow of each Tj extreme point) contains four lists:
#             A - amplitude of each rainflow started at each Tj extrem point
#             i_flow_end - the end point of each rainflow
#             t_cyc_start - the starting extreme point of each rainflow
#             t_cyc_end - the ending extreme point of each rainflow
#           2.'Tj_cycle_output.csv' (sorted full Tj cycles) contains two lists:
#             cyc_ampl_s - the amplitudes of cycles in descend order
#             cyc_freq_s - the corresponding frequencies of Tj cycles
# return: cyc_ampl_s, cyc_freq_s
def rainflow_count(t_ex,Tj_ex):
    # N (an integer) is the length of the list
    # A (a list) stores the amplitude of each rainflow
    # i_flow_end (a list) stores end/termination point of each rainflow
    # e.g. i_flow_end[7] = 11 means the rainflow started from #7 extreme point stops
    # at #11 extreme point
    N = len(Tj_ex)
    A = [0]*N
    i_flow_end = [0]*N
    # detect the data list starts with a peak or a valley
    # i_pk_0 and j_vl_0 (integers) are the indices of the first peak and the first
    # valley, respectively
    if Tj_ex[0] > Tj_ex[1]:
        i_pk_0 = 0
        j_vl_0 = 1
    else:
        i_pk_0 = 1
        j_vl_0 = 0

    # rain flow from the first peak
    # i_cur_hi_pk (integer) is the index of the current highest peak
    # i_next_pk (integer) is the index of the next peak higher the current peak
    # end_vl (float) is the value of the flow-end valley
    i_cur_hi_pk = i_pk_0
    # the current peak is one of the highest peaks of all points
    if Tj_ex[i_pk_0] == max(Tj_ex[i_pk_0:N]):
        end_vl = min(Tj_ex[i_pk_0+2:N])
        A[i_pk_0] = Tj_ex[i_pk_0] - end_vl
        i_flow_end[i_pk_0] =  Tj_ex.index(end_vl)
    # the current peak is not one of the highest peaks
    else:
        i_next_pk = next(x for x, Tj in enumerate(Tj_ex[i_pk_0:N]) if Tj > Tj_ex[i_pk_0])
        i_next_pk = i_pk_0 + i_next_pk
        end_vl = min(Tj_ex[i_pk_0:i_next_pk])
        A[i_pk_0] = Tj_ex[i_pk_0] - end_vl
        i_flow_end[i_pk_0] =  Tj_ex.index(end_vl)

    # rain flow from the first valley
    # j_cur_lo_vl (integer) is the index of the current lowest valley
    # i_next_vl (integer) is the index of the next valley lower than the current valley
    # end_pk (float) is the value of the flow-end peak
    j_cur_lo_vl = j_vl_0
    # the current valley is one of the lowest valleys of all points
    if Tj_ex[j_vl_0] == min(Tj_ex[j_vl_0:N]):
        end_pk = max(Tj_ex[j_vl_0+2:N])
        A[j_vl_0] = -(Tj_ex[j_vl_0] - end_pk)
        i_flow_end[j_vl_0] = Tj_ex.index(end_pk)
    # the current valley is not one of the lowest valleys
    else:
        i_next_vl = next(y for y, Tj in enumerate(Tj_ex[j_vl_0:N]) if Tj <Tj_ex[j_vl_0])
        i_next_vl = j_vl_0 + i_next_vl
        end_pk = max(Tj_ex[j_vl_0:i_next_vl])
        A[j_vl_0] = -(Tj_ex[j_vl_0] - end_pk)
        i_flow_end[j_vl_0] =  Tj_ex.index(end_pk)

    # calculate rainflow started from each peak
    i = i_pk_0 + 2
    while i < N-1:
        # the current peak higher than the previous peaks
        if Tj_ex[i] > Tj_ex[i_cur_hi_pk]:
            i_cur_hi_pk = i
            # case I: and no lower than the next peaks
            if Tj_ex[i] == max(Tj_ex[i:N]):
                end_vl = min(Tj_ex[i:N])
                A[i] = Tj_ex[i] - end_vl
                i_flow_end[i] = i + Tj_ex[i:N].index(end_vl)
            # case II: but lower than one of the next peaks
            else:
                i_next_pk = next(x for x, Tj in enumerate(Tj_ex[i:N]) if Tj > Tj_ex[i])
                i_next_pk = i + i_next_pk
                end_vl = min(Tj_ex[i:i_next_pk])
                A[i] = Tj_ex[i] - end_vl
                i_flow_end[i] =  i + Tj_ex[i:i_next_pk].index(end_vl)
        # the current peak no higher than the previous peaks
        else:
            i_pre_pk = [x for x, Tj in enumerate(Tj_ex[0:i]) if Tj >= Tj_ex[i]][-1]
            i_pre_pk = i_pre_pk + 0
            pre_vl = min(Tj_ex[i_pre_pk:i])
            i_next_pk = [x for x, Tj in enumerate(Tj_ex[i:N]) if Tj > Tj_ex[i]]
            # but no lower than the next peaks
            if i_next_pk == []:
                next_min_vl = min(Tj_ex[i:N])
                # case III: the next valleys are higher than
                # the lowest valley between the last higher peak and the current peak
                if next_min_vl > pre_vl:
                    A[i] = Tj_ex[i] - next_min_vl
                    i_flow_end[i] = i + Tj_ex[i:N].index(next_min_vl)
                # case IV: one of the next valleys are no higher than
                # the lowest valley between the last higher peak and the current peak
                else:
                    A[i] = Tj_ex[i] - pre_vl
                    i_flow_end[i] = next(x for x, Tj in enumerate(Tj_ex[i:N]) if Tj <= pre_vl) + i
            # and lower than one of the next peaks
            else:
                i_next_pk = i_next_pk[0] + i
                next_min_vl = min(Tj_ex[i:i_next_pk])
                # case V: the next valleys are higher than
                # the lowest valley between the last higher peak and the current peak
                if next_min_vl > pre_vl:
                    A[i] = Tj_ex[i] - next_min_vl
                    i_flow_end[i] = i + Tj_ex[i:i_next_pk].index(next_min_vl)
                # Case VI: one of the next valleys are no higher than
                # the lowest valley between the last higher peak and the current peak
                else:
                    A[i] = Tj_ex[i] - pre_vl
                    i_flow_end[i] = next(x for x, Tj in
                     enumerate(Tj_ex[i:i_next_pk]) if Tj <= pre_vl) + i
        i = i + 2
    # if Tj_ex list ends with a peak
    if i == N-1:
        A[i] = 0
        i_flow_end[i] = i

    # calculate rainflow started from each valley
    j = j_vl_0 + 2
    while j < N-1:
        # the current valley lower than the previous valleys
        if Tj_ex[j] < Tj_ex[j_cur_lo_vl]:
            j_cur_lo_vl = j
            # case 1: and no higher than the next valleys
            if Tj_ex[j] == min(Tj_ex[j:N]):
                end_pk = max(Tj_ex[j:N])
                A[j] = -(Tj_ex[j] - end_pk)
                i_flow_end[j] = j + Tj_ex[j:N].index(end_pk)
            # case 2: but higher than one of the next valleys
            else:
                i_next_vl = next(y for y, Tj in enumerate(Tj_ex[j:N]) if Tj < Tj_ex[j])
                i_next_vl = j + i_next_vl
                end_pk = max(Tj_ex[j:i_next_vl])
                A[j] = -(Tj_ex[j] - end_pk)
                i_flow_end[j] =  j + Tj_ex[j:i_next_vl].index(end_pk)
        # the current valley no lower than the previous valleys
        else:
            i_pre_vl = [y for y, Tj in enumerate(Tj_ex[0:j]) if Tj <= Tj_ex[j]][-1]
            i_pre_vl = i_pre_vl + 0
            pre_pk = max(Tj_ex[i_pre_vl:j])
            i_next_vl = [y for y, Tj in enumerate(Tj_ex[j:N]) if Tj < Tj_ex[j]]
            # but no higher than the next valleys
            if i_next_vl == []:
                next_max_pk = max(Tj_ex[j:N])
                # case 3: the next peaks are lower than
                # the highest peak between the last lower valley and the current valley
                if next_max_pk < pre_pk:
                    A[j] = -(Tj_ex[j] - next_max_pk)
                    i_flow_end[j] = j + Tj_ex[j:N].index(next_max_pk)
                # case 4: one of the next peaks are no lower than
                # the highest peak between the last lower valley and the current valley
                else:
                    A[j] = -(Tj_ex[j] - pre_pk)
                    i_flow_end[j] = next(y for y, Tj in enumerate(Tj_ex[j:N]) if Tj >= pre_pk) + j
            # and higher than one of the next valleys
            else:
                i_next_vl = i_next_vl[0] + j
                next_max_pk = max(Tj_ex[j:i_next_vl])
                # case 5: the next peaks are lower than
                # the highest peak between the last lower valley and the current valley
                if next_max_pk < pre_pk:
                    A[j] = -(Tj_ex[j] - next_max_pk)
                    i_flow_end[j] = j + Tj_ex[j:i_next_vl].index(next_max_pk)
                # Case 6: one of the next peaks are no lower than
                # the highest peak between the last lower valley and the current valley
                else:
                    A[j] = -(Tj_ex[j] - pre_pk)
                    i_flow_end[j] = next(y for y, Tj in
                     enumerate(Tj_ex[j:i_next_vl]) if Tj >= pre_pk) + j # equal
        j = j + 2
    # if Tj_ex list ends with a valley
    if j == N-1:
        A[j] = 0
        i_flow_end[j] = j

    # count number of half cycles with equivalent amplitudes
    from collections import Counter
    # amplitudes and frequencies of half cycles
    # sorted counting results in in descend order of cycle amplitudes
    cyc_count = Counter(A)
    cyc_ampl = list(cyc_count.keys())
    cyc_freq = list(cyc_count.values())
    index = sorted_range(cyc_ampl)
    # cyc_ampl_s (list) stores the amplitudes of cycles in descend order
    # cyc_freq_s (list) stores the corresponding number of half cycles or occurrences
    cyc_ampl_s = [cyc_ampl[i] for i in index]
    cyc_freq_s = [cyc_freq[i] for i in index]
    # convert half cycles into full cycles
    cyc_freq_s = [i*0.5 for i in cyc_freq_s]
    return cyc_ampl_s, cyc_freq_s



# generate histogram of Tj cycles
# input: cyc_ampl_s, cyc_freq_s - rainflow_count() returns
#         bin_width - the width of bin
# output: 'Tj_cycle_histogram.csv' contains two lists
#          bins - amplitude of Tj cycles
#          hists -  number of Tj cycles
# return: bins, hists - histogram bins, number of cycles (list)
def cycle_histogram(cyc_ampl_s, cyc_freq_s, bin_width):
    # determine the number of bins, N_bin (integer)
    N_bin = int(round(cyc_ampl_s[0]/float(bin_width) - 0.501)) + 1
    # bins (a list) of Tj cycles
    bins = [i*bin_width for i in range(N_bin, -1, -1)]
    # hists (list) number of cycle in each bin
    hists = [0]*N_bin
    # count the number of cycles in each bin
    k = N_bin - 1
    i_bin_start = 0
    while k > 0:
        i_bin_end = next(z for z, ampl in enumerate(cyc_ampl_s) if ampl < k*bin_width)
        hists[N_bin - k - 1] = sum(cyc_freq_s[i_bin_start:i_bin_end])
        i_bin_start = i_bin_end
        k = k - 1
    hists[-1] = sum(cyc_freq_s[i_bin_start:-1])
    cyc_ampl_s_rev = cyc_ampl_s[:]
    cyc_ampl_s_rev.reverse()
    bins_rev = bins[:]
    bins_rev.reverse()
    plt.figure(1)
    plt.hist(cyc_ampl_s_rev, bins_rev, edgecolor = 'Red' )
    return bins, hists

# predict lifetime (lifetime model Nf = a/(dTj)^b)
# input: a, b - lifetime parameters
#         bins, hists - cycle_histogram() returns
#          t_period - time period of load in seconds
# return: lifetime - lifetime of IGBT module in years
def predict_lifetime(a, b, bins, hists, t_period):
    N_bin = len(hists)
    # acc_damage (list) stores damages of cycles in bins
    acc_damage = [0]*N_bin
    k = 0
    while k < N_bin:
        acc_damage[k] = hists[k]/(a*pow(bins[k], -b))
        k = k + 1
    # calculate the accumulated damage in a load cycle
    # Miner's rule (linear damage accumulation)
    # number of this load cycle to failure
    PC = 1/sum(acc_damage)
    # lifetime (year)
    lifetime = PC*t_period/float(3600)/24/365
    return lifetime
