local list = {}

-- Rotate n times by 90 degrees in clockwise direction around the origin. An error is generated if n is not an integer
-- The clockwise direction was chosen as this is the standard behavior when using the PLECS editor.
local function rotate(x, y, n)
	if n == 0 then
		return x, y
	elseif n == 1 then 
		-- clockwise rotation
		x,y = y,x
		for k in ipairs(x) do x[k] = x[k] * -1 end 
		return x, y
	elseif n == 2 then
		-- point reflection
		for k in ipairs(x) do x[k] = x[k] * -1 end
		for k in ipairs(y) do y[k] = y[k] * -1 end
		return x, y
	elseif n == 3 then 
		-- counter clockwise rotation
		x,y = y,x
		for k in ipairs(y) do y[k] = y[k] * -1 end 
		return x, y
	end
	error("The 'rotate' parameter must be an integer multiple of 90.")
	return x, y
end

-- Scale by the provided factor.
local function scale(x, y, scaleX, scaleY)
	if scaleX ~= 1 then
		for k in ipairs(x) do x[k] = x[k] * scaleX end
	end
	if scaleY ~= 1 then
		for k in ipairs(y) do y[k] = y[k] * scaleY end
	end
	return x, y
end

-- Translate by (x0, y0).
local function translate(x, y, x0, y0)
	if x0 ~= 0 then
		for k in ipairs(x) do x[k] = x[k] + x0 end
	end
	if y0 ~= 0 then
		for k in ipairs(y) do y[k] = y[k] + y0 end
	end
	return x, y
end

-- Full coordinate transform. It is assumed that params has been validated before
local function transform(x, y, params)
	local scaleX, scaleY = 1, 1
	if params.flip then scaleY = scaleY * -1 end
	-- We always flip around the x axis before rotating. This is equivalent to flipping the icon about a rotating axis.
	x, y = scale(x, y, scaleX, scaleY)
	x, y = rotate(x, y, params.rotate / 90)
	x, y = translate(x, y, params.x0, params.y0)
	return x, y
end

-- Validates the type of all parameters and throws an error for unknown parameters. Mandatory fields are populated with default values (x0, y0, rotate, flip, wireLength).
-- The returned params table contains valid (copied) entries for all fields.
local function validateInput(x0, y0, params, nTerminals, supportedParams)
	if params == nil then params={} end
	assert(type(params) == "table", "Invalid data type for 'params'. A table ('{}') was expected!")

	local validatedParams = {}

	validatedParams.x0 = x0 or 0
	validatedParams.y0 = y0 or 0
	assert(type(validatedParams.x0) == "number", "Invalid data type for 'x0'. A number was expected!")
	assert(type(validatedParams.y0) == "number", "Invalid data type for 'y0'. A number was expected!")
	
	validatedParams.flip = params.flip or false
	validatedParams.rotate = params.rotate or 0
	assert(type(validatedParams.flip) == "boolean", "Invalid data type for 'flip'. A boolean was expected!")
	assert(type(validatedParams.rotate) == "number", "Invalid data type for 'rotate'. A number was expected!")

	-- Restrict the rotate argument to the range [0, 360)
	validatedParams.rotate = math.fmod(math.fmod(validatedParams.rotate, 360) + 360, 360)

	-- Parse the wireLength parameter
	local wireLength = params.wireLength or 0
	assert(type(wireLength) == "number" or type(wireLength) == "table", "Invalid data type for 'wireLength'. Either a number or a table of numbers was expected!")
	if type(wireLength) == "number" then
		validatedParams.wireLength = {}
		for i = 1, nTerminals do 
			validatedParams.wireLength[i] = wireLength
		end
	elseif type(wireLength) == "table" then
		assert((#wireLength == nTerminals), "Invalid number of arguments passed as 'wireLength'.")
		validatedParams.wireLength = {}
		for i = 1, nTerminals do 
			validatedParams.wireLength[i] = wireLength[i]
		end
	end

	-- Check additional parameters

	-- shiftGate
	if params.shiftGate ~= nil and supportedParams.shiftGate then
		assert(type(params.shiftGate) == "boolean", "Invalid data type for 'shiftGate'. A boolean was expected!")
		validatedParams.shiftGate = params.shiftGate
	end
	-- deviceType
	if params.deviceType ~= nil and supportedParams.deviceType then
		assert(type(params.deviceType) == "string", "Invalid data type for 'deviceType'. A string was expected!")
		validatedParams.deviceType = params.deviceType
	end
	--showPolarity
	if params.showPolarity ~= nil and supportedParams.showPolarity then
		assert(type(params.showPolarity) == "boolean", "Invalid data type for 'showPolarity'. A boolean was expected!")
		validatedParams.showPolarity = params.showPolarity
	end
	--showArrow
	if params.showArrow ~= nil and supportedParams.showArrow then
		assert(type(params.showArrow) == "boolean", "Invalid data type for 'showArrow'. A boolean was expected!")
		validatedParams.showArrow = params.showArrow
	end
	-- showCore
	if params.showCore ~= nil and supportedParams.showCore then
		assert(type(params.showCore) == "boolean", "Invalid data type for 'showCore'. A boolean was expected!")
		validatedParams.showCore = params.showCore
	end
	-- polarity
	if params.polarity ~= nil and supportedParams.polarity then
		assert(type(params.polarity) == "string", "Invalid data type for 'polarity'. A string was expected!")
		validatedParams.polarity = params.polarity
	end

	-- Check user preferences

	-- DrawANSI
	validatedParams.drawANSI = Preferences:get('DrawANSI')

	-- Report unknown parameters
	for key in pairs(params) do
		assert(validatedParams[key] ~= nil, "Unknown parameter '" .. key .."'.")
	end

	return validatedParams
end

-- Helper function for drawing a table of Vectors
local function drawLines(x, y)
	assert(#x == #y, "Fatal error occurred, please report this to support@plexim.com")
	for k in ipairs(x) do
		Icon:line(x[k],y[k])
	end
end

-- resistor(-5, 5, {rotate=0, flip=false, wireLength={0, 0}})
function list.resistor(x0, y0, params)
	local params = validateInput(x0, y0, params, 2)
	local wire = params.wireLength
	
	if params.drawANSI then
		local x = {Vector{            0,     0,   5, -5, 5, -5,  5,    0,            0}}
		local y = {Vector{-20 - wire[1], -12.5, -10, -5, 0,  5, 10, 12.5, 20 + wire[2]}}
		drawLines(transform(x, y, params))
	else
		local x = {Vector{  5,  5, -5,  -5,   5}, Vector{  0,             0}, Vector{ 0,            0}}
		local y = {Vector{-15, 15, 15, -15, -15}, Vector{-15, -20 - wire[1]}, Vector{15, 20 + wire[2]}}
		drawLines(transform(x, y, params))
	end
end

-- capacitor(-5, 5, {rotate=0, flip=false, wireLength={0, 0}, showPolarity=true})
function list.capacitor(x0, y0, params)
	local params = validateInput(x0, y0, params, 2, {showPolarity=true})
	local wire = params.wireLength
	if params.drawANSI then
		local x = {Vector{ -8, 8}, Vector{-8, -6, -2, 2, 6, 8}, Vector{0,              0}, Vector{0,            0}}
		local y = {Vector{-2, -2}, Vector{ 4,  3,  2, 2, 3, 4}, Vector{-2, -10 - wire[1]}, Vector{2, 10 + wire[2]}}
		drawLines(transform(x, y, params))
	else
		local x = {Vector{ 8, -8}, Vector{8, -8}, Vector{0,              0}, Vector{0,            0}}
		local y = {Vector{-2, -2}, Vector{2,  2}, Vector{-2, -10 - wire[1]}, Vector{2, 10 + wire[2]}}
		drawLines(transform(x, y, params))

		if params.showPolarity then
			local xMeas, yMeas = {Vector{4, 8}, Vector{6, 6}}, {Vector{-8, -8}, Vector{-10, -6}}
			drawLines(transform(xMeas, yMeas, params))
		end
	end
end

-- inductor(-5, 5, {rotate=0, flip=false, wireLength={0, 0}, showArrow=true})
function list.inductor(x0, y0, params)
	local params = validateInput(x0, y0, params, 2, {showArrow=true})
	local wire = params.wireLength

	-- draw wires
	local x = {Vector{            0,   0}, Vector{0,             0}}
	local y = {Vector{-20 - wire[1], -14}, Vector{14, 20 + wire[2]}}
	drawLines(transform(x, y, params))

	-- draw arcs and arc extensions
	local K, rot, flip = 1, params.rotate, params.flip
	local x_arc, y_arc, rx, ry, start, span = 0, 0, 3.5*K, 3.5*K, 90+rot, 180
	if start>360 then start=start-360 elseif start<0 then start=start+360 end
	if rot == 90 or rot == 270 then span = -span end
	local x_line, y_line = 0, 0
	for deltaY = -10.5*K, 10.5*K, 7*K do
		x_arc = {Vector{-1.5*K}}
		y_arc = {Vector{deltaY}}
		x_arc,y_arc = transform(x_arc,y_arc,params)
		Icon:arc(x_arc[1][1], y_arc[1][1], rx, ry, start, span)
	end
	for yPos = -14*K, 14*K, 7*K do
		drawLines(transform({Vector{0, -1.5*K}}, {Vector{yPos, yPos}}, params))
	end

	if params.showArrow then
		local xMeas, yMeas = {Vector{-2, 0, 2}}, {Vector{-16, -19, -16}}
		drawLines(transform(xMeas, yMeas, params))
	end
end

-- transformer(-5, 5, {rotate=0, flip=false, wireLength={0, 0, 0, 0}, showCore=true, showPolarity=true, polarity="+-"})
function list.transformer(x0, y0, params)
	local params = validateInput(x0, y0, params, 4, {showPolarity=true, polarity=true, showCore=true})
	local wire = params.wireLength

	-- Primary winding
	local xShift, yShift = transform({Vector{-10}}, {Vector{0}}, params)
	list.inductor(xShift[1][1], yShift[1][1], {rotate=params.rotate+180, flip=params.flip, wireLength={wire[2], wire[1]}})
	
	-- Secondary winding
	xShift, yShift = transform({Vector{10}}, {Vector{0}}, params)
	list.inductor(xShift[1][1], yShift[1][1], {rotate=params.rotate, flip=params.flip, wireLength={wire[3], wire[4]}})

	-- magnetic core decoration
	if params.showCore then
		local xCore = {Vector{-1.5, -1.5}, Vector{1.5, 1.5}}
		local yCore = {Vector{ -15,   15}, Vector{-15,  15}}
		drawLines(transform(xCore, yCore, params))
	end

	-- polarity
	if params.showPolarity then
		local xPol1, xPol2, yPol1, yPol2
		if params.polarity == '+' or params.polarity == "++" then
			xPol1, yPol1 = transform({Vector{-6.0}}, {Vector{-18.0}}, params)
			xPol2, yPol2 = transform({Vector{ 6.5}}, {Vector{-17.5}}, params)
		elseif params.polarity == "-" or params.polarity == "--" then
			xPol1, yPol1 = transform({Vector{-6.0}}, {Vector{18.0}}, params)
			xPol2, yPol2 = transform({Vector{ 6.5}}, {Vector{17.5}}, params)
		elseif params.polarity == "+-" then
			xPol1, yPol1 = transform({Vector{-6.0}}, {Vector{-18.0}}, params)
			xPol2, yPol2 = transform({Vector{ 6.5}}, {Vector{17.5}}, params)
		elseif params.polarity == "-+" then
			xPol1, yPol1 = transform({Vector{-6.0}}, {Vector{18.0}}, params)
			xPol2, yPol2 = transform({Vector{ 6.5}}, {Vector{-17.5}}, params)
		else
			assert(false, "Unsupported 'polarity' (Use +, -, +-, or -+)")
		end
		Icon:circle(xPol1[1][1], yPol1[1][1], 1, false)
		Icon:circle(xPol2[1][1], yPol2[1][1], 0.5, false)
	end
end

-- diode(-5, 5, {rotate=0, flip=false, wireLength={0, 0}})
function list.diode(x0, y0, params)
	-- Note that changes to this function also affect the thyristor, the mosfetDiode, and the igbtDiode
	local params = validateInput(x0, y0, params, 2)
	local wire = params.wireLength

	local x = {Vector{-7, 7,  0, -7}, Vector{-7,  7}, Vector{ 0,             0}, Vector{0,            0}}
	local y = {Vector{ 6, 6, -6,  6}, Vector{-6, -6}, Vector{-6, -15 - wire[1]}, Vector{6, 15 + wire[2]}}
	drawLines(transform(x, y, params))
end

-- thyristor(-5, 5, {rotate=0, flip=false, wireLength={0, 0, 0}})
function list.thyristor(x0, y0, params)
	local params = validateInput(x0, y0, params, 3)
	local wire = params.wireLength

	list.diode(params.x0, params.y0, {rotate=params.rotate, flip=params.flip, wireLength={wire[1], wire[3]}})
	
	local x = {Vector{ 0,  0, -4,  -7, -15 - wire[2]}}
	local y = {Vector{-6, -6, -6, -10,           -10}}
	drawLines(transform(x, y, params))
end

-- igbt(-5, 5, {rotate=0, flip=false, wireLength={0, 0, 0}, deviceType="n", shiftGate=false})
function list.igbt(x0, y0, params)
	local params = validateInput(x0, y0, params, 3, {deviceType=true, shiftGate=true})
	local wire = params.wireLength
	local gateYOffset = params.shiftGate and 8 or 0

	local igbtType = params.deviceType or "n"
	local xArrow, xArrow
	if igbtType == "n" then
		xArrow = Vector{-5.650, -1.667, -3.864}
		yArrow = Vector{ 8.634,  9.000,  5.658}
	elseif igbtType == "p" then
		xArrow = Vector{-6.136, -8.333, -4.350}
		yArrow = Vector{ 8.342,  5.000,  5.366}
	else
		assert(false, "Unsupported 'deviceType' (Use n or p)")
	end

	local x = {Vector{            0,   0, -10, -10,  0,            0}, Vector{-10, -10}, Vector{-14, -14}, Vector{        -14, -20 - wire[2]}, xArrow}
	local y = {Vector{-20 - wire[1], -10,  -4,   4, 10, 20 + wire[3]}, Vector{ -8,   8}, Vector{ -8,   8}, Vector{gateYOffset,   gateYOffset}, yArrow}
	drawLines(transform(x, y, params))
end

-- igbtDiode(-5, 5, {rotate=0, flip=false, wireLength={0, 0, 0}, deviceType="p", shiftGate=true})
function list.igbtDiode(x0, y0, params)
	local params = validateInput(x0, y0, params, 3, {deviceType=true, shiftGate=true})

	list.igbt(params.x0, params.y0, {rotate=params.rotate, flip=params.flip, wireLength={params.wireLength[1], params.wireLength[2], params.wireLength[3]}, deviceType=params.deviceType, shiftGate=params.shiftGate})
	
	-- Rotation of diode origin
	local xD, yD = transform({Vector{10}}, {Vector{0}}, params)
	list.diode(xD[1][1], yD[1][1], {rotate=params.rotate, flip=params.flip, wireLength={-1, -1}})

	-- iGBT/diode connections
	local xConn = {Vector{  0,  10}, Vector{ 0, 10}}
	local yConn = {Vector{-14, -14}, Vector{14, 14}}
	drawLines(transform(xConn, yConn, params))
end

-- mosfet(-5, 5, {rotate=0, flip=false, wireLength={0, 0, 0}, deviceType="n", shiftGate=false})
function list.mosfet(x0, y0, params)
	local params = validateInput(x0, y0, params, 3, {deviceType=true, shiftGate=true})
	local wire = params.wireLength
	local gateYOffset = params.shiftGate and 10 or 0

	local mosType = params.deviceType or "n"
	local yArrow = Vector{-2, 0, 2}
	local xArrow
	if mosType == "n"then
		xArrow = Vector{-4, -8, -4}
	elseif mosType == "p" then
		xArrow = Vector{-8, -4, -8}
	else
		assert(false, "Unsupported 'deviceType' (Use n or p)")
	end

	local x = {Vector{            0,   0, -10}, Vector{-10, -10}, Vector{-10, -10}, Vector{-10, -10}, Vector{-10, 0,  0}, Vector{-10,  0,            0}, Vector{-15, -15}, Vector{        -15, -20 - wire[2]}, xArrow}
	local y = {Vector{-20 - wire[1], -10, -10}, Vector{-13,  -7}, Vector{ -3,   3}, Vector{  7,  13}, Vector{  0, 0, 10}, Vector{ 10, 10, 20 + wire[3]}, Vector{-10,  10}, Vector{gateYOffset,   gateYOffset}, yArrow}
	drawLines(transform(x, y, params))
end

-- mosfetDiode(-5, 5, {rotate=0, flip=false, wireLength={0, 0, 0}, deviceType="p", shiftGate=true})
function list.mosfetDiode(x0, y0, params)
	local params = validateInput(x0, y0, params, 3, {deviceType=true, shiftGate=true})

	list.mosfet(params.x0, params.y0, {rotate=params.rotate, flip=params.flip, wireLength={params.wireLength[1], params.wireLength[2], params.wireLength[3]}, deviceType=params.deviceType, shiftGate=params.shiftGate})
	
	-- Rotation of diode origin
	local xD, yD = transform({Vector{12}}, {Vector{0}}, params)
	list.diode(xD[1][1], yD[1][1], {rotate=params.rotate, flip=params.flip, wireLength={-1, -1}})

	-- mosfet/diode connections
	local xConn = {Vector{  0,  12}, Vector{ 0, 12}}
	local yConn = {Vector{-14, -14}, Vector{14, 14}}
	drawLines(transform(xConn, yConn, params))
end

-- bjt(-5, 5, {rotate=0, flip=false, wireLength={0, 0, 0}, deviceType="npn"})
function list.bjt(x0, y0, params)
	local params = validateInput(x0, y0, params, 3, {deviceType=true})
	local wire = params.wireLength

	local bjtType = params.deviceType or "npn"
	local xArrow, xArrow
	if bjtType == "npn" then
		xArrow = Vector{-5.650, -1.667, -3.864}
		yArrow = Vector{ 8.634,  9.000,  5.658}
	elseif bjtType == "pnp" then
		xArrow = Vector{-6.136, -8.333, -4.350}
		yArrow = Vector{ 8.342,  5.000,  5.366}
	else
		assert(false, "Unsupported 'deviceType' (Use npn or pnp)")
	end

	local x = {Vector{            0,   0, -10, -10,  0,            0}, Vector{-10, -10}, Vector{-10, -20 - wire[2]}, xArrow}
	local y = {Vector{-20 - wire[1], -10,  -4,   4, 10, 20 + wire[3]}, Vector{ -8,   8}, Vector{  0,             0}, yArrow}
	drawLines(transform(x, y, params))
end

-- jfet(-5, 5, {rotate=0, flip=false, wireLength={0, 0, 0}, deviceType="n", shiftGate=false})
function list.jfet(x0, y0, params)
	local params = validateInput(x0, y0, params, 3, {deviceType=true, shiftGate=true})
	local wire = params.wireLength
	local gateYOffset = params.shiftGate and 10 or 0

	local jfetType = params.deviceType or "n"
	local xArrow, xArrow
	if jfetType == "n" then
		xArrow = Vector{-16, -12, -16}
		yArrow = Vector{  2,   0,  -2} + gateYOffset
	elseif jfetType == "p" then
		xArrow = Vector{-12, -16, -12}
		yArrow = Vector{  2,   0,  -2} + gateYOffset
	else
		assert(false, "Unsupported 'deviceType' (Use n or p)")
	end

	local x = {Vector{            0,   0, -10, -10,  0,            0}, Vector{-10, -10}, Vector{        -10, -20 - wire[2]}, xArrow}
	local y = {Vector{-20 - wire[1], -10, -10,  10, 10, 20 + wire[3]}, Vector{-12,  12}, Vector{gateYOffset,   gateYOffset}, yArrow}
	drawLines(transform(x, y, params))
end

function list.node(x0,y0)
	Icon:circle(x0,y0,1.25,true)
end

--
-- Legacy functions (for PLECS 4.9)
--

-- Convert the legacy transform to the new transform
local function convert_transform(params)
	if type(params) ~= "table" then return {} end
	
	local flip = params.flip or false
	local rotate = params.rot or 0
	if rotate<0 then rotate = rotate +360 elseif rotate>360 then rotate = rotate-360 end
	rotate = math.max(rotate, 0)
	rotate = math.floor(rotate/90)*90

	if flip then 
		params.rotate = rotate + 180
		params.flip = true -- The previous library flipped the component even if a wrong type was passed, as anything other than nil or false is evaluated to true in lua...
	else
		params.rotate = -rotate
		params.flip = false
	end
	params.rot = nil

	-- Ignore the len parameter rather than not drawing anything
	params.len = nil

	return params
end

-- Deprecated function
function list.Resistor(x0, y0, params)
	local params = convert_transform(params)
	list.resistor(x0, y0, params)
end

-- Deprecated function
function list.Capacitor(x0, y0, params)
	local params = convert_transform(params)

	if params.meas then
		params.showPolarity = true
		params.meas = nil
	end

	list.capacitor(x0, y0, params)
end

-- Deprecated function
function list.Diode(x0, y0, params)
	local params = convert_transform(params)

	if params.ter ~= nil then
		params.wireLength = {params.ter - 5, params.ter - 5}
		params.ter = nil
	end

	list.diode(x0, y0, params)
end

-- Deprecated function
function list.Thyristor(x0, y0, params)
	local params = convert_transform(params)
	list.thyristor(x0, y0, params)
end

-- Deprecated function
function list.IGBT(x0, y0, params)
	local params = convert_transform(params)
	list.igbt(x0, y0, params)
end

-- Deprecated function
function list.IGBTD(x0, y0, params)
	local params = convert_transform(params)
	list.igbtDiode(x0, y0, params)
end

-- Deprecated function
function list.MOSFET(x0, y0, params)
	local params = convert_transform(params)
	list.mosfet(x0, y0, params)
end

-- Deprecated function
function list.MOSFETD(x0, y0, params)
	local params = convert_transform(params)
	list.mosfetDiode(x0, y0, params)
end

-- Deprecated function
function list.Inductor(x0, y0, params)
	-- In the previous lib, the inductor had a different orientation
	local params = params or {}
	if params.rot ~= nil then
		params.rot = params.rot - 90
	else
		params.rot = -90
	end
	params = convert_transform(params)

	if params.meas then
		params.showArrow = true
		params.meas = nil
	end

	local ter
	if params.ter == nil then ter = true else ter = params.ter end
	if not ter then
		params.wireLength = {-6, -6}
	end
	params.ter = nil

	list.inductor(x0, y0, params)
end

-- Deprecated function
function list.Winding(x0, y0, params)
	if params == nil then params = {} end
	params.meas = nil
	list.Inductor(x0, y0,params)
end

-- Deprecated function
function list.Transformer(x0, y0, params)
	local params = convert_transform(params)

	if params.core then
		params.showCore = true
		params.core = nil
	end

	local ter
	if params.ter == nil then ter = false else ter = params.ter end
	if not ter then
		params.wireLength = {-6, -6, -6, -6}
	end
	params.ter = nil

	list.transformer(x0, y0, params)
end

-- Deprecated function
function list.Node(x0, y0)
	list.node(x0, y0)
end

return list
