function b = plCheck(fcn, val, opt1, opt2, opt3, opt4)
%%%%%%   DEFINE TSP DATA TYPES FOR MATLAB HELPER FUNCTIONS:
% 'Finite' is a number excluding +/-Inf and NaN and empty []
%  *** TSP DEVS exclude +/-Inf and NaN ***
% 'Vector' is a Scalar or a Vector of Finite values
%          this data is equivalent to what in Lua we call ScalarOrVector data.
% 'Scalar' is a Vector of length 1x1

% INTERESTING FACTS:
% Empty text widgets evaluate as NaN (which is a 1x1 matrix!)
% Variables all have global scope in a file (doesn't matter if in 'if' etc)

switch(fcn)
    case 'finite'
        b = isreal(val) && isnumeric(val) ... % these functions always return boolean
            && (numel(val) > 0) ... % reject `[]`
            && ~any(isnan(val)) && ~any(isinf(val)); % these functions return vectors
    case 'nonEmpty'
        % Note: 'nonEmpty' is not a required check here. It is redundant
        % 'finite' as a constraint in the MATLAB functions, however, it is
        % commonly used as a constraint in the model files, so adding this check
        % for compatibility with 'Constraints' listed there.
        b = plCheck('finite', val);
    case 'integer'
        b = plCheck('finite', val) && all(mod(val, 1) == 0);
    case 'scalar'
        b = plCheck('finite', val) && isscalar(val);
    case 'positive'
        b = plCheck('finite', val) && all(val > 0);
    case 'nonNegative'
        b = plCheck('finite', val) && all(val >= 0);
    case 'Constraints'
        % opt1 will be a string comma separated list of the above basic checks,
        %  same as the 'Constraints' in the PLECS file
        constraintsList = strsplit(opt1, ',');
        for i = 1:length(constraintsList)
            if ~plCheck(constraintsList{i}, val)
                b = false;
                return;
            end
        end
        b = true;

    case 'inClosedInterval' % isInClosedInterval(val, minLimit, maxLimit)
        minLimit = opt1;
        maxLimit = opt2;
        if ~plCheck('Constraints', minLimit, 'scalar') ...
        || ~plCheck('Constraints', maxLimit, 'scalar')
            plecs('error', 'Invalid limits to plCheck(''inClosedInterval'').');
            b = false; return;
        end
        b = ~any(val < minLimit) && ~any(maxLimit < val) && plCheck('finite', val);

    case 'inSet'
        setList = opt1;
        if ~plCheck('finite', setList)
            plecs('error', 'Invalid setList provided to plCheck(''inSet'').');
            b = false; return;
        end
        if ~plCheck('finite', val)
            b = false; return;
        end
        for i = 1:numel(val)
            if ~any(setList == val(i))
                b = false; return;
            end
        end
        b = true;
    otherwise
        plecs('error', ['Invalid first argument to plCheck(): ''' fcn '''']);
        b = false;
end

end
