--[[
  Copyright (c) 2023 by Plexim GmbH
  All rights reserved.

  A free license is granted to anyone to use this software for any legal
  non safety-critical purpose, including commercial applications, provided
  that:
  1) IT IS NOT USED TO DIRECTLY OR INDIRECTLY COMPETE WITH PLEXIM, and
  2) THIS COPYRIGHT NOTICE IS PRESERVED in its entirety.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
  OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
  AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
  OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
  SOFTWARE.
--]]

local iL = { }
local R = 7.75 -- Radius for block and badge corners
local patchSize = 8  -- Width of patch

local color = {
  plexim_blue = { 30, 155, 218 },
  white = {light = {255, 255, 255}, dark = {255, 255, 255}}
}

local function _drawEmptyPatch(x1,y1,x2,y2,pos)
  local ptx = {}
  local pty = {}
  local radius = R-0.1
  if pos then
    for i = 0, 90 do
      local angle = i * math.pi / 180
      table.insert(ptx, x1 + radius * math.cos( angle ))
      table.insert(pty, y1 + radius * math.sin( angle ))
    end
    for i = 180, 270 do
      local angle = i * math.pi / 180
      table.insert(ptx, x2 + radius * math.cos( angle ))
      table.insert(pty, y2 + radius * math.sin( angle ))
    end
  else
    for i = 90, 180 do
      local angle = i * math.pi / 180
      table.insert(ptx, x1 + radius * math.cos( angle ))
      table.insert(pty, y1 + radius * math.sin( angle ))
    end
    for i = 270, 360 do
      local angle = i * math.pi / 180
      table.insert(ptx, x2 + radius * math.cos( angle ))
      table.insert(pty, y2 + radius * math.sin( angle ))
    end
  end
  Icon:patch(ptx, pty)
end

local function _drawPatchBorders(x1,y1,x2,y2,patchSize,pos)
  local dir = nil
  if pos then
    dir = -90
  else
    dir = 90
  end
  Icon:arc(x1,y1,R,R,180,dir)
  Icon:arc(x2,y2,R,R,0,dir)
  Icon:line({x1-R+0.5, x1-R+patchSize-0.5}, {y1, y1})
  Icon:line({x2+R-patchSize+0.5, x2+R-0.5}, {y2, y2})
end

local function _drawDiode(x,y,g)
  Icon:line({x+2.5,x+2.5},{y-2.25*g,y-6.25*g})
  Icon:line({x+1.5,x-2,x-2,x+1.5},{y-4.25*g,y-6.25*g,y-2.25*g,y-4.25*g})
  Icon:line({x+1,x-1.5,x-1.5,x+1},{y-4.25*g,y-5.25*g,y-3.25*g,y-4.25*g})
  Icon:line({x+1,x-1,x-1,x+1},{y-4.25*g,y-4.25*g,y-4.25*g,y-4.25*g})
end

function iL.drawPatch()

  Icon:color(color.plexim_blue)
  local flipped = Block:get("Flipped")
  local direction = Block:get("Direction")
  local frame = Block:get("Frame")
  local x_start = frame[1]
  local y_start = frame[2]
  local x_total = frame[3]
  local y_total = frame[4]
  local delta = (patchSize + R)/2
  local x, y, g, position
  local xT = nil
  local yT = nil

  if direction == 0.0 then
    x = -x_start+y_start
    y = x_start-y_start
    if flipped then
      xT = x_start + R/2
      yT = y_start + delta
      position = 'upLeft'
      g = 1
    else
      xT = x_start + R/2
      yT = -y_start - delta
      position = 'upRight'
      g = -1
    end
  end
  if direction == 0.25 then
    x = 0
    y = 0
    if flipped then
      xT = -x_start - delta
      yT = y_start + R/2
      position = 'upRight'
      g = -1
    else
      xT = -x_start - delta
      yT = -y_start - R/2
      position = 'lowRight'
      g = 1
    end
  end
  if direction == 0.5 then
    x = -x_start+y_start
    y = x_start-y_start
    if flipped then
      xT = -x_start - R/2
      yT = -y_start - delta
      position = 'lowRight'
      g = 1
    else
      xT = -x_start - R/2
      yT = y_start + delta
      position = 'lowLeft'
      g = -1
    end
  end
  if direction == 0.75 then
    x = 0
    y = 0
    if flipped then
      xT = x_start + delta
      yT = -y_start - R/2
      position = 'lowLeft'
      g = -1
    else
      xT = x_start + delta
      yT = y_start + R/2
      position = 'upLeft'
      g = 1
    end
  end

  if position == 'upLeft' then
    _drawEmptyPatch((x_start+patchSize+x), (y_start+y), (x_start+R+x), (y_start+R+y), true)
    Icon:color("normalFg")
    _drawPatchBorders((x_start+R+x), (y_start+R+y), (x_start+patchSize+x), (y_start+y), patchSize, true)
    Icon:color(color.white)
    _drawDiode((x_start+R+x), (y_start+R+y),g)
  elseif position == 'upRight' then
    _drawEmptyPatch((-x_start-patchSize-x), (y_start+y), (-x_start-R-x), (y_start+R+y), false)
    Icon:color("normalFg")
    _drawPatchBorders((-x_start-patchSize-x), (y_start+y), (-x_start-R-x), (y_start+R+y), patchSize, false)
    Icon:color(color.white)
    _drawDiode((-x_start-patchSize-x), (y_start+y),g)
  elseif position == 'lowRight' then
    _drawEmptyPatch((-x_start-R-x), (-y_start-R-y), (-x_start-patchSize-x), (-y_start-y), true)
    Icon:color("normalFg")
    _drawPatchBorders((-x_start-patchSize-x), (-y_start-y), (-x_start-R-x), (-y_start-R-y), patchSize, true)
    Icon:color(color.white)
    _drawDiode((-x_start-patchSize-x), (-y_start-y),g)
  elseif position == 'lowLeft' then
    _drawEmptyPatch((x_start+R+x), (-y_start-R-y), (x_start+patchSize+x), (-y_start-y), false)
    Icon:color("normalFg")
    _drawPatchBorders((x_start+R+x), (-y_start-R-y), (x_start+patchSize+x), (-y_start-y), patchSize, false)
    Icon:color(color.white)
    _drawDiode((x_start+R+x), (-y_start-R-y),g)
  end
end

local function _drawTriggerLines(x,y, trigger_type)
  if trigger_type == "rising" then
    Icon:line({x, x}, {y-3, y+3})
    Icon:line({x, x+2.5}, {y-3, y-3})
    Icon:line({x, x-2.5}, {y+3, y+3}) 
    Icon:line({x-2, x}, {y+1, y-1})
    Icon:line({x+2, x}, {y+1, y-1})

  elseif trigger_type == "falling" then
    Icon:line({x, x}, {y-3, y+3})
    Icon:line({x, x+2.5}, {y+3, y+3})
    Icon:line({x, x-2.5}, {y-3, y-3})  
    Icon:line({x-2, x}, {y-1, y+1})
    Icon:line({x+2, x}, {y-1, y+1})

  elseif trigger_type == "either" then
    x = x - 2.5
    Icon:line({x, x}, {y-3, y+3})
    Icon:line({x, x+2.5}, {y-3, y-3})
    Icon:line({x, x-2.5}, {y+3, y+3})  
    Icon:line({x-2, x}, {y+1, y-1})
    Icon:line({x+2, x}, {y+1, y-1})
    x = x + 5
    Icon:line({x, x}, {y-3, y+3})
    Icon:line({x, x+2.5}, {y+3, y+3})
    Icon:line({x, x-2.5}, {y-3, y-3})  
    Icon:line({x-2, x}, {y-1, y+1})
    Icon:line({x+2, x}, {y-1, y+1})
  end
end

function iL.drawTrigger(x, y, triggerType)
-- draw trigger symbol for blocks that can be triggered
  triggerType = tonumber(triggerType)
  if triggerType > 3 then
    return
  end
  local TriggerTypeName = {"rising", "falling", "either"} -- must match block dialog
  local t = TriggerTypeName[triggerType]

  local flipped = Block:get("Flipped")
  local direction = Block:get("Direction")

  Icon:color("normalFg")

  if direction == 0.0 then
    if flipped then
     _drawTriggerLines(-y, x, t)
    else
     _drawTriggerLines(y, x, t)
    end
  end
  if direction == 0.25 then
    if flipped then
     _drawTriggerLines(-x, -y, t)
    else
     _drawTriggerLines(-x, y, t)
    end  
  end
  if direction == 0.5 then
    if flipped then
     _drawTriggerLines(y, -x, t)
    else
     _drawTriggerLines(-y, -x, t)
    end 
  end
  if direction == 0.75 then
    if flipped then
     _drawTriggerLines(x, y, t) 
    else
     _drawTriggerLines(x, -y, t)
    end
  end
end

return iL