/*
   Copyright (c) 2020 by Plexim GmbH
   All rights reserved.

   A free license is granted to anyone to use this software for any legal
   non safety-critical purpose, including commercial applications, provided
   that:
   1) IT IS NOT USED TO DIRECTLY OR INDIRECTLY COMPETE WITH PLEXIM, and
   2) THIS COPYRIGHT NOTICE IS PRESERVED in its entirety.

   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
   OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
   SOFTWARE.
 */

#ifndef SFP_H_
#define SFP_H_

#include "plexim/hw_wrapper.h"

#ifdef __cplusplus
extern "C"
{
#endif

extern float* plxSfpOutputPtr[];

#define plxSetSFPOut(p, n, x) \
{ \
   *(plxSfpOutputPtr[p] + ((2*((n) & 0xfffe)) | (n & 1))) = x; \
}

/** @brief Send value to SFP0 buffer
 * 
 * @param n index of SFP0 output
 * @param x value to set
 */ 
#define setSFP0Out(n, x) plxSetSFPOut(0, n, x)

/**
 * @brief Send value to SFP1 buffer
 * 
 * @param n index of SFP2 output
 * @param x value to set
 */ 
#define setSFP1Out(n, x) plxSetSFPOut(1, n, x)

/**
 * @brief Send value to SFP2 buffer
 * 
 * @param n index of SFP2 output
 * @param x value to set
 */ 
#define setSFP2Out(n, x) plxSetSFPOut(2, n, x)

/**
 * @brief Send value to SFP3 buffer
 * 
 * @param n index of SFP3 output
 * @param x value to set
 */ 
#define setSFP3Out(n, x) plxSetSFPOut(3, n, x)

/** @brief Send value to SFP4 buffer
 * 
 * @param n index of SFP4 output
 * @param x value to set
 */ 
#define setSFP4Out(n, x) plxSetSFPOut(4, n, x)

/**
 * @brief Send value to SFP5 buffer
 * 
 * @param n index of SFP5 output
 * @param x value to set
 */ 
#define setSFP5Out(n, x) plxSetSFPOut(5, n, x)

/**
 * @brief Send value to SFP6 buffer
 * 
 * @param n index of SFP6 output
 * @param x value to set
 */ 
#define setSFP6Out(n, x) plxSetSFPOut(6, n, x)

/**
 * @brief Send value to SFP7 buffer
 * 
 * @param n index of SFP7 output
 * @param x value to set
 */ 
#define setSFP7Out(n, x) plxSetSFPOut(7, n, x)

extern int plxSfpInitialized[];
extern float* plxSfpInputPtr[];
extern const float* plxSfpInitData[];

#define plxGetSFPIn(p, n) \
   (plxSfpInitialized[p] ? (*(plxSfpInputPtr[p] + (2*(n) - (((n) & 0x1) ? 1 : 0)))) : (*(plxSfpInitData[p] + (n))))

/**
 * @brief Returns the conversion result of the n-th SFP0 input.
 * 
 *
 * @param n ID of SFP0 input.
 */   
#define getSFP0In(n) plxGetSFPIn(0, n)

/**
 * @brief Returns the conversion result of the n-th SFP1 input.
 * 
 *
 * @param n index of SFP1 input.
 */   
#define getSFP1In(n) plxGetSFPIn(1, n)

/**
 * @brief Returns the conversion result of the n-th SFP2 input.
 * 
 *
 * @param n index of SFP2 input.
 */   
#define getSFP2In(n) plxGetSFPIn(2, n)

/**
 * @brief Returns the conversion result of the n-th SFP3 input.
 * 
 *
 * @param n index of SFP3 input.
 */   
#define getSFP3In(n) plxGetSFPIn(3, n)

/**
 * @brief Returns the conversion result of the n-th SFP0 input.
 * 
 *
 * @param n index of SFP4 input.
 */   
#define getSFP4In(n) plxGetSFPIn(4, n)

/**
 * @brief Returns the conversion result of the n-th SFP1 input.
 * 
 *
 * @param n index of SFP5 input.
 */   
#define getSFP5In(n) plxGetSFPIn(5, n)

/**
 * @brief Returns the conversion result of the n-th SFP2 input.
 * 
 *
 * @param n index of SFP6 input.
 */   
#define getSFP6In(n) plxGetSFPIn(6, n)

/**
 * @brief Returns the conversion result of the n-th SFP3 input.
 * 
 *
 * @param n index of SFP7 input.
 */   
#define getSFP7In(n) plxGetSFPIn(7, n)

void plxInitSFP();

u32 plxGetNumStepsSFPInputs();
u32 plxGetNumStepsSFPOutputs();

void setupSFPIn(u32 aPort, u32 aNumInputs, const float* aInitData);

#define setupSFP0In(aNumInputs, aInitData) setupSFPIn(0, aNumInputs, aInitData)
#define setupSFP1In(aNumInputs, aInitData) setupSFPIn(1, aNumInputs, aInitData)
#define setupSFP2In(aNumInputs, aInitData) setupSFPIn(2, aNumInputs, aInitData)
#define setupSFP3In(aNumInputs, aInitData) setupSFPIn(3, aNumInputs, aInitData)
#define setupSFP4In(aNumInputs, aInitData) setupSFPIn(4, aNumInputs, aInitData)
#define setupSFP5In(aNumInputs, aInitData) setupSFPIn(5, aNumInputs, aInitData)
#define setupSFP6In(aNumInputs, aInitData) setupSFPIn(6, aNumInputs, aInitData)
#define setupSFP7In(aNumInputs, aInitData) setupSFPIn(7, aNumInputs, aInitData)

void setupSFPOut(u32 aPort, u32 aNumOutputs);

#define setupSFP0Out(aNumOutputs) setupSFPOut(0, aNumOutputs)
#define setupSFP1Out(aNumOutputs) setupSFPOut(1, aNumOutputs)
#define setupSFP2Out(aNumOutputs) setupSFPOut(2, aNumOutputs)
#define setupSFP3Out(aNumOutputs) setupSFPOut(3, aNumOutputs)
#define setupSFP4Out(aNumOutputs) setupSFPOut(4, aNumOutputs)
#define setupSFP5Out(aNumOutputs) setupSFPOut(5, aNumOutputs)
#define setupSFP6Out(aNumOutputs) setupSFPOut(6, aNumOutputs)
#define setupSFP7Out(aNumOutputs) setupSFPOut(7, aNumOutputs)

void setupSFPSyncMaster(u32 aUseSFPA, u32 aUseSFPB, u32 aUseSFPC, u32 aUseSFPD,
                        u32 aUseSFPE, u32 aUseSFPF, u32 aUseSFPG, u32 aUseSFPH);
void setupSFPSyncSlave(u32 aMasterChannel, u32 aSyncTime, u32 aSyncStartup);
PlxStatus plxWaitForSFPSync();

#ifdef __cplusplus
}
#endif

#endif
