local Module = {}

local Utils = require('blocks.BlockUtils')

function Module.getBlock(globals, aFpgaInput)
  local BaseGateSignalCombination = require('blocks.block').getBlock(globals)
  local self = BaseGateSignalCombination

  self.InitCode = StringList:new()
  self.OutputCode = StringList:new()
  self.OutputSignal = {}
  self.OutputMetaData = {}
  self.assertionCode = StringList:new()
  self.signalMap = {}
  
  self.isConst0 = {}
  self.canUseFpga = true
  self.channel = {}
  self.polarity = {}
  self.inputsOnSameBoard = true
  self.errorMsg = ""
  self.needsFpgaPreproc = Utils.stringToNumber(Block.InputSignal[aFpgaInput][1], true) > 0

  for idx = 1, #Block.InputSignal-1 do
    local signalValue = Utils.stringToNumber(Block.InputSignal[idx][1], true)
    self.isConst0[idx] = signalValue == 0
    if not self.isConst0[idx] then
      if Block.InputMetaData[idx] == nil 
      or Block.InputMetaData[idx][1] == nil 
      or Block.InputMetaData[idx][1]['port'] == nil 
      or Block.InputMetaData[idx][1]['polarity'] == nil
      or Block.InputType[idx][1] ~= "float" -- ensure consistent data type for muxing / data type propagation
      then
        self.canUseFpga = false          
      else
        self.channel[idx] = tonumber(Block.InputMetaData[idx][1]['port'])
        self.polarity[idx] = tonumber(Block.InputMetaData[idx][1]['polarity'])
      end
    end
  end

  if self.canUseFpga and Target.Name == "PLECS RT Box 3" then
    local startIdx = Utils.firstCaptureIndex(self.isConst0)
    if startIdx then
      for idx = startIdx+1, #Block.InputSignal-1 do
        if (self.channel[startIdx] >= 32 and (not self.isConst0[idx]) and self.channel[idx] < 32) or
          (self.channel[startIdx] < 32 and (not self.isConst0[idx]) and self.channel[idx] >= 32) then
          self.inputsOnSameBoard = false
          break;
        end
      end
    end
  end
  
  
  function BaseGateSignalCombination:forwardInputs(aForwardTable)
    for portIdx = 1, #aForwardTable do
      self.OutputSignal[portIdx] = {}
      self.OutputMetaData[portIdx] = {}
      for sigIdx = 1, #aForwardTable[portIdx] do
        local inputIdx = aForwardTable[portIdx][sigIdx]
        self.OutputSignal[portIdx][sigIdx] = Block.InputSignal[inputIdx][1]
        self.signalMap[inputIdx] = { portIdx, sigIdx }
        if self.canUseFpga and Block.InputMetaData[inputIdx] and Block.InputMetaData[inputIdx][1]  then
          self.OutputMetaData[portIdx][sigIdx] = Block.InputMetaData[inputIdx][1]
        else
          self.OutputMetaData[portIdx][sigIdx] = {}
        end
      end
    end
  end


  function BaseGateSignalCombination:generateAndAssertion(aCh1, aCh2)
    local assertions = Block.Mask.assertions - 1
    local preprocReg = Target.Coder.getPwmPreprocReg()
    
    if assertions == 1 and (not self.isConst0[aCh1]) and (not self.isConst0[aCh2]) then
      local useFpgaPreproc = false
      local preproc
      if self.canUseFpga and self.inputsOnSameBoard then
        useFpgaPreproc = true
        preproc = preprocReg:getPreprocForAnd2Assertion(
          self.channel[aCh1], self.polarity[aCh1], 
          self.channel[aCh2], self.polarity[aCh2]
        )
        if preproc == preprocReg.Result.NO_RESOURCE then
          useFpgaPreproc = false
        elseif preproc == preprocReg.Result.REDUNDANT then
          return
        end
      end
      if useFpgaPreproc then
        self.assertionCode:append("plxCheckAssertion(%i)" % { preproc })
      else
        -- fall back to preprocessing on CPU
        self:generateDirectInput(aCh1)
        self:generateDirectInput(aCh2)
        self.assertionCode:append("((%s) + (%s) < 1.000001)" % { 
          Block.InputSignal[aCh1][1], Block.InputSignal[aCh2][1] 
        })
      end
    end
  end


  function BaseGateSignalCombination:generateTripleAndAssertion(aCh1, aCh2, aCh3, aVar)
    local preprocReg = Target.Coder.getPwmPreprocReg()
    local assertions = Block.Mask.assertions - 1
    if assertions == 1 and (not self.isConst0[aCh1]) 
      and (not self.isConst0[aCh2]) and (not self.isConst0[aCh3]) then
      local useFpgaPreproc = false
      local preproc
      if self.canUseFpga and self.inputsOnSameBoard then
        useFpgaPreproc = true
        preproc = preprocReg:getPreprocForAnd3Assertion(
          self.channel[aCh1], self.polarity[aCh1],
          self.channel[aCh2], self.polarity[aCh2],
          self.channel[aCh3], self.polarity[aCh3])
        if preproc == preprocReg.Result.NO_RESOURCE then
          if self.needsFpgaPreproc then
            self.errorMsg = "%sPWM channel %d is connected to too many gate inputs.\n" % { self.errorMsg, self.channel[aCh1] }
          else
            useFpgaPreproc = false
          end
        end
      end
      if useFpgaPreproc then
        self.assertionCode:append("plxCheckAssertion(%i)" % { preproc })
      else
        -- fall back to preprocessing on CPU
        self:generateDirectInput(aCh3)
        self.assertionCode:append("(%s + (%s) < 1.000001)" % { 
          aVar, Block.InputSignal[aCh3][1] 
        })
      end
    end
  end


  function BaseGateSignalCombination:generateOrInput(aCh1, aCh2)
    local preprocReg = Target.Coder.getPwmPreprocReg()
    if self.isConst0[aCh1] then
      -- input[aCh1][1] is 0, use values from from input[aCh2][1] on both outputs
      local sigIdx = self.signalMap[aCh1]
      self.OutputMetaData[sigIdx[1]][sigIdx[2]] = Block.InputMetaData[aCh2][1]
      self.OutputSignal[sigIdx[1]][sigIdx[2]] = Block.InputSignal[aCh2][1]
    elseif not self.isConst0[aCh2] then
      local useFpgaPreproc = false
      if self.canUseFpga and self.inputsOnSameBoard then
        useFpgaPreproc = true
        local preproc = preprocReg:getPreprocForOr2Capture(
          self.channel[aCh1], self.polarity[aCh1], self.channel[aCh2], self.polarity[aCh2]
        )
        if preproc == preprocReg.Result.NO_RESOURCE then
          if self.needsFpgaPreproc then
            self.errorMsg = "%sPWM channel %d is connected to too many gate inputs.\n" % { self.errorMsg, self.channel[aCh1] }
          else
            useFpgaPreproc = false
          end
        end
      end
      if useFpgaPreproc then
        self.InitCode:append("plxConfigureOrInput(%d, %d, %d, %d);" % {
          self.channel[aCh1], self.polarity[aCh1], self.channel[aCh2], self.polarity[aCh2]
        })
      else
        -- fall back to preprocessing on CPU
        self:generateDirectInput(aCh1)
        self:generateDirectInput(aCh2)
        local maxVar = Block:AddWorkVariable("float")
        self.OutputCode:append("%(var)s = (%(sig1)s) > (%(sig2)s) ? (%(sig1)s) : (%(sig2)s);" %
        {
          var = maxVar,
          sig1 = Block.InputSignal[aCh1][1],
          sig2 = Block.InputSignal[aCh2][1],
        })
        local sigIdx = self.signalMap[aCh1]
        self.OutputSignal[sigIdx[1]][sigIdx[2]] = maxVar
        self.OutputMetaData[sigIdx[1]][sigIdx[2]] = {}
      end
    end
  end

  function BaseGateSignalCombination:generateDirectInput(aCh1)
    if not self.isConst0[aCh1] and self.channel[aCh1] then
      local preprocReg = Target.Coder.getPwmPreprocReg()
      local preproc = preprocReg:getPreprocForDirectCapture(self.channel[aCh1])
      if preproc == preprocReg.Result.NO_RESOURCE then
        self.errorMsg = "%sPWM channel %d is connected to too many gate inputs.\n" % { self.errorMsg, self.channel[aCh1] }
      end
    end
  end
  
  
  function BaseGateSignalCombination:generateAndInput(aCh1, aCh2)
    local preprocReg = Target.Coder.getPwmPreprocReg()
    local sigIdx = self.signalMap[aCh1]
    if self.isConst0[aCh1] or self.isConst0[aCh2] then
      self.OutputSignal[sigIdx[1]][sigIdx[2]] = "0."
      self.OutputMetaData[sigIdx[1]][sigIdx[2]] = {}
    else
      local useFpgaPreproc = false
      if self.canUseFpga and self.inputsOnSameBoard then
        useFpgaPreproc = true
        local preproc = preprocReg:getPreprocForAnd2Capture(
          self.channel[aCh1], self.polarity[aCh1], self.channel[aCh2], self.polarity[aCh2]
        )
        if preproc == preprocReg.Result.NO_RESOURCE then
          if self.needsFpgaPreproc then
            self.errorMsg = "%sPWM channel %d is connected to too many gate inputs.\n" % { self.errorMsg, self.channel[aCh1] }
          else
            useFpgaPreproc = false
          end
        end
      end
      if useFpgaPreproc then
        self.InitCode:append("plxConfigureAndInput(%d, %d, %d, %d);" % {
          self.channel[aCh1], self.polarity[aCh1], self.channel[aCh2], self.polarity[aCh2]
        })
      else
        -- fall back to preprocessing on CPU
        self:generateDirectInput(aCh1)
        self:generateDirectInput(aCh2)
        local minVar = Block:AddWorkVariable("float")
        self.OutputCode:append("%(var)s = (%(sig1)s) < (%(sig2)s) ? (%(sig1)s) : (%(sig2)s);" %
        {
          var = minVar,
          sig1 = Block.InputSignal[aCh1][1],
          sig2 = Block.InputSignal[aCh2][1],
        })
        self.OutputSignal[sigIdx[1]][sigIdx[2]] = minVar
        self.OutputMetaData[sigIdx[1]][sigIdx[2]] = {}
        return minVar
      end
    end
    return nil
  end


  function BaseGateSignalCombination:getAssertionCode()
    if #self.assertionCode > 0 then
      return table.concat(self.assertionCode, "&&")
    else
      return "1"
    end
  end


  return BaseGateSignalCombination

end
return Module