local Module = {}

local static = {}

local Utils = require('blocks.BlockUtils')

function Module.getBlock(globals)
  local Nanostep = require('blocks.block').getBlock(globals)

  function Nanostep:getDirectFeedthroughCode()

    local InitCode = StringList:new()
    local OutputCode = StringList:new()
    local OutputSignal = {}
    local OutputMetaData = {}
    
    local sampleTime = Block.SampleTime[1]
    
    local onFpga = Utils.stringToNumber(Block.InputSignal[7][1], true)
    local solverInstance = Target.Coder.requireNanostepSolver(3, onFpga)
    if solverInstance == 0 then
      return "No more nanostep solvers available."
    end
    self.solverInstance = solverInstance
    print("%s: Using Nanostep solver %d, 3/3" % { Utils.splitPathTail(Block.Path, 3), solverInstance })
    
    local precision = 26
    if Target.Name == "PLECS RT Box 1" then
      precision = 24
    end
    local is_partial = false
    local bit_shift = 4
    local coefficients, coef_err = Utils.scale_nanostep_coefficients(Block, precision, bit_shift, is_partial)
    if coef_err then
      return coef_err
    end

    InitCode:append("{")
    InitCode:append("nanostep_parameters_t parameters = {\n")
    for k, v in pairs(coefficients) do
      InitCode:append(".%s = %d,\n" % { k, v })
    end
    InitCode:append(".sampletime = %0.17f,\n" % { sampleTime })
    InitCode:append(".singlephase = %d,\n" % { Block.Mask.singlephase })
    InitCode:append(".twophase = %d,\n" % { Block.Mask.twophase })
    InitCode:append(".threephase = %d,\n" % { Block.Mask.threephase })
    InitCode:append(".psfb = %d,\n" % { Block.Mask.psfb })
    InitCode:append(".usePhysicalModel = %d,\n" % { onFpga })
    InitCode:append(".send_iL_vC = %d,\n" % { 1 })
    local clamp_a = Block.Mask['clamp_a']
    local clamp_b = Block.Mask['clamp_b']
    InitCode:append(".clamp_a = {%d, %d, %d},\n" % { clamp_a[1], clamp_a[2], clamp_a[3] })
    InitCode:append(".clamp_b = {%d, %d, %d},\n" % { clamp_b[1], clamp_b[2], clamp_b[3] })
    InitCode:append(".commonrail_a = %d,\n" % {Block.Mask.commonrail_a})
    InitCode:append(".commonrail_b = %d,\n" % {Block.Mask.commonrail_b})
    InitCode:append(".inject_v_ac = %d,\n" % {Block.Mask.inject_v_ac})
    local sap = Block.Mask['sa_pos_en']
    local san = Block.Mask['sa_neg_en']
    local sbp = Block.Mask['sb_pos_en']
    local sbn = Block.Mask['sb_neg_en']
    InitCode:append(".sw_a_pos_en = {%d, %d, %d, %d, %d, %d},\n" % {
      sap[1], sap[2], sap[3], sap[4], sap[5], sap[6]
    })
    InitCode:append(".sw_a_neg_en = {%d, %d, %d, %d, %d, %d},\n" % {
      san[1], san[2], san[3], san[4], san[5], san[6]
    })
    InitCode:append(".sw_b_pos_en = {%d, %d, %d, %d, %d, %d},\n" % {
      sbp[1], sbp[2], sbp[3], sbp[4], sbp[5], sbp[6]
    })
    InitCode:append(".sw_b_neg_en = {%d, %d, %d, %d, %d, %d},\n" % {
      sbn[1], sbn[2], sbn[3], sbn[4], sbn[5], sbn[6]
    })
    -- All Nanostep accumulators are always enabled, except the ones for the
    -- capacitors which depend on the capacitance values.
    local enable_computations = 0xffc7
    local c1_used = Block.Mask['nanostepParameters']['Cx_vec_1'][1] ~= 0
    local c2_used = Block.Mask['nanostepParameters']['Cx_vec_2'][1] ~= 0
    local c3_used = Block.Mask['nanostepParameters']['Cx_vec_3'][1] ~= 0
    if c1_used or c3_used then
      enable_computations = enable_computations | (1 << 3)
    end
    if c2_used or c3_used then
      enable_computations = enable_computations | (1 << 4)
    end
    InitCode:append('.enable_computations = 0x%x\n' % {enable_computations})

    InitCode:append("};\n")

    local pwmPortIdx = { 3, 3, 3, 3, 3, 3, 4, 4, 4, 4, 4, 4, 5, 5, 5, 5, 5, 5, 6, 6, 6, 6, 6, 6 }
    local pwmSignalIdx = { 1, 2, 3, 4, 5, 6, 1, 2, 3, 4, 5, 6, 1, 2, 3, 4, 5, 6, 1, 2, 3, 4, 5, 6 }

    for i = 1, 24 do
      local ok, result, constVal = pcall(Utils.extractPwmIndex, pwmPortIdx[i], pwmSignalIdx[i])
      if not ok then return nil, Utils.processError(result) end
      if constVal < 0 then
        InitCode:append("parameters.switch_mask[%i] = %i;\n" % { i-1, 0 })
        InitCode:append("parameters.switch_indices[%i] = %i;\n" % { i-1, result })
      else
        InitCode:append("parameters.switch_mask[%i] = %i;\n" % { i-1, 1 })
        InitCode:append("parameters.switch_mask_val[%i] = %i;\n" % { i-1, constVal })
      end
    end

    InitCode:append("setupFullNanostep(%d, &parameters);\n" % { 
      solverInstance-1
    })

    -- The setComponentUsingNanostep function expects to be given a pointer to
    -- a string which will remain valid after the init function completes.
    InitCode:append("static char* componentPath = \"%s\";\n" % {Utils.splitPathTail(Block.Path, 3)})
    InitCode:append("setComponentUsingNanostep(%d, %d, componentPath);\n" % {
      solverInstance-1, -1,
    })
    -- This function call calculates the limit in case the solver communicates
    -- with the CPU. If it communicates with a Flexarray solver, the limit will
    -- be re-computed later.
    InitCode:append("calculateNanostepAccumulatorAmplitudeLimit(%f);\n" % {
      sampleTime
    })

    InitCode:append("};")

    OutputSignal[1] = { "getNanostepPrimaryCurrent(%i,0)" % { solverInstance-1 }, "getNanostepPrimaryCurrent(%i,1)" % { solverInstance-1 }, "getNanostepPrimaryCurrent(%i,2)" % { solverInstance-1 } }
    OutputSignal[2] = { "getNanostepSecondaryCurrent(%i,0)" % { solverInstance-1 }, "getNanostepSecondaryCurrent(%i,1)" % { solverInstance-1 }, "getNanostepSecondaryCurrent(%i,2)" % { solverInstance-1 } }
    OutputSignal[3] = { "getNanostepInductorCurrent(%i,0)" % { solverInstance-1 }, "getNanostepInductorCurrent(%i,1)" % { solverInstance-1 }, "getNanostepInductorCurrent(%i,2)" % { solverInstance-1 } }
    OutputSignal[4] = { "getNanostepCapacitorVoltage(%i,0)" % { solverInstance-1 }, "getNanostepCapacitorVoltage(%i,1)" % { solverInstance-1 }, "getNanostepCapacitorVoltage(%i,2)" % { solverInstance-1 } }
    OutputMetaData[1] = {}
    OutputMetaData[1][1] = { sourceInfo = { fromNanostep = 1, meterIndex = 0, nanostepIndex = (solverInstance-1)*3 } }
    OutputMetaData[1][2] = { sourceInfo = { fromNanostep = 1, meterIndex = 0, nanostepIndex = (solverInstance-1)*3+1 } }
    OutputMetaData[1][3] = { sourceInfo = { fromNanostep = 1, meterIndex = 0, nanostepIndex = (solverInstance-1)*3+2 } }
    OutputMetaData[2] = {}
    OutputMetaData[2][1] = { sourceInfo = { fromNanostep = 1, meterIndex = 1, nanostepIndex = (solverInstance-1)*3 } }
    OutputMetaData[2][2] = { sourceInfo = { fromNanostep = 1, meterIndex = 1, nanostepIndex = (solverInstance-1)*3+1 } }
    OutputMetaData[2][3] = { sourceInfo = { fromNanostep = 1, meterIndex = 1, nanostepIndex = (solverInstance-1)*3+2 } }
    OutputMetaData[3] = {{}, {}, {}}
    OutputMetaData[4] = {{}, {}, {}}

    return {
      Include = "plexim/Nanostep.h",
      InitCode = InitCode,
      OutputCode = OutputCode,
      OutputSignal = OutputSignal,
      OutputMetaData = OutputMetaData,
      UserData = {bid = Nanostep:getId()}
    }
  end


  function Nanostep:getNonDirectFeedthroughCode()

    local InitCode = StringList:new()
    local UpdateCode = StringList:new()
    local onFpga = Utils.stringToNumber(Block.InputSignal[7][1], true)

    -- A Full Nanostep Solver may be used in a circuit where some of the voltage
    -- inputs are connected and others aren't. These unconnected inputs should
    -- be set to zero.
    --
    -- If we're using the CPU, we get this zero value esentially for free.
    --
    -- However, with the Physical Model, we would have to have a meter that
    -- just ouputs zero. To avoid this waste, we arbitrarily set the meter index
    -- to zero and instruct the Nanostep Solver to ignore that voltage.
    --
    -- A similar situation can occur if two voltage meters are measuring the
    -- same voltage but one of them is inverted. One of them will get optimized
    -- away and we may have to connect the Nanostep solver to the other one
    -- and invert the voltage value.
    for i = 1, 3 do
      if onFpga ~= 0 then
        local source_1, source_2 = Utils.nanostep_get_flexarray_volt_sources(Block, i)
        if source_1.idx or source_2.idx then
          InitCode:append("FPGAPhysicalModel_setupOutputToNanostep(%i, %i, %i);\n" % {
            (self.solverInstance-1)*3+(i-1),
            source_1.idx or 0,
            source_2.idx or 0
          })
        end
        if not source_1.idx then
          InitCode:append("nanostep_ignore_voltage(%i, %i);\n" % {
            self.solverInstance-1,
            i-1
          })
        elseif source_1.gain < 0 then
          InitCode:append("nanostep_invert_voltage(%i, %i);\n" % {
            self.solverInstance-1,
            (i-1) * 2
          })
        end
        if not source_2.idx then
          InitCode:append("nanostep_ignore_voltage(%i, %i);\n" % {
            self.solverInstance-1,
            i-1 + 3
          })
        elseif source_2.gain < 0 then
          InitCode:append("nanostep_invert_voltage(%i, %i);\n" % {
            self.solverInstance-1,
            (i-1) * 2 + 1
          })
        end
      else
        if Utils.isUnconnected(Block.InputSignal[1][i]) then
          InitCode:append("setNanostepPrimaryVoltageOut(%i, %i, 0);\n" % { self.solverInstance-1, i-1 })
        else
          UpdateCode:append("setNanostepPrimaryVoltageOut(%i, %i, %s);\n" % { self.solverInstance-1, i-1, Block.InputSignal[1][i] })
        end
        if Utils.isUnconnected(Block.InputSignal[2][i]) then
          InitCode:append("setNanostepSecondaryVoltageOut(%i, %i, 0);\n" % { self.solverInstance-1, i-1 })
        else
          UpdateCode:append("setNanostepSecondaryVoltageOut(%i, %i, %s);\n" % { self.solverInstance-1, i-1, Block.InputSignal[2][i] })
        end
      end
    end

    local retInclude = {}
    if inputsFromFpga then
      retInclude = { "plexim/physicalModel.h" }
    end

    return {
      Include = retInclude,
      InitCode = InitCode,
      UpdateCode = UpdateCode,
    }
  end
    
  return Nanostep  
end
return Module
