local Module = {}

local static = {}

local Utils = require('blocks.BlockUtils')

function Module.getBlock(globals)
  local Nanostep = require('blocks.block').getBlock(globals)

  Nanostep.partial = 0
  
  function Nanostep:getDirectFeedthroughCode()

    local InitCode = StringList:new()
    local OutputCode = StringList:new()
    local OutputSignal = {}
    local OutputMetaData = {}

    --[[
      The code block has 4 2-wide input ports for the switching signals. These
      indices control how they are mapped to the 8 switches needed by one
      partial Nanostep solver. For example, switch 7 will be the second signal
      of port 6.
    --]]
    local pwmPortIdx = { 3, 3, 4, 4, 5, 5, 6, 6 }
    local pwmSignalIdx = { 1, 2, 1, 2, 1, 2, 1, 2 }

    local sampleTime = Block.SampleTime[1]
    
    --[[
      RT Box 2 and 3 have two Nanostep solvers. One solver can be used to
      simulate different topologies:
        - one PSFB
        - one three phase DAB
        - three single phase DABs
      
      So if we use one solver for one single phase DAB, we still have two left.
      However, we cannot simulate a PSFB or a three phase DAB any more. The
      requireNanostepSolver function takes care of this resource tracking.
    --]]
    local usePhysicalModel = Utils.stringToNumber(Block.InputSignal[7][1], true)
    local requiredPartials = 1
    local solverInstance, partial = Target.Coder.requireNanostepSolver(requiredPartials, usePhysicalModel)
    if solverInstance == 0 then
      return "No more nanostep solvers available."
    end
    self.solverInstance = solverInstance
    self.partial = partial
    print("%s: Using Nanostep solver %d, %d/3" % { Utils.splitPathTail(Block.Path, 3), solverInstance, partial })
    
    local precision = 26
    if Target.Name == "PLECS RT Box 1" then
      precision = 24
    end
    local is_partial = true
    local bit_shift = 4
    local coefficients, coef_err = Utils.scale_nanostep_coefficients(Block, precision, bit_shift, is_partial)
    if coef_err then
      return coef_err
    end

    --[[
      The nanostep_parameters_t struct contains all fields needed to configure
      one Nanostep solver. However, since we're only configuring "one third"
      here, we don't set all of the values. The setupPartialNanostep
      function will write them to the correct registers based on the "partial"
      value.

      Note that some parameters apply to the whole solver an not only one
      partial. These will get written each time we initialize (up to 3 times per
      solver), but since we assume them to be the same for all partials it's not
      an issue.
    --]]
    InitCode:append("{")
    InitCode:append("nanostep_parameters_t parameters = {\n")
    InitCode:append(".A1_11 = %d,\n" % { coefficients.A1_11 })
    InitCode:append(".A1_14 = %d,\n" % { coefficients.A1_14 })
    InitCode:append(".B1_11 = %d,\n" % { coefficients.B1_11 })
    InitCode:append(".Cx_vec_1 = %d,\n" % { coefficients.Cx_vec_1 })
    InitCode:append(".sampletime = %0.17f,\n" % { sampleTime })
    InitCode:append(".singlephase = 1,\n")
    InitCode:append(".twophase = 0,\n")
    InitCode:append(".threephase = 0,\n")
    InitCode:append(".psfb = 0,\n")
    InitCode:append(".usePhysicalModel = %d,\n" % { usePhysicalModel })
    InitCode:append(".send_iL_vC = %d,\n" % { 1 })
    InitCode:append(".clamp_a = {%d, 0, 0},\n" % { Block.Mask['clamp_a_single'][1] })
    InitCode:append(".clamp_b = {%d, 0, 0},\n" % { Block.Mask['clamp_b_single'][1] })
    local sap = Block.Mask['sa_pos_en']
    local san = Block.Mask['sa_neg_en']
    local sbp = Block.Mask['sb_pos_en']
    local sbn = Block.Mask['sb_neg_en']
    InitCode:append(".sw_a_pos_en = {%d, %d, 0, 0, 0, 0},\n" % {
      sap[1], sap[2]
    })
    InitCode:append(".sw_a_neg_en = {%d, %d, 0, 0, 0, 0},\n" % {
      san[1], san[2]
    })
    InitCode:append(".sw_b_pos_en = {%d, %d, 0, 0, 0, 0},\n" % {
      sbp[1], sbp[2]
    })
    InitCode:append(".sw_b_neg_en = {%d, %d, 0, 0, 0, 0},\n" % {
      sbn[1], sbn[2]
    })
    -- All Nanostep accumulators are always enabled, except the one for the
    -- capacitors which depends on the capacitance value.
    local enable_computations = 0xffc7
    local capacitor_used = Block.Mask['nanostepParameters']['Cx_vec_1'][1] ~= 0
    if capacitor_used then
      enable_computations = enable_computations | (1 << 3)
    end
    InitCode:append('.enable_computations = 0x%x\n' % {enable_computations})
    InitCode:append("};\n")

    --[[
      The Nanostep solver has 24 switching inputs. The arrays
          - parameters.switch_indices,
          - parameters.switch_mask,
          - parameters.switch_mask_val,
      specify whether they should be connected to a Digital Input of the RT Box
      or a constant value.

      When using partials, the 24 inputs are split into three disjoint sets.
          - partial 0: [19:18] [13:12]  [7:6] [1:0],
          - partial 1: [21:20] [15:14]  [9:8] [3:2],
          - partial 2: [23:22] [17:16] [11:10] [5:4].

      Inputs from the set of partial 0 have no effect on the other partials and
      vice versa.
    --]]
    local sw_set_indices = { 0, 1, 6, 7, 12, 13, 18, 19 }
    for i = 1, 8 do
      sw_set_indices[i] = sw_set_indices[i] + 2 * (partial-1)
    end

    for i = 1, 8 do
      --[[
        extractPwmIndex uses the signal meta data to determine which digital
        input or constant value to use. Therefore, the source of the signal in
        PLECS must be either a PWM capture block (with pin number) or a
        constant.
      --]]
      local ok, result, constVal = pcall(Utils.extractPwmIndex, pwmPortIdx[i], pwmSignalIdx[i])
      if not ok then return nil, Utils.processError(result) end
      if constVal < 0 then
        InitCode:append("parameters.switch_indices[%i] = %i;\n" % { sw_set_indices[i], result })
        InitCode:append("parameters.switch_mask[%i] = %i;\n" % { sw_set_indices[i], 0 })
      else
        InitCode:append("parameters.switch_mask[%i] = %i;\n" % { sw_set_indices[i], 1 })
        InitCode:append("parameters.switch_mask_val[%i] = %i;\n" % { sw_set_indices[i], constVal })
      end
    end
    
    InitCode:append("setupPartialNanostep(%d, %d, &parameters);\n" % {
      solverInstance-1, partial-1
    })

    -- The setComponentUsingNanostep function expects to be given a pointer to
    -- a string which will remain valid after the init function completes.
    InitCode:append("static char* componentPath = \"%s\";\n" % {Utils.splitPathTail(Block.Path, 3)})
    InitCode:append("setComponentUsingNanostep(%d, %d, componentPath);\n" % {
      solverInstance-1, partial-1,
    })
    -- This function call calculates the limit in case the solver communicates
    -- with the CPU. If it communicates with a Flexarray solver, the limit will
    -- be re-computed later.
    InitCode:append("calculateNanostepAccumulatorAmplitudeLimit(%f);\n" % {
      sampleTime
    })

    InitCode:append("};")

    OutputSignal[1] = { "getNanostepPrimaryCurrent(%i,%i)" % { solverInstance-1, partial-1 } }
    OutputSignal[2] = { "getNanostepSecondaryCurrent(%i,%i)" % { solverInstance-1, partial-1 } }
    OutputSignal[3] = { "getNanostepInductorCurrent(%i,%i)" % { solverInstance-1, partial-1 } }
    OutputSignal[4] = { "getNanostepCapacitorVoltage(%i,%i)" % { solverInstance-1, partial-1 } }
    OutputMetaData[1] = {}
    OutputMetaData[1][1] = { sourceInfo = { fromNanostep = 1, meterIndex = 0, nanostepIndex = (solverInstance-1)*3+(partial-1) } }
    OutputMetaData[2] = {}
    OutputMetaData[2][1] = { sourceInfo = { fromNanostep = 1, meterIndex = 1, nanostepIndex = (solverInstance-1)*3+(partial-1) } }
    OutputMetaData[3] = {{}}
    OutputMetaData[4] = {{}}

    return {
      Include = "plexim/Nanostep.h",
      InitCode = InitCode,
      OutputCode = OutputCode,
      OutputSignal = OutputSignal,
      OutputMetaData = OutputMetaData,
      UserData = {bid = Nanostep:getId()}
    }
  end


  function Nanostep:getNonDirectFeedthroughCode()

    local InitCode = StringList:new()
    local UpdateCode = StringList:new()

    -- Since we are configuring a Third of a Nanostep Solver, the expectation is
    -- that both indices are either not nil or both are nil. Other options are
    -- only possible when configuring a Full Nanostep Solver.
    --
    -- Furthermore, we always want to look up the indices for the "first"
    -- partial. These are the only ones connected in the PLECS model.
    local source_1, source_2 = Utils.nanostep_get_flexarray_volt_sources(Block, 1)
    if source_1.idx and source_2.idx then
      InitCode:append("FPGAPhysicalModel_setupOutputToNanostep(%i, %i, %i);\n" % {
        (self.solverInstance-1)*3+(self.partial-1),
        source_1.idx,
        source_2.idx
      })
      if source_1.gain < 0 then
        InitCode:append("nanostep_invert_voltage(%i, %i);\n" % {
          self.solverInstance-1,
          (self.partial-1) * 2
        })
      end
      if source_2.gain < 0 then
        InitCode:append("nanostep_invert_voltage(%i, %i);\n" % {
          self.solverInstance-1,
          (self.partial-1) * 2 + 1
        })
      end
    else
      UpdateCode:append("setNanostepPrimaryVoltageOut(%i, %i, %s);\n" % { self.solverInstance-1, self.partial-1, Block.InputSignal[1][1] })
      UpdateCode:append("setNanostepSecondaryVoltageOut(%i, %i, %s);\n" % { self.solverInstance-1, self.partial-1, Block.InputSignal[2][1] })
    end

    local retInclude = {}
    if inputsFromFpga then
      retInclude = { "plexim/physicalModel.h" }
    end

    return {
      Include = retInclude,
      InitCode = InitCode,
      UpdateCode = UpdateCode,
    }
  end
    
  return Nanostep  
end
return Module
