local Module = {}

local Utils = require('blocks.BlockUtils')

local static = {
  instances = {}
}


function Module.getBlock(globals)
  local PWMGen = require('blocks.block').getBlock(globals)

  function PWMGen:getDirectFeedthroughCode(aSimplePWM, aFreq_pu, aPhase, aDelay)
    local InitCode = StringList:new()
    local TriggerOutputSignal = StringList:new()
    local OutputCode = StringList:new()
    local Require = ResourceList:new()

    if not string.find(Target.Family, "PLECS RT Box") then
      return "This block can only be used with a PLECS RT Box target."
    end

    local variable_delay = Block.Mask.variable_delay or 0
    local delay_sec = Block.Mask.delay_sec
    local min_delay_sec = Block.Mask.min_delay_sec
    local type = Block.Mask.type
    local update = Block.Mask.update
    local offset = Block.Mask.offset
    local sync = (Block.Mask.sync ~= 0)
    local limit_delta = Block.Mask.limit_delta
    local carrierFrequency = Block.Mask.fc
    local sampleTime = Target.Variables.SAMPLE_TIME
    local fpgaClk = Target.Coder.getFPGAClk()
    local prescale = 0
    local gain = 0
    local limit_delta = Block.Mask.limit_delta
    local pspUnit = Block.Mask.pspUnit
    local active_polarity = Block.Mask.active_polarity

    variable_delay = Utils.stringToNumber(variable_delay) or 0

    local core = Block.Task.Core;
    local nonBaseSampleRate = Block.Task.SampleTime[1] > Target.Variables.SAMPLE_TIME
    local needsBuffer = (nonBaseSampleRate == true and core > 0)
    if needsBuffer == true then
      Target.Coder.requirePwmBuffer("plxPwmCmpBuffer")
    end

    local num_period_ticks = math.floor(fpgaClk / carrierFrequency + 0.5)
    local num_bin_digits = math.floor(math.log(num_period_ticks)/math.log(2)) + 1
    if num_bin_digits > 16 then
      prescale = num_bin_digits - 16;
    else
      prescale = 0;
    end

    OutputCode:append("{\n")
    local period = math.floor(fpgaClk/ (carrierFrequency * 2^prescale) + 0.5)
    local periodPWM_int
    local freq_pu = Utils.stringToNumber(aFreq_pu)
    if freq_pu then
      local periodPWM
      if sync then
        if freq_pu < 1 then
          freq_pu = 1
        end
        periodPWM = period/math.floor(freq_pu + 0.5)
      else
        if freq_pu < 0 then
          freq_pu = 0
          periodPWM = 0
        else
          periodPWM = period/freq_pu
        end
      end
      periodPWM_int = math.floor(periodPWM/(type+1) + 0.5)
      if periodPWM_int > 65535 then
        periodPWM_int = 65535
      elseif periodPWM_int < 20 then
        periodPWM_int = 20
      end
      for i = 1, #Block.InputSignal[1] do
        local channel = Block.Mask.channel[i]
        InitCode:append("setPWMOutPrd(%i, %i);\n" % { channel, periodPWM_int-1 })
      end
      OutputCode:append("const int periodPWM_int = %i;\n" % { periodPWM_int })
      OutputCode:append("const float gain = %.9ef;\n" % { (periodPWM_int-0.5)/limit_delta })
    else    
      OutputCode:append("float freq_pu = %s;\n" % { aFreq_pu })

      if (sync) then
        OutputCode:append("if (freq_pu < 1) freq_pu = 1;\n")
        OutputCode:append("float periodPWM = (float)%i/(int)(freq_pu+0.5);\n" % { period })
      else
        OutputCode:append("float periodPWM;\n")
        OutputCode:append("if (freq_pu < 0)\n")
        OutputCode:append("{freq_pu = 0;\n")
        OutputCode:append("periodPWM = 0;}\n")
        OutputCode:append("else\n")
        OutputCode:append("periodPWM = %i/freq_pu;\n" % { period })
      end
      OutputCode:append("int periodPWM_int = (int)(periodPWM/%i + 0.5);\n" % { type+1} )
      OutputCode:append("if (periodPWM_int > 65535) \n")
      OutputCode:append("{periodPWM_int = 65535;\n")
      OutputCode:append("periodPWM = 65535;}\n")
      OutputCode:append("else if (periodPWM_int < 20) \n")
      OutputCode:append("{periodPWM_int = 20;\n")
      OutputCode:append("periodPWM = 20;}\n")
      for i = 1, #Block.InputSignal[1] do
        local channel = Block.Mask.channel[i]
        if needsBuffer == true then
          Target.Coder.requirePwmBuffer("plxPwmPrdBuffer")
          OutputCode:append("plxPwmPrdBuffer[%i] = periodPWM_int;\n" % {channel });
          Target.Coder.addIoCode(core, "setPWMOutPrd(%i, plxPwmPrdBuffer[%i]-1);" % { channel, channel })
        else
          OutputCode:append("setPWMOutPrd(%i, periodPWM_int-1);\n" % { channel })
        end
      end
      OutputCode:append("float gain = (periodPWM_int-0.5)/%.9ef;\n" % { limit_delta })
    end

    local allPhasesConst = true
    for i = 1, #aPhase do
      if Utils.stringToNumber(aPhase[i]) == nil then
        allPhasesConst = false
        break
      end
    end
    
    if allPhasesConst == true and freq_pu then
      local phase0 = Utils.stringToNumber(aPhase[1])
      for i = 1, #aPhase do
        local channel = Block.Mask.channel[i]
        local shiftPWM
        local phase = Utils.stringToNumber(aPhase[i])
        if sync and i > 1 and (not aSimplePWM) then
          phase = phase + phase0
          if phase > 1 then
            phase = phase - 1
          end
        end
        shiftPWM = math.floor((type+1)*periodPWM_int*phase+0.5)
        if shiftPWM < 0 or shiftPWM >= (type+1)*periodPWM_int then
          InitCode:append("setPWMOutShft(%i, 0);\n" % { channel })
        else
          if type == 0 then -- sawtooth carrier
            if shiftPWM > 1 then
              InitCode:append("setPWMOutShft(%i, %i);\n" % { channel, periodPWM_int - shiftPWM + 1 })
            elseif shiftPWM == 1 then
              InitCode:append("setPWMOutShft(%i, 0);\n" % { channel })
            else
              InitCode:append("setPWMOutShft(%i, 1);\n" % { channel })
            end
          else -- triangular carrier
            if shiftPWM == 0 then
              InitCode:append("setPWMOutShft(%i, 0);\n" % { channel })
            elseif shiftPWM <= periodPWM_int and shiftPWM ~= 0 then
              InitCode:append("setPWMOutShft(%i, %i);\n" % { channel, 32768 + shiftPWM - 1 })
            else
              InitCode:append("setPWMOutShft(%i, %i);\n" % { channel, 2*periodPWM_int - shiftPWM })
            end
          end
        end
      end
    else
      OutputCode:append("float phase0 = %s;\n" % { aPhase[1] })
      for i = 1, #aPhase do
        local channel = Block.Mask.channel[i]
        OutputCode:append("{\n")
        if sync and i > 1 then
          OutputCode:append("float phase = %s+phase0;\n" % { aPhase[i] })
          OutputCode:append("if (phase > 1) phase -= 1;\n")
        else
          OutputCode:append("float phase = %s;\n" % { aPhase[i] })
        end
        OutputCode:append("int shiftPWM = (int)(%i*periodPWM_int*phase+0.5);\n" % { type+1 })
        OutputCode:append("if (shiftPWM < 0 || shiftPWM >= %i*periodPWM_int)\n" % { type+1 })
        OutputCode:append("{shiftPWM = 0;}\n")
        OutputCode:append("int shiftValue;\n")
        if type == 0 then -- sawtooth carrier
          OutputCode:append("if (shiftPWM == 0)\n")
          OutputCode:append("{shiftValue = 1;}\n")
          OutputCode:append("else if (shiftPWM == 1)\n")
          OutputCode:append("{shiftValue = 0;}\n")
          OutputCode:append("else\n")
          OutputCode:append("{shiftValue = periodPWM_int - shiftPWM + 1;}")
        else
          OutputCode:append("if (shiftPWM == 0)\n")
          OutputCode:append("{shiftValue = 0;}\n")
          OutputCode:append("else if (shiftPWM <= periodPWM_int)\n")
          OutputCode:append("{shiftValue = (shiftPWM - 1) | 0x8000;}\n")
          OutputCode:append("else\n")
          OutputCode:append("{shiftValue = (2*periodPWM_int - shiftPWM) & 0x7fff;}")
        end
        if needsBuffer == true then
          Target.Coder.requirePwmBuffer("plxPwmShiftBuffer")
          OutputCode:append("plxPwmShiftBuffer[%i] = shiftValue;\n" % {channel });
          Target.Coder.addIoCode(core, "setPWMOutShft(%i, plxPwmShiftBuffer[%i]);" % { channel, channel })
        else
          OutputCode:append("setPWMOutShft(%i, shiftValue);\n" % { channel })        
        end
        OutputCode:append("}\n")
      end
    end
   
    if variable_delay == 1 then
      local maxDelay = num_period_ticks
      if maxDelay > 0xFFFF then
         maxDelay = 0xFFFF
      end
      OutputCode:append("const int maxDelay = %i;\n" % { maxDelay })
      for i = 1, #aDelay do
        local channel = Block.Mask.channel[i]
        local minDelay = math.floor(min_delay_sec[i] * fpgaClk + 0.5)
        if minDelay > maxDelay then
           minDelay = maxDelay
        end
        OutputCode:append("{\n")
        OutputCode:append("const int minDelay = %i;\n" % { minDelay })
        OutputCode:append("int delay = %s * %.9ef;\n" % { aDelay[i], fpgaClk })
        OutputCode:append("if (delay < minDelay)\n")
        OutputCode:append("{delay = minDelay;}\n")
        OutputCode:append("else if (delay > maxDelay)\n")
        OutputCode:append("{delay = maxDelay;}\n")
        OutputCode:append("plxChangePWMDelay(%i, delay);\n" % { channel })
        OutputCode:append("}\n")
       end
    end

    for i = 1, #Block.InputSignal[1] do
      local channel = Block.Mask.channel[i]
      local polarity = Block.Mask.polar[i]
      local slave = (i > 1) and (not sync) and (not aSimplePWM)
      local masterID = slave and Block.Mask.channel[1] or channel
      local delay = math.floor(delay_sec[i] * fpgaClk + 0.5)
      local safeState = Block.Mask.safe_state[i]
      local trig = Block.Mask.trig

      if slave and ((channel < 32 and masterID >= 32) or (channel >= 32 and masterID < 32)) then
        return "Error: PWM master and slave channels cannot be mixed between " ..
               "lower and upper 32 channels."
      end

      if delay > 0xFFFF then
        return "Error: PWM turn-on delay cannot exceed %.6fs" % { 0xFFFF / fpgaClk }
      end

      Require:add("Digital output", channel)

      Target.Coder.registerPwmForTask(Block.Task.Name, channel)

      if active_polarity[i] == 1 then
        InitCode:append("setupDigitalOut(%i, DO_PWM, DO_NINV);\n" % { channel })
      else
        InitCode:append("setupDigitalOut(%i, DO_PWM, DO_INV);\n" % { channel })
      end
      InitCode:append("setupPWMGen(%i, %i, %i, %i, %i, %i, %i, %i, %i, %i);\n" % { channel, delay, prescale, trig, slave, masterID, sync, polarity, type, update }) 
      InitCode:append("plxSetPWMSafeState(%i, %i);\n" % { channel, safeState}) 
      InitCode:append("plxSetPWMProtectionUnit(%i, PLX_PSP_UNIT%i);\n" % { channel, pspUnit}) 
   
      OutputCode:append("{\n")
   
      OutputCode:append("int cmpPWM = (int)(((%s) + %.9ef)*gain + 0.5);\n" %  { Block.InputSignal[1][i], offset })
      OutputCode:append("if (cmpPWM < 0) \n")
      OutputCode:append("{cmpPWM = 0;}\n")
      OutputCode:append("else if (cmpPWM > periodPWM_int) \n")
      OutputCode:append("{cmpPWM = periodPWM_int;}\n")
      if needsBuffer == true then
        OutputCode:append("plxPwmCmpBuffer[%i] = cmpPWM;\n" % {channel });
        Target.Coder.addIoCode(core, "setPWMOutCmp(%i, plxPwmCmpBuffer[%i]);" % { channel, channel })
      else
        OutputCode:append("setPWMOutCmp(%i, cmpPWM);\n" % { channel })
      end
      OutputCode:append("}\n")

      TriggerOutputSignal:append("{ adctrig = %i }" % { channel })
    end
    OutputCode:append("}\n")

    -- Register that pspUnit is protecting the channel. Used to prevent a digital
    -- override of the same channel.
    for i = 1, #Block.InputSignal[1] do
      local channel = Block.Mask.channel[i]
      Target.Coder.UsePowerstageProtection(pspUnit, channel)
    end

    return {
        Include = "plexim/DigitalOut.h",
        Require = Require,
        InitCode = InitCode,
        OutputSignal = { TriggerOutputSignal },
        OutputCode = OutputCode
    }
  end
  
  return PWMGen
end  
return Module
