local Xcp = { }

local FileUtils = require("FileUtils")

local NameRegistry = { }

local function time2asam(time)
  local timeMicro = math.floor(time * 100 * 1000 * 1000+0.5)
  local timeUnit = 1 -- 10ns
  while timeMicro >= 256 do
    timeMicro = math.floor(timeMicro/10+0.5)
    timeUnit = timeUnit + 1
  end
  return timeMicro, timeUnit
end

local function tasks()
  local taskName
  if #Model.TaskConfiguration > 0 then
    taskName = Model.Tasks[1].Name
  else
    taskName = "Base task"
  end
  local timeMicro, timeUnit = time2asam(Target.Variables.SAMPLE_TIME)
  local task = "/begin EVENT \"%s\" \"%s\" 0x0 DAQ 0xFF 0x%X 0x%X 0x00 CONSISTENCY DAQ /end EVENT" % {
    taskName,
    taskName,
    timeMicro,
    timeUnit,
  }
  return task
end

local function cleanName(name)
  local newName = string.gsub(name, " ", "_")
  newName = string.gsub(newName, "\\/", "_") -- replace escaped slashes with _
  newName = string.gsub(newName, "/", ":") -- replace remaining slashes with :
  newName = string.gsub(newName, ":", ".", 1) -- replace first colon with dot
  newName = string.gsub(newName, ":", "_") -- replace all remaining colons with _
  newName = string.gsub(newName, "[^0-9A-Za-z%._]", "")
  newName = string.gsub(newName, "^([^A-Za-z_])", "x%1")
  return newName
end

local function tryMakeUnique(name)
  local newName
  local numStart, numEnd = name:reverse():find("%d+_")
  local basePart = ""
  local numPart = ""
  if numStart ~= 1 then
    newName = name .. "_1"
  else
    numStart = string.len(name) - numEnd + 1
    basePart = string.sub(name, 1, numStart - 1)
    numPart = string.sub(name, numStart+1)
    newIdx = numPart + 1
    newName = "%s_%d" % { basePart, newIdx}
  end  
  return newName
end

local function makeUnique(name)
  local newName = cleanName(name)
  while nameRegistry[newName] do
    newName = tryMakeUnique(newName)
  end
  nameRegistry[newName] = 1
  return newName
end

local function minMaxForType(dataType)
  local min = 0
  local max = 1e12
  if     dataType == "int8_t" then min=-128 max=127
  elseif dataType == "uint8_t" then min=0 max=255
  elseif dataType == "int16_t" then min=-32768 max=32767
  elseif dataType == "uint16_t" then min=0 max=65535
  elseif dataType == "int32_t" then min=-2147483648 max=2147483647
  elseif dataType == "uint32_t" then min=0 max=4294967295
  elseif dataType == "bool" then min=0 max=1
  elseif dataType == "float" then min=-1e12 max=1e12
  elseif dataType == "double" then min=-1e12 max=1e12
  end
  return min, max
end

local function sizeForType(dataType)
  local size = 1
  if     dataType == "int8_t" then size = 1
  elseif dataType == "uint8_t" then size = 1
  elseif dataType == "int16_t" then size = 2
  elseif dataType == "uint16_t" then size = 2
  elseif dataType == "int32_t" then size = 4
  elseif dataType == "uint32_t" then size = 4
  elseif dataType == "bool" then size = 4
  elseif dataType == "float" then size = 4
  elseif dataType == "double" then size = 8
  end
  return size
end

local function asamType(dataType)
  local asamType = ""
  if     dataType == "int8_t" then asamType = "SBYTE"
  elseif dataType == "uint8_t" then asamType = "UBYTE"
  elseif dataType == "int16_t" then asamType = "SWORD"
  elseif dataType == "uint16_t" then asamType = "UWORD"
  elseif dataType == "int32_t" then asamType = "SLONG"
  elseif dataType == "uint32_t" then asamType = "ULONG"
  elseif dataType == "bool" then asamType = "UBYTE"
  elseif dataType == "float" then asamType = "FLOAT32_IEEE"
  elseif dataType == "double" then asamType = "FLOAT64_IEEE"
  end
  return asamType
end

local function offsetForType(dataType, currentOffset)
  local size = sizeForType(dataType)
  local alignment = size
  local remain = currentOffset % alignment
  local pad = 0
  if remain > 0 then
    pad = alignment - remain
  end
  local offset = currentOffset + pad
  local nextOffset = offset + size
  return offset, nextOffset
end

local function parameters()
  local param = ""
  local blockRegistry = { }
  nameRegistry = { }
  local offset = 0
  local p
  for p = 1, #Model.ExtModeParameters do
    local paramDesc = Model.ExtModeParameters[p]
    local min, max = minMaxForType(paramDesc.OrigType)
    local path = paramDesc.BlockPath:sub(paramDesc.BlockPath:find("/") + 1)
    local paramName = makeUnique("%s_%s" % { paramDesc.BlockName, paramDesc.Name})
    local nextOffset
    offset, nextOffset = offsetForType(paramDesc.OrigType, offset)
    if blockRegistry[path] then
      blockRegistry[path][#blockRegistry[path] + 1] = paramName
    else
      blockRegistry[path] = { }
      blockRegistry[path][1] = paramName
    end
    param = param .. [=[
/begin CHARACTERISTIC %(name)s "%(comment)s" VALUE 0x%(addr)X _%(type)s 0 NO_COMPU_METHOD %(valmin)g %(valmax)g /end CHARACTERISTIC
]=]
    % {
      name =paramName, 
      comment = paramDesc.Name,
      type = asamType(paramDesc.OrigType),
      valmin = min, 
      valmax = max,
      addr = offset
    }
    offset = nextOffset
  end
  local block
  local params
  local groups = ""
  for block, params in pairs(blockRegistry) do
    groups = groups .. "/begin GROUP %s \"%s\" /begin REF_CHARACTERISTIC" % {
      cleanName(block),
      block
    }
    for p = 1, #params do
      groups = groups .. " %s" % { params[p] }
    end
    groups = groups .. " /end REF_CHARACTERISTIC /end GROUP\n"
  end
  return param .. groups
end

local function signals()
  local sig = ""
  local scope
  local signalCounter = 1
  local names = { }
  nameRegistry = { }
  for scope = 1, #Model.ExtModeSignals do
    local scopeDesc = Model.ExtModeSignals[scope]
    local plot
    names[scope] = { }
    for plot = 1, #scopeDesc.Plots do
      local s
      local plotSig = 1;
      names[scope][plot] = { }
      for s = 1, #scopeDesc.Plots[plot].Signals do
        local sigDesc = scopeDesc.Plots[plot].Signals[s]
        local min, max = minMaxForType(sigDesc.OrigDataType)
        local signalName = makeUnique(sigDesc.Name)
        names[scope][plot][plotSig] = signalName
    
        sig = sig .. [=[
/begin MEASUREMENT %(name)s "%(comment)s" %(type)s NO_COMPU_METHOD 0 0 %(valmin)g %(valmax)g ECU_ADDRESS 0x%(addr)X /end MEASUREMENT
]=]
        % {
          name = signalName, 
          comment = sigDesc.Name,
          type = Target.Variables.FLOAT_TYPE == "double" and "FLOAT64_IEEE" or "FLOAT32_IEEE",
          valmin = min, 
          valmax = max,
          addr = (signalCounter-1)*16
        }
        signalCounter = signalCounter+1
        plotSig = plotSig + 1
      end
    end
  end
  local groups = ""
  for scope = 1, #Model.ExtModeSignals do
    local scopeDesc = Model.ExtModeSignals[scope]
    local scopeName = cleanName(scopeDesc.BlockName)
    local path = scopeDesc.BlockPath:sub(scopeDesc.BlockPath:find("/") + 1)
    path = cleanName(path)
    for plot = 1, #scopeDesc.Plots do
      local name = scopeDesc.Plots[plot].Name
      if name == "" then
        name = "Plot %d" % {plot}
      end
      groups = groups .. "/begin GROUP %s \"%s\" /begin REF_MEASUREMENT" % {
        scopeName .. "." .. cleanName(name),
        path .. "/" .. name,
      }
      local i
      for i = 1, #names[scope][plot] do
        groups = groups .. " %s" % { names[scope][plot][i] }
      end
      groups = groups .. " /end REF_MEASUREMENT /end GROUP\n"
    end
  end
  return sig .. groups
end

local function getA2lHeaderName()
  return Target.Variables.BASE_NAME .. "_a2l.h"
end

function Xcp.generateA2l()
  local templateFile = FileUtils.FullPath({Target.Variables.TARGET_ROOT, "src", "rtbox.a2l"})
  local outputFile = FileUtils.FullPath({Target.Variables.BUILD_ROOT, Target.Variables.BASE_NAME .. ".a2l"})
  local hostId
  if Target.Variables.xcpSlaveIdentity == "1" then
    hostId = Target.Variables.TARGET_DEVICE
  elseif Target.Variables.xcpSlaveIdentity == "2" then
    hostId = Target.Variables.xcpSlave
  elseif Target.Variables.xcpSlaveIdentity == "3" then
    hostId = Target.Variables.xcpSlave
  end
  local dictionary = {
    ["|>TASKS<|"] = tasks(),
    ["|>HOSTNAME<|"] = hostId,
    ["|>PARAMETER_DEFINITIONS<|"] = parameters(),
    ["|>SIGNAL_DEFINITIONS<|"] = signals(),
  }
  if not FileUtils.CopyTemplateFile(templateFile, outputFile, dictionary) then
    return FileUtils.Error;
  end
  local inputHandle, fileError = io.open(outputFile, "r")
  if not inputHandle then
    return fileError
  end
  local outputHandle
  local a2l_h = FileUtils.FullPath({Target.Variables.BUILD_ROOT, getA2lHeaderName()})
  outputHandle, fileError = io.open(a2l_h, "w")
  outputHandle:write("static const char* plxA2l =\n")
  while true do
    local line = inputHandle:read()
    if line == nil then break end
    line  = string.gsub(line, "\"", "\\\"") -- escape quotes
    outputHandle:write("\"%s\\n\" \\\n" % {line} )
  end
  outputHandle:write(";\n")
  inputHandle:close()
  outputHandle:close()
end

function Xcp.getEmptyA2lDefinitions()
  local a2ldefs = [=[
const char* plxGetA2l(void) 
{ 
   return "";
}
]=]
  return a2ldefs
end

function Xcp.getA2lDefinitions()
  local a2ldefs = [=[
#include "%(header)s"

const char* plxGetA2l(void) 
{ 
   return plxA2l;
}
]=] % {
  header = getA2lHeaderName()
}
  return a2ldefs
end

return Xcp
