-- YInfo.lua
-- 
-- Copyright (c) 2023 Plexim GmbH
-- All rights reserved.

local SparseMatrix = require('blocks.FPGACoder.SparseMatrix')
local OrderedSet = require('blocks.FPGACoder.OrderedSet')

local mat_C;
local mat_D;

local function areRowsEqual(aRow1, aRow2)
  return mat_C:areRowsEqual(aRow1, aRow2) and mat_D:areRowsEqual(aRow1, aRow2)
end

local function areRowsNegated(aRow1, aRow2)
  return mat_C:areRowsNegated(aRow1, aRow2) and mat_D:areRowsNegated(aRow1, aRow2)
end

local YInfo = {}

function YInfo:new()
  o = {
    mOriginalIndices = {},
    mPermutedIndices = {},
    mOldExternalMeters = {},
    mNewExternalMeters = {},
    mMeterAliases = {},
    mNegated = {}
  }
  self.__index = self
  setmetatable(o, self)
  return o
end

function YInfo:getRowForMeter(aMeterIndex)
  return self.mPermutedIndices[self.mMeterAliases[aMeterIndex]]
end

function YInfo.fromHash(aHash)
  local dcs = OrderedSet:new()
  local dcsWithinUhb = OrderedSet:new()
  local uhb = OrderedSet:new()
  local activeMeters = OrderedSet:new()
  local externalMeters = OrderedSet:new()

  local y = YInfo:new()
  
  for i = 0, aHash.ny - 1 do
    y.mMeterAliases[i] = i
    y.mNegated[i] = false
  end
  
  -- find rows for DCS and UHB meters
  for i, dev in pairs(aHash.dcsDevices) do
    dcs:insert(dev.meterIndex)
    if dev.uhbData ~= nil then
      dcsWithinUhb:insert(dev.meterIndex)
      for j, m in pairs(dev.uhbData.meterIndices) do
        uhb:insert(m)
      end
    end
    if dev.isMeterNeededExternally then
      externalMeters:insert(dev.meterIndex)
      table.insert(y.mOldExternalMeters, dev.meterIndex)
    end
  end
  --print("dcs: %s\n" % { dump(dcs:keys()) })
  --print("uhb: %s\n" % { dump(uhb:keys()) })
   
  -- collect all active meter indices
  for x, m in pairs(aHash.activeMeters) do
    for x, meterIndex in pairs(m.meterIndices) do
      activeMeters:insert(meterIndex)
    end
  end
  --print("active: %s\n" % { dump(activeMeters:keys()) })
  
  -- state space order output
  -- 1. dcs meters associated with uhbs
  -- 2. dcs meters not associated with uhbs
  -- 3. uhb meters
  -- 4. remaining meters
  
  -- dcs meters should _not_ be aliased, simply add them
  for x, key in pairs(dcsWithinUhb:keys()) do
    table.insert(y.mOriginalIndices, key)
  end

  for x, key in pairs(dcs:keys()) do
    if not dcsWithinUhb:hasKey(key) then
      table.insert(y.mOriginalIndices, key)
    end
  end
  
  -- 3. uhb meters may be aliased if an equivalent meter is found
  -- At the moment they are scaled by the firmware and cannot be external
  mat_C = SparseMatrix.fromHash(aHash.mat_C)
  mat_D = SparseMatrix.fromHash(aHash.mat_D)
  
  for x, uhbMeterIdx in pairs(uhb:keys()) do
    local aliasFound = false
    for x, existingIdx in pairs(y.mOriginalIndices) do
      if areRowsEqual(uhbMeterIdx, existingIdx) then
        -- print("UHB meter %d is identical to %d" % {uhbMeterIdx, existingIdx} )
        y.mMeterAliases[uhbMeterIdx] = existingIdx
        aliasFound = true
        break
      end
    end
    if not aliasFound then
      table.insert(y.mOriginalIndices, uhbMeterIdx)
    end
  end
  
  -- 4. remaining meters
  for x, i in pairs(activeMeters:keys()) do
    if not dcs:hasKey(i) and not uhb:hasKey(i) then
    
      local aliasFound = false      
      table.insert(y.mOldExternalMeters, i)
      for x, existingIdx in pairs(y.mOriginalIndices) do
      
        if not uhb:hasKey(existingIdx) then -- rows for UHB are scaled and cannot be reused
          if areRowsEqual(i, existingIdx) then
            -- print("Meter %d is identical to %d" % {i, existingIdx} )
            y.mMeterAliases[i] = existingIdx
            externalMeters:insert(existingIdx)
            aliasFound = true
            break
          end

          if areRowsNegated(i, existingIdx) then
            -- print("Meter %d is identical to negated %d" % {i, existingIdx} )
            y.mMeterAliases[i] = existingIdx
            externalMeters:insert(existingIdx)
            y.mNegated[i] = true
            aliasFound = true
            break
          end
        end
      end
      if not aliasFound then
        externalMeters:insert(i)
        table.insert(y.mOriginalIndices, i)
      end
    end
  end
  
  y.mNewExternalMeters = externalMeters:keys()

  for i, idx in pairs(y.mOriginalIndices) do
    y.mPermutedIndices[idx] = i - 1
  end
  
  return y
end

function YInfo:getExternalMeterOffset(aMeterIndex)
  local aliasedIdx = self.mMeterAliases[aMeterIndex]
  for pos, idx in pairs(self.mNewExternalMeters) do
    if idx == aliasedIdx then
      return pos - 1
    end
  end
  print("No index found for %d (%s)\n" % { aMeterIndex, dump(aliasedIdx) })
  return nil
end

function YInfo:numY_ext()
  return #self.mNewExternalMeters
end

function YInfo:numY()
  return #self.mOriginalIndices
end


return YInfo
