local Module = {}

function Module.getBlock(globals)
  local SpiController = require('blocks.block').getBlock(globals)

  function SpiController:getDirectFeedthroughCode()
    local InitCode = StringList:new()
    local OutputSignal = StringList:new()
    local OutputCode = StringList:new()
    local Require = ResourceList:new()

    local moduleId = SpiController:getInstanceNumber()
    
    if moduleId > 1 then
      return "A maximum of two SPI blocks (controller or peripheral) can be used in one model."
    end

    -- register the hardware resources
    Require:add("SPI", moduleId + 1)

    local transferTime = Block.Mask.t_trans
    local numChannels = Block.Mask.numChannels
    local numWords = Block.Mask.numWords
    local sampleTime = Block.SampleTime[1]
    if sampleTime == math.huge then
      sampleTime = Block.NativeTask.SampleTime[1]
    end
    local extraTime = sampleTime - Target.Variables.SAMPLE_TIME

    if (transferTime > sampleTime) then
       return "Error in SPI module SPI%d: Data transmission time is longer than model step size." % { moduleId }
    end

    -- Setup Config struct
    InitCode:append("{\n")
    InitCode:append("const SPI_CONF conf = {\n")
    InitCode:append(".PERIPHERAL = 0,\n")
    InitCode:append(".CLK_SPC1 = %i,\n" % { Block.Mask.ticksSetupCS })
    InitCode:append(".CLK_SPC2 = %i,\n" % { Block.Mask.ticksHoldCS })
    InitCode:append(".SPIMODE = %i,\n" % { Block.Mask.spiMode-1 })
    InitCode:append(".DataBits = %i,\n" % { Block.Mask.wordSize })
    InitCode:append(".bitOrder = %i,\n" % { Block.Mask.bitOrder-1 })
    InitCode:append(".HS_MODE = %i,\n" % { Block.Mask.useClockInput-1 })
    InitCode:append(".STRMBR = %i,\n" % { numChannels-1 })
    InitCode:append(".CS_POLARITY = 0,\n")
    InitCode:append(".TRGDELAY = 0\n")
    InitCode:append("};\n")

    local numSPIQWords = math.floor((numChannels*numWords+3)/4)
    if Target.Name == "PLECS RT Box 2" or Target.Name == "PLECS RT Box 3" then
       numSPIQWords = 2*numSPIQWords
    end
    InitCode:append("static uint16_t outputBuffer[%i];\n" % { 2*4*numSPIQWords })

    -- Setup SPI moduleID for Controller Mode with numWords sent words per channel
    InitCode:append("setupSPI(%i, &conf, %.17e,%i, %.17e, %i, outputBuffer, %i*sizeof(uint16_t), %.17e);\n" % 
      { moduleId, Block.Mask.fClock, numWords, Block.Mask.tDelayCS, Block.Mask.delayCSMode-1, 4*numSPIQWords, extraTime })
    InitCode:append("}\n")

    -- Program Inputs in Init Code
    for i = 1, 5 do
       local channel = Block.Mask.inputVec[i]
       if (channel >= 0) then
          -- Input configured to be used for SPI   		
          Require:add("Digital input", channel);
          InitCode:append("setupSPIInput(%i, %i, %i);\n" % { moduleId, i-1, channel })
       end
    end

    local clkoutVecLength = Block.Mask.clkoutVecL
    -- Program CLK output channels
    for i = 1, clkoutVecLength do
       local channel = Block.Mask.clkoutVec[i];
   
        -- Output configured to be used for SPI   		
        Require:add("Digital output", channel);
       InitCode:append("setupSPIOutput(%i, 0, %i);\n" % { moduleId, channel })
    end

    local csoutVecLength = Block.Mask.csoutVecL
    -- Program CS output channels
    for i = 1, csoutVecLength do
       local channel = Block.Mask.csoutVec[i]
   
        -- Output configured to be used for SPI   		
        Require:add("Digital output", channel)
       InitCode:append("setupSPIOutput(%i,2, %i);\n" % { moduleId, channel })
    end


    -- Program digital output channels
    local outid = {1,3,4,5}
    for i = 1, numChannels do
       local channel = Block.Mask.outputVec[i]
   
       if channel >= 0 then
           -- Output configured to be used for SPI   		
           Require:add("Digital output", channel)
          InitCode:append("setupSPIOutput(%i,%i, %i);\n" % { moduleId, outid[i], channel })
       end
    end

    for i = 1, 4 do
       OutputSignal[i] = {}
       for j = 1, Block.NumOutputSignals[i] do
          if i <= numChannels and j <= numWords then 
             OutputSignal[i][j] = "getSPIIn%i(%i)" % { moduleId, (i-1)+(j-1)*numChannels }
          else
             OutputSignal[i][j] = "0"
          end
       end
    end

    OutputSignal[5] = {}
    OutputSignal[5][1] = "getSPIInputState%i()" % { moduleId }

    OutputCode:append("{\n")
    OutputCode:append("int currReadBufferIdx = plxSPICurrentReadBuffer[%i];\n" % { moduleId } )
    OutputCode:append("int newReadBufferIdx = !currReadBufferIdx;\n" % { moduleId } )
    OutputCode:append("plxSPIUpdateFlags[%i][currReadBufferIdx] = 0;\n" % { moduleId } )
    OutputCode:append("if (plxSPIUpdateFlags[%i][newReadBufferIdx]) {\n" % { moduleId } )
    OutputCode:append("   plxSPIRxBufferPtr%i=plxSPIOutputBuffer[%i]+newReadBufferIdx*plxSPIBufLength[%i];\n" % { moduleId, moduleId, moduleId } )
    OutputCode:append("   plxSPICurrentReadBuffer[%i]=newReadBufferIdx;\n" % { moduleId } )
    OutputCode:append("}\n")
    OutputCode:append("}\n")

    return {
      Include = "plexim/SPI.h",
      Require = Require,
      InitCode = InitCode,
      OutputSignal = OutputSignal,
      OutputCode = OutputCode,
      UserData = { bid = SpiController:getId() }
    }
  end
  
  
  function SpiController:getNonDirectFeedthroughCode()
    local UpdateCode = StringList:new()
    local numChannels = Block.Mask.numChannels
    local numWords = Block.Mask.numWords
    local moduleId = SpiController:getInstanceNumber()

    -- SPI Outgoing Data  	 
    local id = 0;
    for i = 1, numWords do
        for j = 1, numChannels do	
            -- SPI Outgoing Data   		
        UpdateCode:append("setSPIOut%i(%i, %s);\n" % { moduleId, id, Block.InputSignal[j][i] })
          id = id + 1
        end
    end
    UpdateCode:append("*plxIo.SPIEnable |= %i;\n" % { moduleId+1 })

    return {
        UpdateCode = UpdateCode
    }
  end
  
  return SpiController
end  
return Module
