import sys
if sys.version_info.major == 3:
    import xmlrpc.client as xmlrpclib
    import http.client as httplib
    from io import BytesIO
else:
    import xmlrpclib
    import httplib
import jsonrpc_requests as jsonrpclib
import collections.abc      # to make jsonrpc_requests usable for Python 3.10+
collections.Mapping = collections.abc.Mapping
import requests
import socket
import time
import os
import inspect
from robot.libraries.DateTime import convert_time, get_current_date, add_time_to_date
from robot.api import logger
import base64
import json
from threading import Thread
from threading import Condition
from struct import unpack
import csv


class Threaded(object):
    def __init__(self, pollingIntervalInSeconds=0.1, **kwargs):
        super(Threaded, self).__init__(**kwargs)
        self._thread = None
        self._threadShouldRun = True
        self._threadLeftRunLoop = False
        self._pollingIntervalInSeconds = pollingIntervalInSeconds
        self._sleep_condition = Condition()

    def setupThread(self, name=None):
        if not name:
            name = self.__class__.__name__

        self._thread = Thread(target=self._run, name=name)
        self._thread.setDaemon(True)

    def startThread(self):
        if self._thread is None:
            self.setupThread()
        self._thread.start()

    def stopThread(self):

        self._threadShouldRun = False
        if self._thread.is_alive():
            self.waitOnThreadToLeave()

        del self._thread
        self._thread = None

    def waitOnThreadToLeave(self, timeout=1):
        waitTime = timeout
        decrValue = 0.2

        if self._threadLeftRunLoop:
            return

        while waitTime > 0:
            time.sleep(decrValue)
            waitTime -= decrValue
            if self._threadLeftRunLoop:
                break

    def _threadSleepInterval(self, sleepIntervalInSeconds=None):
        if sleepIntervalInSeconds is not None:
            waitTime = sleepIntervalInSeconds
        else:
            waitTime = self._pollingIntervalInSeconds

        if self._sleep_condition.acquire(blocking=False):
            self._sleep_condition.wait(timeout=waitTime)
            self._sleep_condition.release()


class PlecsXMLJSONRPC(Threaded):
    ROBOT_LIBRARY_SCOPE = 'SUITE'

    def __init__(self, rpcmethod):
        super(PlecsXMLJSONRPC, self).__init__()
        self._rpcmethod = rpcmethod
        self._rpclib = jsonrpclib if self._rpcmethod == 'JSON' else xmlrpclib
        self._plecs = self._rpclib.Server("http://127.0.0.1:1080/RPC2")
        self._models = []
        counter = 0
        counterLimit = 5
        test_json = {
            "jsonrpc": "2.0",
            "method": "load",
            "params": ["SomeInvalidPath"]
        }
        while counter < counterLimit:
            try:
                if self._rpcmethod == 'JSON':
                    response = requests.post("http://127.0.0.1:1080/RPC2", json=test_json)
                    response.raise_for_status()  # Raise an HTTPError for bad responses
                else:
                    self._plecs.plecs.load('SomeInvalidPath')
                
                break
            except (socket.error, requests.exceptions.RequestException):
                print("Waiting for PLECS to get ready...")
                time.sleep(1)
            except xmlrpclib.Fault:
                break
            counter += 1
        if counter == counterLimit:
            raise Exception(
                "Gave up waiting for PLECS to get ready. (PLECS could not be started, hangs, has RPC disabled or "
                "uses a different port.)")
        stats = self._plecs.plecs.statistics()
        self._plecsVersion = '.'.join(str(s) for s in stats['version'][0:3]) + ' Build ' + stats['version'][3]
        self._pollingIntervalInSeconds = None
        self._signal_width = None
        self._file_name = None
        self._ip = None
        self._rtbox = None
        self._noDataReceived = True
        self._header = []

    def __del__(self):
        for model in self._models:
            self._plecs.plecs.close(model)

    def rtBoxSetupServer(self, rtbox):
        self._ip = socket.gethostbyname(rtbox)
        self._rtbox = self._rpclib.Server("http://" + self._ip + ":9998/RPC2")

    def plecsLoadModel(self, model, relativePath=''):
        path = os.path.abspath(inspect.getfile(inspect.currentframe()) + '/..')
        stats = self._plecs.plecs.statistics()
        getOpenModelNames = []
        numberOfOpenModels = stats['numberOfOpenModels']
        for modelIndex in range(numberOfOpenModels):
            getOpenModelNames.append(stats['models'][modelIndex]['id'])

        if model not in getOpenModelNames:
            if relativePath != '':
                relativePath += '/'
            self._plecs.plecs.load(path + '/' + relativePath + model)
        self._models.append(model)

    def plecsGetVersion(self):
        logger.info('Using PLECS ' + self._plecsVersion)
        return self._plecsVersion

    def plecsGenerateCode(self, model, sub='', rtboxtype='', **codegenVars):
        if sub == '':
            path = model
        else:
            path = model + "/" + sub
        if rtboxtype != '':
            self._plecs.plecs.set(path, 'CodeGenTarget', rtboxtype)
            logger.info('Generating code for ' + rtboxtype)
        if codegenVars:
            modelArgs = {'ModelVars': codegenVars}
            logger.info(str(modelArgs))
            self._plecs.plecs.codegen(path, modelArgs)
        else:
            self._plecs.plecs.codegen(path)

    def rtBoxUpload(self, model, sub=''):
        if sub == '':
            sub = model
        with open(model + "_codegen/" + sub + ".elf", "rb") as f:
            binary = base64.b64encode(f.read()).decode() if self._rpcmethod == 'JSON' else xmlrpclib.Binary(f.read())
            self._rtbox.rtbox.load(binary)
        f.close()
        serials = self._rtbox.rtbox.serials()
        applicationInfo = self._rtbox.rtbox.querySimulation()
        logger.info('Using RT Box with FPGA ' + serials['fpgaVersion'] + ', Firmware ' + serials['firmwareVersion'] \
                    + ' Build ' + serials['firmwareBuild'] + ', Application ' + applicationInfo['applicationVersion'])

    def rtBoxReboot(self):
        self._rtbox.rtbox.reboot('reboot')

    def plecsGetCircuitBitmap(self, subsystem=''):
        modelName = os.path.basename(self._models[0])
        if subsystem != '':
            modelName = modelName + '/' + subsystem
        logger.info(modelName)
        schematic = self._plecs.plecs.webserver('getCircuitBitmap', modelName, {})
        schematic_base64 = schematic['imageData'] if self._rpcmethod == 'JSON' else base64.b64encode(schematic['imageData'].data).decode('ascii')
        logger.info('<img src="data:image/png;base64,' + schematic_base64 + '">', html=True)

    def rtBoxStart(self, waitForTrigger=False):
        self._rtbox.rtbox.start(1 if waitForTrigger else 0)

    def rtBoxStop(self):
        self._rtbox.rtbox.stop()

    def rtBoxGetStatus(self, identifier=''):
        statusList = ['temperature', 'fanSpeed', 'logPosition', 'applicationLog', 'clearLog', 'modelTimeStamp']
        if identifier in statusList:
            return self._rtbox.rtbox.status(0, 0)[identifier]
        else:
            return self._rtbox.rtbox.status(0, 0)

    def rtBoxGetVersion(self, identifier=''):
        versionList = ['cpu', 'mac', 'boardRevision', 'firmwareVersion', 'firmwareBuild', 'fpgaVersion', 'ipAddresses',
                       'board']
        if identifier in versionList:
            return self._rtbox.rtbox.serials()[identifier]
        else:
            return self._rtbox.rtbox.serials()

    def rtBoxGetHostname(self):
        return self._rtbox.rtbox.hostname()

    def rtBoxQuerySimulation(self, identifier=''):
        querySimulationList = ['sampleTime', 'modelName', 'applicationVersion', 'modelTimeStamp']
        if identifier in querySimulationList:
            return self._rtbox.rtbox.querySimulation()[identifier]
        else:
            return self._rtbox.rtbox.querySimulation()

    def rtBoxQueryCounter(self, identifier=''):
        queryCounterList = ['maxCycleTime', 'runningCycleTime', 'maxCycleTime1', 'runningCycleTime1', 'maxCycleTime2',
                            'runningCycleTime2', 'maxCycleTime3', 'runningCycleTime3']
        if identifier in queryCounterList:
            return self._rtbox.rtbox.queryCounter()[identifier]
        else:
            return self._rtbox.rtbox.queryCounter()

    def rtBoxResetCounter(self):
        self._rtbox.rtbox.resetCounter()

    def rtBoxGetLEDs(self):
        headers = {"Content-type": "application/x-www-form-urlencoded"}
        conn = httplib.HTTPConnection(self._ip)
        conn.request("POST", "/cgi-bin/front-panel.cgi", "", headers)
        response = conn.getresponse()
        json_data = response.read()
        data = json.loads(json_data)
        conn.close()
        return data['leds']

    def rtBoxSetValue(self, path, value):
        self._rtbox.rtbox.setProgrammableValue(path, eval(value))

    def rtBoxGetValue(self, path, triggerCount=0.0, waitInterval=1.0):
        counter = 0
        counterLimit = waitInterval * 10.0
        while self._rtbox.rtbox.getCaptureTriggerCount(path) == triggerCount and counter < counterLimit:
            time.sleep(0.1)
            counter += 1
        if counter == counterLimit:
            logger.warn('The data capturing in Data Capture Block {path} was not triggered during the wait interval of '
                        '{counterLimit} seconds!'.format(path=path, counterLimit=counterLimit / 10.0))
            return []
        else:
            return self._rtbox.rtbox.getCaptureData(path)['data']

    def rtBoxCaptureTriggerCount(self, path):
        return self._rtbox.rtbox.getCaptureTriggerCount(path)

    def rtBoxWaitForTrigger(self, path, timeout):
        finalTime = add_time_to_date(get_current_date(), convert_time(timeout))
        while self._rtbox.rtbox.getCaptureTriggerCount(path) == 0:
            if get_current_date() > finalTime:
                raise Exception("Timeout while waiting for RT Box trigger")

    def rtBoxIsRunning(self):
        ledsData = self.rtBoxGetLEDs()['running']
        if ledsData == '1':
            return True
        else:
            return False

    def rtBoxGetDataCaptureBlocks(self):
        return self._rtbox.rtbox.getDataCaptureBlocks()

    def rtBoxGetProgrammableValueBlocks(self):
        return self._rtbox.rtbox.getProgrammableValueBlocks()

    def rtBoxBackgroundLogging(self, value=False, signal_width=2, sample_time=0.1, file_name='data.csv', header=''):
        self._pollingIntervalInSeconds = sample_time / 10
        self._signal_width = signal_width
        self._file_name = file_name
        if value:
            if header.startswith('[') and header.endswith(']'):
                header = header[1:-1]
            self._header = list(header.split(', '))
            if not len(self._header) == signal_width:
                logger.error(
                    'Number of header entries is not consistent with the signal width. Header will be ignored!')
                self._header = []
            self._enable()
        else:
            self._disable()

    def rtBoxBackgroundLoggingEnabled(self):
        return True if self._thread and self._thread.is_alive() else False

    def _enable(self):
        # Listen on all interfaces
        host = '0.0.0.0'
        port = 52345
        addr = (host, port)
        self.UDPSock = socket.socket(socket.AF_INET, socket.SOCK_DGRAM)
        self.UDPSock.bind(addr)
        logger.info('Enabling background logging')
        self._threadShouldRun = True
        self._threadLeftRunLoop = False

        self.setupThread()
        self.startThread()

    def _disable(self):
        if self._thread:
            logger.info('Disabling background logging')
            self.stopThread()
            logger.info('Background logging disabled')
            if self._noDataReceived:
                logger.error('No data received over UDP during the Robot Framework test, please check the configured '
                             'IP address!')
        else:
            logger.info('Background logging already disabled')

    def _run(self):
        data_file = open(self._file_name, 'w')
        writer = csv.writer(data_file, delimiter=',')
        if self._header:
            writer.writerow(self._header[:])
        while self._threadShouldRun:
            (data, addr) = self.UDPSock.recvfrom(4 * self._signal_width)
            value = []
            timestamp = unpack('f', data[0:4])[0]
            value.append(timestamp)
            for i in range(1, self._signal_width):
                value.append(unpack('f', data[4 * i:4 * i + 4])[0])
            if value:
                self._noDataReceived = False

            writer.writerow(value[:])
            if self._threadShouldRun:
                self._threadSleepInterval(self._pollingIntervalInSeconds)

        self._threadLeftRunLoop = True
        self.UDPSock.close()
        data_file.close()
