/*
   Copyright (c) 2017-2020 by Plexim GmbH
   All rights reserved.

   A free license is granted to anyone to use this software for any legal
   non safety-critical purpose, including commercial applications, provided
   that:
   1) IT IS NOT USED TO DIRECTLY OR INDIRECTLY COMPETE WITH PLEXIM, and
   2) THIS COPYRIGHT NOTICE IS PRESERVED in its entirety.

   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
   OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
   SOFTWARE.
 */
 
#ifndef SPI_H_
#define SPI_H_

#include "plexim/hw_wrapper.h"

#ifdef __cplusplus
extern "C"
{
#endif

extern int plxNumQWordsSPI0;

extern int plxNumQWordsSPI1;
extern void* plxSPIOutputBuffer[2];
extern uint16_t* plxSPIRxBufferPtr0;
extern uint16_t* plxSPIRxBufferPtr1;
extern int plxSPIBufLength[2];
extern int plxSPICurrentReadBuffer[2];
extern int plxSPIUpdateFlags[2][2];

/// Defining the configuration structure for SPI module
typedef struct 
{
  u8 PERIPHERAL;///<Single/Multi Peripheral Mode (0,1)
  u8 CLK_SPC1;///<SPI CLK spaces between CS edge and first WR event (1 to 15).
  u8 CLK_SPC2;///<SPI CLK spaces between last RD event and CS edge (1 to 15).
  u8 SPIMODE;///<SPI MODE (00 to 11 (CKPHA, CKPOL)).
  u8 DataBits;///< Bits per character (1-16)
  u8 bitOrder;///< MSB/LSB first (0, 1)
  u8 HS_MODE;///< 0 - SPI_CLK_OUT used for ReaderCLK, 1 - SPI_CLK_IN used for ReaderCLK
  u8 STRMBR;///< Number of parallel stream signals
  u8 CS_POLARITY;///< ACTIVE polarity of CS
  u16 TRGDELAY;
} SPI_CONF;

/**
* @brief Setup the SPI Controller.
* 
* 
*/
void setupSPI(int aModule, const SPI_CONF* aConfig, double aClockFrequency, 
              int aNumWordsInput0, double aDelayCS, int aDelayCSMode, uint16_t* aOutputBuffer, int aBufLength, double aExtraTime);

/**
* @brief Setup a SPI Input Channel.
* 
* ID Map: [D0,D1,D2,D3, CLK,CS0]
* ID Map: [0, 1, 2 , 3 , 4 , 5 ]
*/
void setupSPIInput(int aModule, int aID, int aChannel);

/**
* @brief Setup a SPI Output Channel.
* 
* ID Map: [CLK,D0,CS0,D1,D2,D3 ]
* ID Map: [ 0, 1, 2 , 3, 4, 5  ]
*/
void setupSPIOutput(int aModule,int aID, int aChannel);

/** @brief Send value to SPI buffer 0
 * 
 * @param 
 */ 
#define setSPIOut0(n, x) \
{ \
   *(plxIoOut.SPIOutput0 + ((2*((n) & 0xfffc)) | ((n) & 3))) = x; \
}

/**
 * @brief read value from SPI buffer 0
 * 
 *
 * @param n ID of SPI input.
 */   
#define getSPIIn0(n) ((float)*(plxSPIRxBufferPtr0 + ((2*((n) & 0xfffc)) | ((n) & 3))))

#define getSPIInputState0() (plxSPIUpdateFlags[0][plxSPICurrentReadBuffer[0]])
#define getSPIInputState1() (plxSPIUpdateFlags[1][plxSPICurrentReadBuffer[1]])

/** @brief Send value to SPI buffer 1
 * 
 * @param 
 */ 
#define setSPIOut1(n, x) \
{ \
   *(plxIoOut.SPIOutput1 + ((2*((n) & 0xfffc)) | ((n) & 3))) = x; \
}

/**
 * @brief read value from SPI buffer 1
 * 
 *
 * @param n ID of SPI input.
 */   
#define getSPIIn1(n) ((float)*(plxSPIRxBufferPtr1 + ((2*((n) & 0xfffc)) | ((n) & 3))))

#define plxEnableSpiModule(n) *plxIoOut.SPIEnable |= (n+1)

void plxUpdateSPIBuffer0();
void plxUpdateSPIBuffer1();

#ifdef __cplusplus
}
#endif

#endif /*SPI_H_*/
