-- BlockUtils.lua
-- 
-- Copyright (c) 2024 Plexim GmbH
-- All rights reserved.

local BlockUtils = {}

function BlockUtils.stringToNumber(aSignal, aConvertBools)
  local n = tonumber(aSignal)
  if n ~= nil then
    return n
  end
  -- test if aSignal is float literal like 0.1f
  if string.sub(aSignal, -1, -1) == "f" then
    return tonumber(string.sub(aSignal, 1, -2))
  end
  if aConvertBools == true then
    if aSignal == "true" then
      return 1
    elseif aSignal == "false" then
      return 0
    end
  end
  return nil
end

function BlockUtils.inputSignalToNumber(aTermIndex, aSignalIdx, aConvertBools)
  local signalValue = Block.InputSignal[aTermIndex][aSignalIdx]
  local n = BlockUtils.stringToNumber(signalValue, aConvertBools)
  if n ~= nil then
    return n
  end
  if BlockUtils.isUnconnectedInputSignal(aTermIndex, aSignalIdx) then
    return 0
  end
  return nil
end

function BlockUtils.firstCaptureIndex(aConst0)
  for idx = 1, #aConst0 do
    if aConst0[idx] == false then
      return idx
    end
  end
  return nil
end

function BlockUtils.splitPathTail(aPath, aNumElements)
  local len = string.len(aPath)
  local elemCount = 0;
  for idx = 1, len do
    if string.byte(aPath, -idx) == 47 and string.byte(aPath, -(idx+1)) ~= 92 then -- / and not \/
      elemCount = elemCount + 1
      if elemCount == aNumElements then
        return string.sub(aPath, 1, len-idx)
      end
    end
  end
  return ""
end

function BlockUtils.isUnconnectedMetaData(aMetaData)
  if aMetaData ~= nil then
    if aMetaData["SignalProperties"] ~= nil and aMetaData["SignalProperties"] ~= "" then
      local unconnectedProperty = aMetaData["SignalProperties"]["Unconnected"]
      return (unconnectedProperty == true or unconnectedProperty == 1)
    end
  end
  return false
end

function BlockUtils.isUnconnectedInputSignal(aTermIndex, aSignalIndex)
  local metaData = Block.InputMetaData[aTermIndex][aSignalIndex]
  return BlockUtils.isUnconnectedMetaData(metaData)
end


--[[
  The extractPwmIndex function makes sure that signals connected to Nanostep
  switches originate either from PWM capture blocks or have a constant value.
  In the former case, the corresponding digital input pin number is returned.
--]]
function BlockUtils.extractPwmIndex(aTermIndex, aSignalIndex)
  local metaData = Block.InputMetaData[aTermIndex][aSignalIndex]
  if metaData ~= nil then
    if metaData["port"] ~= nil and metaData["port"] ~= "" then
      local port = metaData["port"]
      if tonumber(port) > 63 then
        error({ errMsg = ("Invalid port number %s.") % { port }})
      end
      return port, -1
    elseif metaData["sourceInfo"] ~= nil then
      local constVal = metaData["sourceInfo"]["constant"]
      if constVal == 0 or constVal == 1 then
        return -1, constVal
      end
    end
  end 
  -- fall back to evaluating signal value
  local constVal = BlockUtils.inputSignalToNumber(aTermIndex, aSignalIndex, true)
  if constVal == 0 or constVal == 1 then
    return -1, constVal
  else
    error({ errMsg = ("For FPGA simulation, power modules must " ..
      "be connected directly to PWMCapture blocks (input %d, signal %d).") % { aTermIndex, aSignalIndex }
    })
  end
end

function BlockUtils.processError(aVal, level)
 if level == nil then
   level = 1
 end
 if aVal == nil then
   return ({ message = "Error information is nil", level = level})
 elseif type(aVal) == "string" then
   return ({ message = aVal, level = level})
 elseif aVal.errMsg ~= nil then
   return ({ message = aVal.errMsg, level = level})
 else
   return ({ message = "Unknown error information.", level = level})
 end
end

function BlockUtils.round (a)
	return math.floor(a + 0.5)
end

function BlockUtils.pairsByKeys (t, f)
  local a = {}
  for n in pairs(t) do table.insert(a, n) end
  table.sort(a, f)
  local i = 0      -- iterator variable
  local iter = function ()   -- iterator function
    i = i + 1
    if a[i] == nil then return nil
    else return a[i], t[a[i]]
    end
  end
  return iter
end

--[[
  This function checks if a partial of the Nanostep solver receives voltages
  from the FlexArray solver. If it doesn't it returns nil. If it does, it
  returns a table with two fields:
      - idx, which is the index of the meter that provides the voltage
      - gain, which is either 1 or -1 and indicates if the voltage should be
        inverted before being applied to the Nanostep solver
--]]
function BlockUtils.nanostep_get_flexarray_volt_sources(block, partial)
  local source_1 = {}
  local source_2 = {}
  if block.InputMetaData then
    local inputMetaDataV1 = block.InputMetaData[1][partial]
    local inputMetaDataV2 = block.InputMetaData[2][partial]
    if (inputMetaDataV1 and inputMetaDataV1.sourceInfo and inputMetaDataV1.sourceInfo.fromFPGA == 1) then
      source_1.idx = inputMetaDataV1.sourceInfo.meterIdx
      source_1.gain = inputMetaDataV1.sourceInfo.gain
    end
    if (inputMetaDataV2 and inputMetaDataV2.sourceInfo and inputMetaDataV2.sourceInfo.fromFPGA == 1) then
      source_2.idx = inputMetaDataV2.sourceInfo.meterIdx
      source_2.gain = inputMetaDataV2.sourceInfo.gain
    end
  end
  return source_1, source_2
end

function find_max_err_and_coef(coefs_to_check, coefs, precision)
  local max_err = 0
  local max_coef = 0
  for _, name in ipairs(coefs_to_check) do
    local c = coefs[name]
    if c ~= 0 then
      local rel_err = math.abs(BlockUtils.round(c * 2^precision) / 2^precision / c - 1);
      max_err = math.max(rel_err, max_err)
    end
    max_coef = math.max(math.abs(c), max_coef)
  end
  return max_err, max_coef
end

--[[
  The check_nanostep_coefficients function makes sure the coefficients are not
  too big or too small. If the check fails, an error string is returned.
--]]
function check_nanostep_coefficients(block, scaled_coefficients, precision, is_partial)
  local coefs_a_to_check = {};
  local coefs_b_to_check = {};

  if block.Mask['singlephase'] ~= 0 or block.Mask['psfb'] ~= 0 or block.Mask['totempole'] ~= 0 then
    table.insert(coefs_a_to_check, 'A1_14');
    table.insert(coefs_a_to_check, 'A1_25');
    table.insert(coefs_a_to_check, 'A1_36');
    table.insert(coefs_b_to_check, 'B1_11');
    table.insert(coefs_b_to_check, 'B1_22');
    table.insert(coefs_b_to_check, 'B1_33');
  end

  if block.Mask['twophase'] ~= 0 then
    table.insert(coefs_a_to_check, 'A12_13');
    table.insert(coefs_a_to_check, 'A12_14');
    table.insert(coefs_a_to_check, 'A23_23');
    table.insert(coefs_a_to_check, 'A23_24');
    table.insert(coefs_a_to_check, 'A13_13');
    table.insert(coefs_a_to_check, 'A13_14');
    table.insert(coefs_b_to_check, 'B12_11');
    table.insert(coefs_b_to_check, 'B23_22');
    table.insert(coefs_b_to_check, 'B13_13');
  end

  if block.Mask['threephase'] ~= 0 then
    table.insert(coefs_a_to_check, 'A3_13');
    table.insert(coefs_a_to_check, 'A3_14');
    table.insert(coefs_a_to_check, 'A3_23');
    table.insert(coefs_a_to_check, 'A3_24');
    table.insert(coefs_b_to_check, 'B3_11');
    table.insert(coefs_b_to_check, 'B3_13');
    table.insert(coefs_b_to_check, 'B3_22');
    table.insert(coefs_b_to_check, 'B3_23');
    if block.Mask['L3'] ~= 0 then
      table.insert(coefs_b_to_check, 'B3_12');
      table.insert(coefs_b_to_check, 'B3_21');
    end
  end

  if block.Mask['inject_v_ac'] ~= 0 then
    coefs_b_to_check = table.move(coefs_b_to_check, 1, #coefs_a_to_check, #coefs_b_to_check+1, coefs_a_to_check)
    coefs_a_to_check = {}
  end

  local R1, R2, R3 = block.Mask['R1'], block.Mask['R2'], block.Mask['R3']
  local L1, L2, L3 = block.Mask['L1'], block.Mask['L2'], block.Mask['L3']
  local C1, C2, C3 = block.Mask['C1'], block.Mask['C2'], block.Mask['C3']

  local RbyL1, RbyL2, RbyL3, LC1, LC2, LC3 = 0, 0, 0, 0, 0, 0
  if is_partial then
    RbyL1 = R1/L1;
    RbyL2 = R2/L2;
    RbyL3 = R3/L3;
    LC1 = 1/C1/L1;
    LC2 = 1/C2/L2;
    LC3 = 1/C3/L3;
  else
    RbyL1 = (R1+1/(1/R2+1/R3)) / (L1+1/(1/L2+1/L3));
    RbyL2 = (R2+1/(1/R1+1/R3)) / (L2+1/(1/L1+1/L3));
    RbyL3 = (R3+1/(1/R1+1/R2)) / (L3+1/(1/L1+1/L2));
    LC1 = 1/(1/C1+1/(C2+C3)) * (L1+1/(1/L2+1/L3));
    LC2 = 1/(1/C2+1/(C1+C3)) * (L2+1/(1/L1+1/L3));
    LC3 = 1/(1/C3+1/(C1+C2)) * (L3+1/(1/L1+1/L2));
  end

  local tauMinInv = math.max(RbyL1, RbyL2, RbyL3)
  if tauMinInv ~= tauMinInv then -- If NaN
    return 'Unable to verify precision. Double check the input parameters.'
  end
  if tauMinInv * 20 * block.Mask['h'] > 1 then
    return 'Numerical stability jeopardized because a resistance value is too large.'
  end

  if is_partial then
    local w_max = math.sqrt(math.max(LC1, LC2, LC3))
    if w_max ~= w_max then -- If NaN
      return 'Unable to verify precision. Double check the input parameters.'
    end
    if w_max * 20 * block.Mask['h'] > 1 then
      return 'Numerical stability jeopardized because a resonant frequency is too high.'
    end
  else
    local w_min = math.sqrt(math.min(LC1, LC2, LC3))
    if w_min ~= w_min then -- If NaN
      return 'Unable to verify precision. Double check the input parameters.'
    end
    if w_min < 20 * block.Mask['h'] then
      return 'Numerical stability jeopardized because a resonant frequency is too high.'
    end
  end

  local max_err_B, max_coef_B = find_max_err_and_coef(coefs_b_to_check, scaled_coefficients, precision)
  if max_err_B > 0.01 then
    return 'Numerical error exceeds 1 percent because an inductance value is too large.'
  end
  if max_coef_B >= 1 then
    return 'Cannot compute numerical solution because an inductance value is too small.'
  end

  local max_err_A, max_coef_A = find_max_err_and_coef(coefs_a_to_check, scaled_coefficients, precision)
  local max_Cx_vec = math.max(scaled_coefficients.Cx_vec_1, scaled_coefficients.Cx_vec_2, scaled_coefficients.Cx_vec_3)
  if max_err_A > 0.01 then
    return 'Numerical error exceeds 1 percent because a capacitance value is too large.'
  end
  if max_coef_A >= 1 or max_Cx_vec >= 1 then
    return 'Cannot compute numerical solution because a capacitance value is too small.'
  end
  
  return nil
end

function downshift_coefficients(coefficients, inject_v_ac, io_ph2to1, is_partial, ph2_or_ph3)

  local downshift_groups = {}
  if is_partial then
    downshift_groups = {
      dbL1 = {'B1_11', 'A1_11', 'A1_14'},
      dbL2 = {'B1_22', 'A1_22', 'A1_25'},
      dbL3 = {'B1_33', 'A1_33', 'A1_36'},
      dbC1 = {'A1_14', 'Cx_vec_1'},
      dbC2 = {'A1_25', 'Cx_vec_2'},
      dbC3 = {'A1_36', 'Cx_vec_3'}
    }
  else
    downshift_groups = {
      dbL1 = {'B1_11', 'A1_11', 'A1_14', 'B3_11', 'B3_12', 'B3_13', 'A3_11', 'A3_12', 'A3_13', 'A3_14', 'B12_11', 'A12_11', 'A12_13', 'A12_14'},
      dbL2 = {'B1_22', 'A1_22', 'A1_25', 'B3_21', 'B3_22', 'B3_23', 'A3_21', 'A3_22', 'A3_23', 'A3_24', 'B23_22', 'A23_22', 'A23_23', 'A23_24'},
      dbL3 = {'B1_33', 'A1_33', 'A1_36', 'B13_13', 'A13_11', 'A13_13', 'A13_14'},
      dbC1 = {'A1_14', 'A3_13', 'A3_23', 'A12_13', 'A23_23', 'A13_13', 'Cx_vec_1'},
      dbC2 = {'A1_25', 'A3_14', 'A3_24', 'A12_14', 'A23_24', 'A13_14', 'Cx_vec_2'},
      dbC3 = {'A1_36', 'Cx_vec_3'}
    }
  end

  -- If io_ph2to1 is enabled, we set B3_12 <= B3_11 and B3_22 <= B3_21, but since
  -- they're part of the same group, we don't have to do anything special here.
  --
  -- However, we do have to assign the parameters B1_2_22, B12_2_11 and B23_2_22
  -- to the right group.
  if io_ph2to1 then
    table.insert(downshift_groups.dbL1, 'B1_2_22')
    table.insert(downshift_groups.dbL3, 'B12_2_11')
    table.insert(downshift_groups.dbL1, 'B23_2_22')
  elseif not is_partial then
    table.insert(downshift_groups.dbL2, 'B1_2_22')
    table.insert(downshift_groups.dbL1, 'B12_2_11')
    table.insert(downshift_groups.dbL2, 'B23_2_22')
  end

  local downshifts = {}
  for group, parameter_names in pairs(downshift_groups) do
    local shift = 1
    -- Disable downshifting for capacitors
    if string.sub(group, 1, 3) == 'dbC' then
      downshifts[group] = false
    else
      -- Check if all parameters in a group can be multiplied by 10 without their
      -- magnitude exceeding 1.
      downshifts[group] = true
      for _, name in ipairs(parameter_names) do
        if math.abs(coefficients[name][1] * 2^10) >= 1 then
          downshifts[group] = false
          break
        end
      end
      
      if downshifts[group] then
        shift = 2^10
      end
    end

    for _, name in ipairs(parameter_names) do
      coefficients[name] = coefficients[name] * shift
    end
  end

  if io_ph2to1 then
    downshifts.dbL1_2 = downshifts.dbL3
    downshifts.dbL2_2 = downshifts.dbL1
  else
    downshifts.dbL1_2 = downshifts.dbL1
    downshifts.dbL2_2 = downshifts.dbL2
  end

  return downshifts, nil
end

--[[
  The Nanostep solver requires the multiplicative coefficients to be passed as
  integers. However, in the block mask they exis as real numbers.
  
  The scale_nanostep_coefficients function does the conversion by mutliplying
  them by 2^precision and applying some other scalling.
--]]
function BlockUtils.scale_nanostep_coefficients(block, precision, bit_shift, is_partial)
  
  local parameters = block.Mask['nanostepParameters'];
  if not parameters then
    return nil, nil, 'nanostepParameters is not defined.'
  end

  -- If AC voltage injection is enabled, some of the A coefficients were
  -- assigned values from B matrices. Therefore, they should only be bitshifted
  -- by b instead of 2b.
  local inject_v_ac = block.Mask['inject_v_ac'] == 1

  local bitshift_map = {
    A1_11 = bit_shift,
    A1_14 = inject_v_ac and bit_shift or 2*bit_shift,
    A1_22 = bit_shift,
    A1_25 = inject_v_ac and bit_shift or 2*bit_shift,
    A1_33 = bit_shift,
    A1_36 = bit_shift,
    A12_11 = bit_shift,
    A12_13 = inject_v_ac and bit_shift or 2*bit_shift,
    A12_14 = inject_v_ac and bit_shift or 2*bit_shift,
    A23_22 = bit_shift,
    A23_23 = inject_v_ac and bit_shift or 2*bit_shift,
    A23_24 = inject_v_ac and bit_shift or 2*bit_shift,
    A13_11 = bit_shift,
    A13_13 = inject_v_ac and bit_shift or 2*bit_shift,
    A13_14 = inject_v_ac and bit_shift or 2*bit_shift,
    A3_11 = bit_shift,
    A3_12 = bit_shift,
    A3_13 = inject_v_ac and bit_shift or 2*bit_shift,
    A3_14 = inject_v_ac and bit_shift or 2*bit_shift,
    A3_21 = bit_shift,
    A3_22 = bit_shift,
    A3_23 = inject_v_ac and bit_shift or 2*bit_shift,
    A3_24 = inject_v_ac and bit_shift or 2*bit_shift,
    B1_11 = bit_shift,
    B1_22 = bit_shift,
    B1_33 = bit_shift,
    B12_11 = bit_shift,
    B23_22 = bit_shift,
    B13_13 = bit_shift,
    B1_2_22 = bit_shift,
    B12_2_11 = bit_shift,
    B23_2_22 = bit_shift,
    B3_11 = bit_shift,
    B3_12 = bit_shift,
    B3_13 = bit_shift,
    B3_21 = bit_shift,
    B3_22 = bit_shift,
    B3_23 = bit_shift,
    Cx_vec_1 = -bit_shift,
    Cx_vec_2 = -bit_shift,
    Cx_vec_3 = -bit_shift
  }

  -- Get all of the coefficients from block.Mask and bitshift them.
  local shifted_coefficients = {}
  for name, bitshift in pairs(bitshift_map) do
    local float_value = parameters[name]
    if not float_value then
      return nil, nil, '%s missing in nanostepParameters' % {name}
    end
    shifted_coefficients[name] = float_value * 2^bitshift
  end

  local io_ph2to1 = block.Mask['io_ph2to1'] == 1
  local ph2_or_ph3 = block.Mask['twophase'] == 1 or block.Mask['threephase'] == 1
  local downshifts, err = downshift_coefficients(shifted_coefficients, inject_v_ac, io_ph2to1, is_partial, ph2_or_ph3)
  if err then
    return nil, nil, err
  end

  local err = check_nanostep_coefficients(block, shifted_coefficients, precision, is_partial)
  if err then
    return nil, nil, err
  end

  -- Convert the floating point values to integers.
  local scaled_coefficients = {}
  for k, v in pairs(shifted_coefficients) do
    scaled_coefficients[k] = BlockUtils.round(v * 2^precision)
  end

  return scaled_coefficients, downshifts, nil
end

function BlockUtils.getPWMPrescaler(sampleTime, FPGAClkFreq)
  
  -- Make sure sampleTime is not too big. Otherwise we will overflow either the
  -- PWMPrescaler or the PWM cycle counter.
  --
  -- Assuming the PWMPrescaler has its maximum allowed value of 1023, the
  -- PWM cycle counter is incremented every 512 FPGA clocks. Its range is 1 to
  -- 4095. Because the RT Box may extend the simulation step by up to 2 clocks
  -- for synchronization, the base period must be at most:
  local maxSampleTime = (512 * 4095 - 2) * 1/FPGAClkFreq

  if sampleTime > maxSampleTime then
    local roundedDownTo5Digits = math.floor(maxSampleTime * 10e5) / 10e5
    return nil, "Discretization step size must be at most %.4e s." % {roundedDownTo5Digits}
  end
  
  -- The +2 is because the box can spend up to 2 extra clocks per simulation
  -- step for synchronization.
  local maxClocksInSamplePeriod = math.floor(sampleTime * FPGAClkFreq)+2

  -- The PWM cycle counters have a reset value of 1 and a maximum value of 4095
  -- (if we increment them 4095 times they overflow).
  -- They are incremented at a frequency of 2*FPGAClkFreq by default and this
  -- clock needs to be prescaled to avoid the overflow.
  local maxPossibleIncrements = 2*maxClocksInSamplePeriod
  local maxSupportedIncrements = 4095
  local PWMPrescaler = math.floor(maxPossibleIncrements / maxSupportedIncrements)

  return PWMPrescaler, nil
end

return BlockUtils

