-- ModbusUtils.lua
-- 
-- Copyright (c) 2024 Plexim GmbH
-- All rights reserved.

local ModbusUtils = {}
local Utils = require('blocks.BlockUtils')

ModbusUtils.typeSizes = { 1, 1, 1, 1, 2, 2, 2, 4}
ModbusUtils.typeNames = {
  "uint8_t",
  "int8_t",
  "uint16_t",
  "int16_t",
  "uint32_t",
  "int32_t",
  "float",
  "double"
}
ModbusUtils.typeSizes[0] = 1

function ModbusUtils.getModbusAddressToConfigMaps(aConfig)
  local bitAddressToConfigMap = {}
  local regAddressToConfigMap = {}
  for s, c in pairs(aConfig) do
    if c.type == 0 then
      bitAddressToConfigMap[c['start']] = { width = c.width, sig = s}
    else
      regAddressToConfigMap[c['start']] = { width = c.width * ModbusUtils.typeSizes[c.type], sig = s}
    end
  end
  return bitAddressToConfigMap, regAddressToConfigMap
end

function ModbusUtils.getSignalToBufferAddressMap(aModbusAddressToConfigMap)
  local ret = {}
  local offset = 0
  for s, c in Utils.pairsByKeys(aModbusAddressToConfigMap) do
    ret[c.sig] = offset
    offset = offset + c.width
  end
  return ret, offset
end

-- find consecutive address ranges
function ModbusUtils.getRanges(aModbusAddressToConfigMap)
  local ret = {}
  local addr = 0
  local length = 0
  local numEntries = 0
  for start, desc in Utils.pairsByKeys(aModbusAddressToConfigMap) do
    if start ~= addr + length then
      if length > 0 then
        ret[addr] = length
        numEntries = numEntries + 1
      end
      addr = start
      length = desc.width
    else
      length = length + desc.width
    end
  end
  if length > 0 then
    ret[addr] = length
    numEntries = numEntries + 1
  end
  return ret, numEntries
end

-- create bit to input assignment map
function ModbusUtils.getBitToInputMap(aConfig)
  local bitAddressToConfigMap = {}
  for s, c in pairs(aConfig) do
    if c.type == 0 then
      bitAddressToConfigMap[c['start']] = { width = c.width, sig = s}
    else
      break
    end
  end
  local ret = {}
  local addr = 0
  for start, desc in Utils.pairsByKeys(bitAddressToConfigMap) do
    for i = 1, desc.width do
      ret[addr+i] = { sig = desc.sig, idx = i }      
    end
    addr = addr + desc.width
  end
  return ret
end

ModbusUtils.defaultValueCode = function(i, j, c)
  return "%.18g" % c['default']
end

ModbusUtils.inputSignalCode = function(i, j, c)
  return Block.InputSignal[i][j]
end

function ModbusUtils.generateBufferAssignmentCode(aConfig, aSignalToBufferAddressMap, aBufferOffset, valueFcn)
  local littleEndianRegisterOrder = (Block.Parameters['RegisterOrder'] == 1)
  local code = StringList:new()
  for i, c in pairs(aConfig) do
    local t = c.type
    if c.type ~= 0 then -- not bool
      local typeSize = ModbusUtils.typeSizes[t]
      if typeSize > 1 then
        for j = 1, c.width do
          code:append("{\n%s u = %s;\n" % { ModbusUtils.typeNames[t], valueFcn(i, j, c)})
          code:append("uint16_t* pu = (uint16_t*)&u;")
          for l = 1, typeSize do
            code:append("*((modbusBuffer)+%i) = *(pu + %i);\n"  % {
              aSignalToBufferAddressMap[i] + aBufferOffset + j + l - 2,
              littleEndianRegisterOrder and l-1 or typeSize-l
            })
          end
          code:append("}\n")
        end
      else
        for j = 1, c['width'] do
          code:append("*(((%s*)modbusBuffer)+%i) = %s;\n" % {
            ModbusUtils.typeNames[t],
            aSignalToBufferAddressMap[i] + aBufferOffset + j - 1, 
            valueFcn(i, j, c),
          })
        end
      end
    end
  end
  return code
end

function ModbusUtils.generateBitBufferAssignmentCode(aConfig, aBufferOffset, valueFcn)
  local code = StringList:new()
  local bitToInputMap = ModbusUtils.getBitToInputMap(aConfig)
  if next(bitToInputMap) == nil then
    -- bitToInputMap is empty
    return code
  end
  local currentOffset = 0
  code:append("{\n")
  code:append("uint16_t newVal = 0;\n")
  for i, c in pairs(bitToInputMap) do
    local addr = i-1
    local newOffset = addr // 16
    if newOffset ~= currentOffset then
      code:append("*(modbusBuffer+%i) = newVal;\n" % {
        currentOffset + aBufferOffset, 
      })
      code:append("newVal = 0;\n")
      currentOffset = newOffset
    end
    local value = valueFcn(c.sig, c.idx, aConfig[c.sig])
    local numValue = Utils.stringToNumber(value, true)
    if numValue ~= nil and numValue ~= 0 then -- value is constant and not 0
      code:append("newVal |= 0x%x;\n" % {
        2 ^ (addr - newOffset * 16),
      })
    elseif numValue == nil then
      code:append("newVal |= (%s) ? 0x%x : 0;\n" % {
        value,
        2 ^ (addr - newOffset * 16),
      })
    end
  end
  code:append("*(modbusBuffer+%i) = newVal;\n" % {
    currentOffset + aBufferOffset, 
  })
  code:append("}\n")
  return code
end

return ModbusUtils
