local Coder = { }

local FileUtils = require("FileUtils")
local MultiTasking = require("MultiTasking")
local Xcp = require("XCP")
local PwmPreprocReg = require("PwmPreprocReg")
local BlockUtils = require('blocks.BlockUtils')

local MaxNanostepMuxSignals = 3

local Registry = {
  NumDataCaptureBlocks = 0,
  NumProgrammableValueBlocks = 0,
  NumCanRxBlocks = 0,
  NumUdpTxBlocks = 0,
  NumUdpRxBlocks = 0,
  NumToFileBlocks = 0,
  NumModbusBufferEntries = 0,
  ToFileFileNames = { },
  ToFileInstancePaths = { },
  ToFileInfo = { },
  NumWallClocks = 0,
  CanTxIdToBlock = { [1] = { }, [2] = { } },
  CanRxIdToBlock = { [1] = { }, [2] = { } },
  CanRxIds = { [1] = { }, [2] = { } },
  UdpRxPortToBlock = { },
  UdpRxPorts = { },
  UseEthercat = 0,
  BlockInstances = {},
  PowerstageProtectionUnits = { },
  PowerstageProtectedChannels = { [1] = { }, [2] = { }, [3] = { }, [4] = { } },
  DigitalOverrideChannels = { },
  Classes = {},
  PwmTasks = {},
  IoCode = { StringList:new(), StringList:new() },
  RequirePwmBuffer = {},
  NanostepSolvers = {},
  NanostepScopeSignals = {},
  NanostepSelectors = {},
  NanostepSignalSinks = {},
}

local function ListNanostepResourceUsage(msg, blockPaths)
  for idx, solver in ipairs(Registry.NanostepSolvers) do
    if solver.usedPartials ~= 0 then
      msg:append("")
      msg:append("Resource allocation for Nanostep solver %i (connected to %s):" % { idx, solver.usesFlexarrayModel ~= 0 and "FlexArray" or "CPU" } )
      for i = 1, #solver.blocks do
        table.insert(blockPaths, solver.blocks[i].path)
        msg:append("- @%i%s" % { #blockPaths + 1, solver.blocks[i].weight > 1 and " (weight %i)" % solver.blocks[i].weight or "" })
      end
      msg:append("Total weight: %i of 3" % { solver.usedPartials })
    end
  end
end

function Coder.CreateTargetBlock(name, instanceType)
  local TargetBlock = require('blocks.%s' % {name}).getBlock({
    instances = Registry.BlockInstances,
    classes = Registry.Classes
  })(name, instanceType)

  local error = TargetBlock:checkMaskParameters({})
  if error == nil then
    error = TargetBlock:checkTspVersion()
  end
  if error ~= nil then
    local errorFunction = {}
    function errorFunction:getDirectFeedthroughCode()
      return error
    end
    return errorFunction
  else
    return TargetBlock
  end
end

function Coder.GetTargetBlock(bid)
  return Registry.BlockInstances[bid]
end


function Coder.RegisterDataCaptureBlock()
  local ret = Registry.NumDataCaptureBlocks
  Registry.NumDataCaptureBlocks = ret+1
  return ret
end

function Coder.RegisterProgrammableValueBlock()
  local ret = Registry.NumProgrammableValueBlocks
  Registry.NumProgrammableValueBlocks = ret+1
  return ret
end

function Coder.RegisterCanTxBlock(channel, id, blockPath)
  local existingPath = Registry.CanTxIdToBlock[channel][id]
  if (existingPath) then
    Registry.CanTxIdToBlock[channel][id] = existingPath .. ", " .. blockPath
  else
    Registry.CanTxIdToBlock[channel][id] = blockPath
  end
  return existingPath
end

function Coder.RegisterCanRxBlock()
  local numInstance = Registry.NumCanRxBlocks
  Registry.NumCanRxBlocks = numInstance+1
  return numInstance
end

function Coder.RegisterCanRxId(numInstance, channel, id, blockPath, frameType)
  local existingPath = Registry.CanRxIdToBlock[channel][id]
  if (existingPath) then
    Registry.CanRxIdToBlock[channel][id] = existingPath .. ", " .. blockPath
  else
    Registry.CanRxIdToBlock[channel][id] = blockPath
  end
  local idInfo = { id, numInstance }
  if (frameType == 3) then
    idInfo[1] = id | 0x80000000  -- CAN_EFF_FLAG from linux/can.h
  elseif (frameType == 2) then
    idInfo[1] = id & 0x7fffffff
  elseif (id > 2047) then -- auto setting and can id not fitting in 11 bit
    idInfo[1] = id | 0x80000000  -- CAN_EFF_FLAG from linux/can.h
  end
  table.insert(Registry.CanRxIds[channel], idInfo)
  return existingPath
end

function Coder.RegisterModbusDevice(aNumBufferEntriesNeeded)
  local ret = Registry.NumModbusBufferEntries
  Registry.NumModbusBufferEntries = ret + aNumBufferEntriesNeeded
  return ret
end

function Coder.RegisterUdpTxBlock()
  local ret = Registry.NumUdpTxBlocks
  Registry.NumUdpTxBlocks = ret + 1
  return ret
end

function Coder.RegisterUdpRxBlock(port, blockPath, bufLength, dataType)
  local numInstance = Registry.NumUdpRxBlocks
  Registry.NumUdpRxBlocks = numInstance+1
  local existingPath = Registry.UdpRxPortToBlock[port]
  if (existingPath) then
    Registry.UdpRxPortToBlock[port] = existingPath .. ", " .. blockPath
  else
    Registry.UdpRxPortToBlock[port] = blockPath
  end
  local idInfo = { port, numInstance, bufLength, dataType }
  table.insert(Registry.UdpRxPorts, idInfo)
  return numInstance, existingPath
end

function Coder.registerPwmForTask(taskName, pwmChannel)
  if not Registry.PwmTasks[taskName] then
    Registry.PwmTasks[taskName] = {}
  end
  table.insert(Registry.PwmTasks[taskName], pwmChannel)
end

function Coder.RegisterToFileBlock(fileName, blockPath, width, numSamples)
  local numInstance = Registry.NumToFileBlocks
  Registry.NumToFileBlocks = numInstance+1
  Registry.ToFileInstancePaths[numInstance] = blockPath
  local existingPath
  local offset = 0
  for instance, name in pairs(Registry.ToFileFileNames) do
    if (name == fileName) then
      existingPath = Registry.ToFileInstancePaths[instance]
      break
    end
  end
  Registry.ToFileFileNames[numInstance] = fileName
  if (#Registry.ToFileInfo > 0) then
    offset = 2 * Registry.ToFileInfo[numInstance][2] * Registry.ToFileInfo[numInstance][3] + Registry.ToFileInfo[numInstance][4]
  end
  local idInfo = {numInstance, width, numSamples, offset }
  table.insert(Registry.ToFileInfo, idInfo) 
  return numInstance, existingPath
end

function Coder.GetToFileOffset(instance)
  local offset = 0
  local errorMsg
  if (instance > 0) then
    offset = Registry.ToFileInfo[instance+1][4]
  end
  if (offset + 2 * Registry.ToFileInfo[instance+1][2] * Registry.ToFileInfo[instance+1][3] > 32768*2) then
    errorMsg = "memory overflow"
  end
  return offset, errorMsg
end

function Coder.RegisterWallClock()
  local ret = Registry.NumWallClocks
  Registry.NumWallClocks = ret + 1
  return ret
end

function Coder.RegisterEthercat()
  Registry.UseEthercat = 1
end

function Coder.RegisterNanostepSelector(aProbe)
  Registry.NanostepSignalSinks[Block.Path] = 1
  for idx = 1, #Registry.NanostepSelectors do
    if Registry.NanostepSelectors[idx] == aProbe then
      return idx - 1
    end
  end
  local ret = #Registry.NanostepSelectors
  Registry.NanostepSelectors[ret+1] = aProbe
    --print(dump(Registry.NanostepSelectors))
  return ret
end

function Coder.RegisterNanostepScopeSignal(aNanostepProbeType, aNanostepProbe, aMetaGain)
  local ret = 0
  local probeIdx
  local mask = 0
  if aNanostepProbeType == 'analog' then
    ret = Coder.RegisterNanostepSelector(aNanostepProbe)
    probeIdx = ret
  else
    if aNanostepProbe > 23 then
      mask = 1 << (aNanostepProbe - 24)
      probeIdx = MaxNanostepMuxSignals + 1
    else
      mask = 1 << aNanostepProbe
      probeIdx = MaxNanostepMuxSignals   
    end
  end
  table.insert(Registry.NanostepScopeSignals, { aNanostepProbeType, probeIdx, aMetaGain, mask })
  return ret
end

function Coder.getFPGAClk()
  if Target.Name == "PLECS RT Box 1" then
    return 400e6/3.0;
  else
    return 150e6;
  end  
end

local PreprocReg = PwmPreprocReg:new()
function Coder.getPwmPreprocReg()
  return PreprocReg
end

function Coder.CreateTunableParamsFunctionCode()
  local code = StringList:new()
  
  if #Model.ExtModeParameters == 0 then
    code:append("static void copyTunableParameters(const double* aData) { (void)aData; }")
  else
    code:append([=[
static void copyTunableParameters(const double* aData)
{]=])
  
    for idx = 1, #Model.ExtModeParameters do
      local numDims = #Model.ExtModeParameters[idx].Dimensions
      if numDims == 0 then
        code:append("  MODEL_TUNABLE_PARAMETERS.%s = (%s)*aData++;" %
                    {Model.ExtModeParameters[idx].Symbol, Model.ExtModeParameters[idx].OrigType})
      elseif numDims == 1 then
        code:append([=[
  for(int i = 0; i < %i; ++i)
    MODEL_TUNABLE_PARAMETERS.%s[i] = (%s)*aData++;]=] % 
         {Model.ExtModeParameters[idx].Dimensions[1],
          Model.ExtModeParameters[idx].Symbol,
          Model.ExtModeParameters[idx].OrigType})
      else
        code:append([=[
  for(int i = 0; i < %i; ++i)
    for(int j = 0; j < %i; ++j)
      MODEL_TUNABLE_PARAMETERS.%s[i][j] = (%s)*aData++;]=] %
         {Model.ExtModeParameters[idx].Dimensions[1],
          Model.ExtModeParameters[idx].Dimensions[2],
          Model.ExtModeParameters[idx].Symbol,
          Model.ExtModeParameters[idx].OrigType})
      end
    end
    
    code:append("}")
  end
  
  return table.concat(code, '\n')
end

function Coder.RegisterPowerstageProtection(unitIdx)
  table.insert(Registry.PowerstageProtectionUnits, unitIdx)
end

function Coder.UsePowerstageProtection(unitIdx, channel)
  table.insert(Registry.PowerstageProtectedChannels[unitIdx], channel)
end

function Coder.RegisterDigitalOverride(channel)
  table.insert(Registry.DigitalOverrideChannels, channel)
end

function Coder.requirePwmBuffer(aBufferName)
  Registry.RequirePwmBuffer[aBufferName] = true
end

function Coder.addIoCode(core, code)
  Registry.IoCode[core]:append(code)
end

function Coder.isNanostepUsed()
  return Registry.NanostepSolvers[0] ~= nil or Registry.NanostepSolvers[1] ~= nil
end

--[[
  We have one or two Nanostep solvers and each one has three partials. The
  requireNanostepSolver function tries to "allocate" the requested number of
  partials.

  If it succeeds, it returns two numbers which specify which partial of which
  solver has been allocated.

  If it fails, it returns (0, 0).
--]]
function Coder.requireNanostepSolver(requiredPartials, useFlexarrayModel)
  local maxInstances = 1
  if Target.Name ~= "PLECS RT Box 1" then
    maxInstances = 2
  end
  -- make sure registry entries are initialized
  for instance = 1, maxInstances do
    if Registry.NanostepSolvers[instance] == nil then
      Registry.NanostepSolvers[instance] = { usedPartials = 0, usesFlexarrayModel = false, blocks = {} }
    end
  end

  --[[
    RT Box 2 and 3 have two Nanostep solvers, which can exchange data either
    with the FlexArray solver or the CPU. Due to the hardware design, only the
    following configurations are possible:
      - both solvers interact with the FlexArray solver,
      - both solvers interact with the CPU,
      - Solver 0 interacts with the FlexArray solver and Solver 1 with the CPU

    Due to this, if requireNanostepSolver is called with useFlexarrayModel = True
    we should first try to allocate Solver 0. Likewise if
    useFlexarrayModel = False, we should prefer to allocate Solver 1.

    If useFlexarrayModel = False, we reverse the direction in which we iterate
    over the registry array. This will cause Solver 1 to be allocated before
    Solver 0.
  --]]
  local start = 1
  local stop = maxInstances
  local dir = 1
  if useFlexarrayModel == 0 then
    start = maxInstances
    stop = 1
    dir = -1
  end
  
  local path = BlockUtils.splitPathTail(Block.Path, 3)
  local usageInfo = { path = path, weight = requiredPartials }
  
  for solverInstance=start,stop,dir do
    local solver = Registry.NanostepSolvers[solverInstance]
    if solver.usedPartials == 0 then
      solver.usedPartials = requiredPartials
      solver.usesFlexarrayModel = useFlexarrayModel
      local partialInstance = 1
      table.insert(solver.blocks, usageInfo)
      return solverInstance, partialInstance
    elseif solver.usedPartials + requiredPartials <= 3 and solver.usesFlexarrayModel == useFlexarrayModel then
      local partialInstance = solver.usedPartials + 1
      solver.usedPartials = solver.usedPartials + requiredPartials
      table.insert(solver.blocks, usageInfo)
      return solverInstance, partialInstance
    end
  end
  
  local msg = StringList:new()
  local blockPaths = { path }
  msg:append("Cannot fulfill request for Nanostep unit with weight %i connected to %s from @2." % { requiredPartials, useFlexarrayModel ~= 0 and "FlexArray" or "CPU"})
  msg:append("")
  msg:append("Number of available Nanostep solvers on %s: %i" % { Target.Name, maxInstances })
  ListNanostepResourceUsage(msg, blockPaths)
  Target:LogMessage("error", table.concat(msg, "\n"), blockPaths)

  return 0, 0
end

function Coder.Initialize()
  local Resources = ResourceList:new()
  local Include = StringList:new()
  local Declarations = {}
  local PreInitCode = StringList:new()
  local PostInitCode = {}
  local TerminateCode = {}
  
  if Target.Variables.SAMPLE_TIME == "NaN" then
    return "Invalid discretization step size."
  end
  
  if Target.Name == "PLECS RT Box 2" then
    Resources:add("Resolver in", 1, 1)
    Resources:add("Resolver out", 1, 1)
    Resources:add("Sigma-delta modulator", 1, 2)
    Resources:add("SFP output", 0, 7)
    Resources:add("SFP input", 0, 7)
    Resources:add("Ethercat", 1, 1)
    Resources:add("EnDat", 1, 1)
  end
  if Target.Name == "PLECS RT Box 3" then
    Resources:add("Analog input", 0, 31)
    Resources:add("Analog output", 0, 31)
    Resources:add("Digital input", 0, 63)
    Resources:add("Digital output", 0, 63)
    Resources:add("Digital override", 0, 63)
    Resources:add("Resolver in", 1, 2)
    Resources:add("Resolver out", 1, 2)
    Resources:add("Sigma-delta modulator", 1, 2)
    Resources:add("SFP output", 0, 7)
    Resources:add("SFP input", 0, 7)
    Resources:add("Ethercat", 1, 1)
    Resources:add("EnDat", 1, 1)
  elseif Target.Name == "PLECS RT Box 4" then
    Resources:add("Analog input", 0, 15)
    Resources:add("Analog output", 0, 15)
    Resources:add("Nanostep output", 0, 5)
    Resources:add("Digital input", 0, 31)
    Resources:add("Digital output", 0, 31)
    Resources:add("Digital override", 0, 31)
    Resources:add("Resolver in", 1, 1)
    Resources:add("Resolver out", 1, 1)
    Resources:add("Sigma-delta modulator", 1, 2)
    Resources:add("SFP output", 0, 7)
    Resources:add("SFP input", 0, 7)
    Resources:add("Ethercat", 1, 1)
    Resources:add("EnDat", 1, 1)
  else
    Resources:add("Analog input", 0, 15)
    Resources:add("Analog output", 0, 15)
    Resources:add("Digital input", 0, 31)
    Resources:add("Digital output", 0, 31)
    Resources:add("Digital override", 0, 31)
    Resources:add("SFP output", 0, 3)
    Resources:add("SFP input", 0, 3)
  end
  Resources:add("NanostepScope", 1, 1)
  Resources:add("Incremental encoder", 1, 2)
  Resources:add("Encoder counter", 0, 1)
  Resources:add("SPI", 1, 2)
  Resources:add("Powerstage protection", 1, 4)

  Include:append("plexim/hw_wrapper.h")
  Include:append("plexim/DigitalOut.h")
  Include:append("plexim/DigitalIn.h")
  Include:append("plexim/AnalogOut.h")
  if Target.Name == "PLECS RT Box 4" then
    Include:append("plexim/NanostepOut.h")
  end
  Include:append("plexim/AnalogIn.h")
  Include:append("plexim/SFP.h")
  Include:append("plexim/HIL_Framework.h")
  Include:append("plexim/DigitalOverride.h")
  
  -- example of a multi-line string definition with variable interpolation
  PreInitCode:append([=[
plxSetAnalogInputVoltage(%(analogInputRange)i);
plxSetupDACs(%(analogOutputRange)i);
plxSetDigitalOutVoltage(%(digitalOutputLevel)i);
plxInitDigitalOut();
initPWMCapture();
plxInitDigitalOverride();
]=] 
  % {
    analogInputRange = Target.Variables.analogInputRange-1,
    analogOutputRange = Target.Variables.analogOutputRange-1,
    digitalOutputLevel = Target.Variables.digitalOutputLevel-1
  })

  if Target.Name == "PLECS RT Box 4" then
    local span;
    if Target.Variables.nanostepDSubOutputRange == 2 then
      span = "DAC_SPAN_BIPOLAR"
    else
      span = "DAC_SPAN_UNIPOLAR"
    end
    PreInitCode:append("plxSetupNanostepDACs(%s);\n" % { span })
  end
  
  if Target.Name == "PLECS RT Box 1" then
    MaxNanostepMuxSignals = 3
  else
    MaxNanostepMuxSignals = 6
  end
  
  if Target.Variables.analogSampling == 1 then
    PreInitCode:append("plxSetupAnalogSampling(1, 0, %.17e);\n" % {Target.Variables.SAMPLE_TIME})
  else
    local samplingDelay = Target.Variables.samplingDelay
    if samplingDelay ~= samplingDelay or samplingDelay < 0 then
      return "Sampling delay must be positive."
    end
    PreInitCode:append("plxSetupAnalogSampling(0, %i, %.17e);\n"
    % {
      math.floor((samplingDelay+1/(2*Coder.getFPGAClk()))*Coder.getFPGAClk()),
      Target.Variables.SAMPLE_TIME
    })
  end
  
  if (Target.Variables.max_overruns ~= math.floor(Target.Variables.max_overruns)
      or Target.Variables.max_overruns < 0) then
    return "Parameter 'Overrun limit' must be a positive integer."
  end
  PreInitCode:append("plxSetMaxNumConsecutiveOverruns(%d);\n" % { Target.Variables.max_overruns })

  local sfpA = Target.Variables.startupSFP_A == 1
  local sfpB = Target.Variables.startupSFP_B == 1
  local sfpC = Target.Variables.startupSFP_C == 1
  local sfpD = Target.Variables.startupSFP_D == 1
  local master = Target.Variables.masterSFP
  local syncTime = Target.Variables.synchronizeTime == 1
  local syncStartupMaster = ((master == 2) and sfpA) or ((master == 3) and sfpB) or ((master == 4) and sfpC) or ((master == 5) and sfpD)
  if Target.Name ~= "PLECS RT Box 1" then
    local sfpE = Target.Variables.startupSFP_E == 1
    local sfpF = Target.Variables.startupSFP_F == 1
    local sfpG = Target.Variables.startupSFP_G == 1
    local sfpH = Target.Variables.startupSFP_H == 1
    syncStartupMaster = syncStartupMaster or ((master == 6) and sfpE) or ((master == 7) and sfpF) or ((master == 8) and sfpG) or ((master == 9) and sfpH)
    PreInitCode:append("plxSetupSFPSyncMaster(%i, %i, %i, %i, %i, %i, %i, %i);\n" %
      { sfpA and (master ~= 2), sfpB and (master ~= 3), sfpC and (master ~= 4), sfpD and (master ~= 5),
        sfpE and (master ~= 6), sfpF and (master ~= 7), sfpG and (master ~= 8), sfpH and (master ~= 9) })
  else
    PreInitCode:append("plxSetupSFPSyncMaster(%i, %i, %i, %i);\n" %
      { sfpA and (master ~= 2), sfpB and (master ~= 3), sfpC and (master ~= 4), sfpD and (master ~= 5) })
  end
  PreInitCode:append("plxSetupSFPSyncSlave(%i, %i, %i);\n" % {master-1, syncTime, syncStartupMaster});
  if (syncTime and master < 2) then
    return "Cannot use master clock with no selected master."
  end
  
  local can1Enable = Target.Variables.can1Enable
  local can2Enable = Target.Variables.can2Enable
  local can1RecoveryDelay = Target.Variables.can1RecoveryDelay
  local can2RecoveryDelay = Target.Variables.can2RecoveryDelay
  if ((can1Enable == 1) or (can2Enable == 1)) then
    Include:append("plexim/Can.h")
  end
  if (can1Enable == 1) then
    local baud = Target.Variables.can1Baud
    if baud ~= baud or baud < 0 or baud > 1000000 then
      return "Baud rate must be a positive value < 1'000'000 bps."
    end
    if Target.Name ~= "PLECS RT Box 1" then
      PreInitCode:append("setupCan(0, %i, %i, %i);\n" % { baud, Target.Variables.can1Termination-1, can1RecoveryDelay })
    else
      PreInitCode:append("setupCan(0, %i, %i, %i, %i);\n" % { baud, Target.Variables.can1TxRx, Target.Variables.can1Termination-1, can1RecoveryDelay })
    end
  end
  if (can2Enable == 1) then
    local baud = Target.Variables.can2Baud
    if baud ~= baud or baud < 0 or baud > 1000000 then
      return "Baud rate must be a positive value < 1'000'000 bps."
    end
    if Target.Name ~= "PLECS RT Box 1" then
      PreInitCode:append("setupCan(1, %i, %i, %i);\n" % { baud, Target.Variables.can2Termination-1, can2RecoveryDelay })
    else
      PreInitCode:append("setupCan(1, %i, %i, %i, %i);\n" % { baud, Target.Variables.can2TxRx, Target.Variables.can2Termination-1, can2RecoveryDelay })
    end
  end

  if Target.Name ~= "PLECS RT Box 1" then
    local status, msg = MultiTasking.CheckTaskConfiguration(Model.TaskConfiguration)
    if status then
      return msg
    end
  end

  return {
     Declarations = {},
     PreInitCode = PreInitCode,
     PostInitCode = {},
     TerminateCode = {},
     Include = Include,
     Resources = Resources,
  }
end


function Coder.Finalize()
  if Coder.isNanostepUsed() then
    local msg = StringList:new()
    local blockPaths = { }
    msg:append("Nanostep solver usage on %s" % { Target.Name })
    ListNanostepResourceUsage(msg, blockPaths)
    Target:LogMessage("info", table.concat(msg, "\n"), blockPaths)
  end

  if #Registry.NanostepSelectors > MaxNanostepMuxSignals then
    local blockPaths = { }
    for path in pairs(Registry.NanostepSignalSinks) do
      table.insert(blockPaths, path)
    end
    local blockMsg = ""
    for idx = 1, #blockPaths do
      blockMsg = blockMsg .. "\n@%i" % { idx + 1 }
    end
    local msg = "This target supports a maximum of %d Nanostep signals (excluding gate signals).\n\n" % { MaxNanostepMuxSignals }
    msg = msg .. "The following blocks use %d signals in total:%s" % { #Registry.NanostepSelectors, blockMsg } 
    Target:LogMessage("error", msg, blockPaths )
  end


  local Include = StringList:new()
  local Declarations = StringList:new()
  local PreInitCode = {}
  local PostInitCode = StringList:new()
  local TerminateCode = {}
  local preBaseStepCode = StringList:new()
  local postBaseStepCode = StringList:new()

  Include:append("plexim/DataCapture.h")
  if (Registry.NumDataCaptureBlocks > 0) then
    Declarations:append("struct PlxDataCaptureRegistry plxDataCaptureRegistry[%i];\n" % { Registry.NumDataCaptureBlocks })
  else
    Declarations:append("struct PlxDataCaptureRegistry plxDataCaptureRegistry[1];\n")
  end
  Declarations:append("const int plxUseEthercat = %i;\n" % Registry.UseEthercat)

  Include:append("plexim/ProgrammableValue.h")
  if (Registry.NumProgrammableValueBlocks > 0) then
    Declarations:append("struct PlxProgrammableValueRegistry plxProgrammableValueRegistry[%i];\n" % { Registry.NumProgrammableValueBlocks })
  else
    Declarations:append("struct PlxProgrammableValueRegistry plxProgrammableValueRegistry[1];\n")
  end

  Include:append("plexim/ToFile.h")
  if (Registry.NumToFileBlocks > 0) then
    Declarations:append("struct PlxToFileRegistry plxToFileRegistry[%i];\n" % { Registry.NumToFileBlocks })
  else
    Declarations:append("struct PlxToFileRegistry plxToFileRegistry[1];\n" )
  end
  
  local ioDeclarations = ""
  for bufferName in pairs(Registry.RequirePwmBuffer) do
    if Target.Name == "PLECS RT Box 3" then
      Declarations:append("uint16_t %s[64];" % { bufferName });
    else
      Declarations:append("uint16_t %s[32];" % { bufferName });
    end
    ioDeclarations = ioDeclarations .. "extern uint16_t %s[];\n" % { bufferName }
  end
  
  for idx = 0, Registry.NumCanRxBlocks-1 do
    Declarations:append("static unsigned char canRxDoubleBuffer%i[2][16];\n" % { idx } )
    Declarations:append("static unsigned char* canRxBufferPtr%i = canRxDoubleBuffer%i[0];\n" % { idx, idx } )
  end;
  if (Registry.NumCanRxBlocks > 0) then
    Declarations:append("int canRxCurrentReadBufferIdx[%i];\n" % { Registry.NumCanRxBlocks } )    
    PostInitCode:append("memset(canRxCurrentReadBufferIdx, 0, sizeof(canRxCurrentReadBufferIdx));\n")
  end
  
  for canCh = 1, 2 do
    if (#Registry.CanRxIds[canCh] > 0) then
      PostInitCode:append("/* Can %i initialization */\n" % { canCh-1 } )
      PostInitCode:append("{\n")
      PostInitCode:append("#pragma pack(push, 4)\n")
      PostInitCode:append("static const struct MyCanRequestIdMsg%i {\n" % { canCh-1 })
      PostInitCode:append("uint32_t mMsg;\n")
      PostInitCode:append("uint32_t mMsgLength;\n")
      PostInitCode:append("uint8_t  mModuleId;\n")
      PostInitCode:append("uint32_t mNumCanIds;\n")
      PostInitCode:append("uint32_t mCanIds[%i];\n" % { #Registry.CanRxIds[canCh] })
      PostInitCode:append("} canRequestMsg = {\n")
      PostInitCode:append(".mMsgLength = %i,\n" % { 2*4 + #Registry.CanRxIds[canCh] * 4} )
      PostInitCode:append(".mModuleId = %i,\n" % { canCh-1 } )
      PostInitCode:append(".mNumCanIds = %i,\n" % { #Registry.CanRxIds[canCh] } )
      PostInitCode:append(".mCanIds = {\n")
      PostInitCode:append("%i" % { Registry.CanRxIds[canCh][1][1] } )
      for idx = 2, #Registry.CanRxIds[canCh] do
        PostInitCode:append(", %i" % { Registry.CanRxIds[canCh][idx][1] } )
      end
      PostInitCode:append("\n}\n")
      PostInitCode:append("};\n")   
      PostInitCode:append("#pragma pack(pop)\n")
      
      PostInitCode:append("static unsigned char* canRxBuffers[%i] = {\n" % { #Registry.CanRxIds[canCh] } )
      PostInitCode:append("(unsigned char*)canRxDoubleBuffer%i" % { Registry.CanRxIds[canCh][1][2] } )
      for idx = 2, #Registry.CanRxIds[canCh] do
        PostInitCode:append(", (unsigned char*)canRxDoubleBuffer%i" % { Registry.CanRxIds[canCh][idx][2] } )
      end
      PostInitCode:append("\n};\n")
      PostInitCode:append("static int* canRxCurrentReadBufferIdzs[%i] = {\n" % { #Registry.CanRxIds[canCh] } )
      PostInitCode:append("&canRxCurrentReadBufferIdx[%i]" % { Registry.CanRxIds[canCh][1][2] } )
      for idx = 2, #Registry.CanRxIds[canCh] do
        PostInitCode:append(", &canRxCurrentReadBufferIdx[%i]" % { Registry.CanRxIds[canCh][idx][2] } )
      end
      PostInitCode:append("\n};\n")
      
      PostInitCode:append("plxRegisterCanRxIds(%i, (struct CanRequestIdMsg*)&canRequestMsg, canRxBuffers, canRxCurrentReadBufferIdzs);\n" % { 
        canCh-1
      } )
      PostInitCode:append("}\n")
    end
  end

  if Registry.NumUdpTxBlocks > 0 then
    PostInitCode:append("setupUdpTx();\n")
  end
  
  if (#Registry.UdpRxPorts > 0) then
    PostInitCode:append("/* UDP initialization */\n")
    PostInitCode:append("{")
    PostInitCode:append("static const uint16_t udpRxBufferLengths[%i] = {\n" % { #Registry.UdpRxPorts } )
    PostInitCode:append("%i*sizeof(%s)" % { Registry.UdpRxPorts[1][3], Registry.UdpRxPorts[1][4] } )
    for idx = 2, #Registry.UdpRxPorts do
      PostInitCode:append(", %i*sizeof(%s)" % { Registry.UdpRxPorts[idx][3], Registry.UdpRxPorts[idx][4] } )
    end
    PostInitCode:append("\n};\n")
    PostInitCode:append("#pragma pack(push, 4)\n")
    PostInitCode:append("static const struct MyUdpRequestPortsMsg {\n")
    PostInitCode:append("uint32_t mMsgType;\n")
    PostInitCode:append("const uint32_t mMsgLength;\n")
    PostInitCode:append("const uint32_t mNumPorts;\n")
    PostInitCode:append("const uint16_t udpRxPorts[%i];\n" % { #Registry.UdpRxPorts } )
    PostInitCode:append("} udpRequestPortsMsg = {\n")
    PostInitCode:append(" .mMsgLength = sizeof(uint32_t) + %i * sizeof(uint16_t),\n" % { #Registry.UdpRxPorts } )
    PostInitCode:append(" .mNumPorts = %i,\n" % { #Registry.UdpRxPorts } )
    PostInitCode:append(" .udpRxPorts = { %i" %  { Registry.UdpRxPorts[1][1] } )
    for idx = 2, #Registry.UdpRxPorts do
      PostInitCode:append(", %i" % { Registry.UdpRxPorts[idx][1] } )
    end
    PostInitCode:append("\n}\n")
    PostInitCode:append("};\n")
    PostInitCode:append("#pragma pack(pop)\n")    
    for idx = 1, #Registry.UdpRxPorts do
      Declarations:append("static %s udpRxDoubleBuffer%i[2][%i];\n" % { Registry.UdpRxPorts[idx][4], idx-1, Registry.UdpRxPorts[idx][3] } )
      Declarations:append("static %s* udpRxBufferPtr%i = udpRxDoubleBuffer%i[0];\n" % { Registry.UdpRxPorts[idx][4], idx-1, idx-1 } )
      PostInitCode:append("memset(udpRxDoubleBuffer%i, 0, 2*udpRxBufferLengths[%i]);\n" % { idx-1, idx-1 } )
    end
    Declarations:append("unsigned char udpRxUpdateFlags[2][%i];" % { #Registry.UdpRxPorts } )
    PostInitCode:append("memset(udpRxUpdateFlags, 0, 2*%i*sizeof(char));\n" % {  #Registry.UdpRxPorts } )
    Declarations:append("int udpRxCurrentReadBufferIdx[%i];\n" % { #Registry.UdpRxPorts } )
    PostInitCode:append("memset(udpRxCurrentReadBufferIdx, 0, sizeof(udpRxCurrentReadBufferIdx));\n")

    PostInitCode:append("static void* udpRxBuffers[%i] = {\n" % { #Registry.UdpRxPorts } )
    PostInitCode:append("udpRxDoubleBuffer%i" % { Registry.UdpRxPorts[1][2] } )
    for idx = 2, #Registry.UdpRxPorts do
      PostInitCode:append(", udpRxDoubleBuffer%i" % { Registry.UdpRxPorts[idx][2] } )
    end
    PostInitCode:append("\n};\n")
    PostInitCode:append("plxRegisterUdpRxPorts((struct UdpRequestPortsMsg*)&udpRequestPortsMsg, udpRxBuffers, udpRxBufferLengths, (unsigned char*)udpRxUpdateFlags, udpRxCurrentReadBufferIdx);\n")
    PostInitCode:append("}")
  end
  
  postBaseStepCode:append("   plxSyncDigitalOuts();")

  if (#Registry.DigitalOverrideChannels > 0) then
    postBaseStepCode:append("   plxSyncDigitalOverrides();")
  end

  if Target.Variables.simulationMode == 2 then -- RCP mode
    if #Model.Tasks > 1 then
      return "RCP mode cannot be used when multitasking is enabled."
    end
    PostInitCode:append("plxEnableRcpMode();")
    postBaseStepCode:append("   plxToggleDmaFlag();")
    preBaseStepCode:append("   plxUpdatePWMCycleRate();")
  else
    preBaseStepCode:append("   plxSetBufferAdresses();")
  end

  if Registry.Classes['SPI'] then
    preBaseStepCode:append("   plxUpdateSPIBuffer0();")
    if Registry.Classes.SPI.instanceCounter > 1 then
      preBaseStepCode:append("   plxUpdateSPIBuffer1();")
    end
  end

  if Registry.Classes['IncrementalEncoder'] then
    preBaseStepCode:append("   plxEnableEncOutput(0);")
    if Registry.Classes.IncrementalEncoder.instanceCounter > 1 then
      preBaseStepCode:append("   plxEnableEncOutput(1);")
    end
  end

  local a2lDefinitions = Xcp.getEmptyA2lDefinitions()
  if Target.Variables.enableXCP == "1" then
    Xcp.generateA2l()
    a2lDefinitions = Xcp.getA2lDefinitions()
    postBaseStepCode:append("   plxXcpEvent();")
  end

  MultiTasking.AnalyzeTasks(Model.Tasks)
  MultiTasking.CalculateTicks(Target.Variables.SAMPLE_TIME)
  if MultiTasking.NumTasksOnCore(1) == 1 then
    postBaseStepCode:append("   plxPostBaseStep();")
  end

  local coreTimingCode = MultiTasking.CreateCoreTimingCode()
  local initFunctionCode = MultiTasking.CreateInitFunctionCode(Registry.PwmTasks)
  local syncFunctionCode = MultiTasking.CreateSyncFunctionCode()
  local stepFunctionCode = MultiTasking.CreateStepFunctionCode(preBaseStepCode,
                                                               postBaseStepCode,
                                                               Registry.PwmTasks,
                                                               Registry.IoCode,
                                                               ioDeclarations)
  local tunableParamsFunctionCode = Coder.CreateTunableParamsFunctionCode()
  local nanostepScopeSignals = StringList:new()
  nanostepScopeSignals:append("struct PlxNanostepSignal plxNanostepSignals[] = {")
  for idx = 1, #Registry.NanostepScopeSignals do
    nanostepScopeSignals:append("   { %i, %i, %.9g, 0x%0x }," % {
      Registry.NanostepScopeSignals[idx][1] == 'analog' and 0 or 1,
      tonumber(Registry.NanostepScopeSignals[idx][2]),
      tonumber(Registry.NanostepScopeSignals[idx][3]),
      tonumber(Registry.NanostepScopeSignals[idx][4]),
    })
  end
  nanostepScopeSignals:append("};")

  for idx = 1, #Registry.NanostepSelectors do
    local nanostepProbe = Registry.NanostepSelectors[idx]
    PostInitCode:append("plxSelectNanostepSignal(%i, %i);\n" % { idx-1, nanostepProbe})
  end
 
  -- Copy template file main.c and set USE_MULTICORE flag depending on
  -- the number of tasks.
  local templateFile;
  if Target.Name == "PLECS RT Box 1" then
    templateFile = FileUtils.FullPath({Target.Variables.TARGET_ROOT, "src", "main.c"})
  else
    templateFile = FileUtils.FullPath({Target.Variables.TARGET_ROOT, "src", "main_rtbox2.c"})
  end;
  local outputFile = FileUtils.FullPath({Target.Variables.BUILD_ROOT, Target.Variables.BASE_NAME .. "_main.c"})
  local dictionary = {
    ["|>BASE_NAME<|"] = Target.Variables.BASE_NAME,
    ["|>MULTITASKING<|"] = #Model.Tasks > 1 and "1" or "0",
    ["|>CORETIMING<|"] = coreTimingCode,
    ["|>INITFUNCTION<|"] = initFunctionCode:gsub("%%", "%%%%"),
    ["|>SYNCFUNCTION<|"] = syncFunctionCode:gsub("%%", "%%%%"),
    ["|>STEPFUNCTIONS<|"] = stepFunctionCode:gsub("%%", "%%%%"),
    ["|>HAVE_NON_BASE_PERIOD<|"] = MultiTasking.HaveNonBasePeriodTask() and "1" or "0",
    ["|>NUM_TASKS_CORE_1<|"] = MultiTasking.NumTasksOnCore(1),
    ["|>TUNABLEPARAMSFUNCTION<|"] = tunableParamsFunctionCode:gsub("%%", "%%%%"),
    ["|>A2LDEFINITIONS<|"] = a2lDefinitions,
    ["|>NUMNANOSTEPSCOPESIGNALS<|"] = #Registry.NanostepScopeSignals,
    ["|>NANOSTEPSCOPESIGNALS<|"] = table.concat(nanostepScopeSignals, "\n"),
    ["|>MAX_PERIOD_TICKS<|"] = MultiTasking.GetMaxPeriodTicks(),
  }
  if not FileUtils.CopyTemplateFile(templateFile, outputFile, dictionary) then
    return FileUtils.Error;
  end

  -- Make sure that a digital override is not being applied to a PWM out that is
  -- protected by Powerstage Protection.
  for _,unitIdx in pairs(Registry.PowerstageProtectionUnits) do
    for _,protectedChannel in pairs(Registry.PowerstageProtectedChannels[unitIdx]) do
      for _,overriddenChannel in pairs(Registry.DigitalOverrideChannels) do
        -- If a channel is protected by a Powerstage Protection block, then it cannot
        -- be used by a Digital Override block.
        if protectedChannel == overriddenChannel then
          return "ERROR: Attempting to override the PWM signal on digital output channel " ..
                 "%i. Channel is protected by Powerstage Protection unit %i." % { protectedChannel, unitIdx }
        end
      end
    end
  end
  
  -- If PWM capture blocks are used, determine the value of the their clock
  -- prescaler. If it's too large, raise an error.
  if Registry.Classes.PWMCapture then
    local PWMPrescaler, errMsg = BlockUtils.getPWMPrescaler(Target.Variables.SAMPLE_TIME, Coder.getFPGAClk())
    if errMsg then
      return errMsg
    else
      PostInitCode:append("plxSetPWMPrescaler(%d);" % {PWMPrescaler})
    end
  end

  PostInitCode:append(PreprocReg:generateAssertionCode())

  return {
     Declarations = Declarations,
     PreInitCode = {},
     PostInitCode = PostInitCode,
     TerminateCode = {},
     Include = Include,
  }
end

function Coder.GetInstructionOverrides()
  return require('InstructionOverrides')
end

return Coder
