#ifndef FLEXARRAY_H
#define FLEXARRAY_H

#include <stdint.h>
#include "plexim/hw_wrapper.h"

#ifdef __cplusplus
extern "C"
{
#endif

enum FlexarrayDiscretizationMethod
{
   FLEXARRAY_DISCRETIZATION_METHOD_TUSTIN,
   FLEXARRAY_DISCRETIZATION_METHOD_RADAU
};

enum FlexarrayDcsConductionState
{
   FLEXARRAY_DCS_CONDSTATE_OPEN,
   FLEXARRAY_DCS_CONDSTATE_UPPER,
   FLEXARRAY_DCS_CONDSTATE_LOWER
};

enum FlexarrayDcsGateValue
{
   FLEXARRAY_DCS_GATE_VALUE_STATIC_0,
   FLEXARRAY_DCS_GATE_VALUE_STATIC_1,
   FLEXARRAY_DCS_GATE_VALUE_DYNAMIC
};

enum FlexarrayDcsGateLogic
{
   FLEXARRAY_DCS_GATE_LOGIC_DIODE,
   FLEXARRAY_DCS_GATE_LOGIC_THYRISTOR,
   FLEXARRAY_DCS_GATE_LOGIC_IGCT
};

struct FlexarrayModelMatrix
{
   const int* rowptr;
   const int* colidx;
   const double* data;
};

struct FlexarrayModel
{
   int nx;
   int nu;
   int ny;
   int ndcs;
   int nuhb;
   int ny_ext;
   const int* y_ext_indices;

   const struct FlexarrayModelMatrix* mat_A;
   const struct FlexarrayModelMatrix* mat_B;
   const struct FlexarrayModelMatrix* mat_C;
   const struct FlexarrayModelMatrix* mat_D;
   const struct FlexarrayModelMatrix* mat_I;
   
   enum FlexarrayDiscretizationMethod discretization_method;

   int step_ratio;

   const float* x_init;

   struct
   {
      double resistance;

      const int* u_indices;
      const int* y_indices;

      const enum FlexarrayDcsConductionState* initial_conduction_state;

      const enum FlexarrayDcsGateLogic* gate_logic;
      const enum FlexarrayDcsGateValue* upper_gate_values;
      const enum FlexarrayDcsGateValue* lower_gate_values;

      const double* static_upper_voltages;
      const double* static_lower_voltages;

      const double* resonator_freq;
   } dcs;

   struct
   {
      const int* dcs_indices;
      const int* widths;

      const int* voltmeters;
      const int* current_sources;
      const int* upper_pwms;
      const int* lower_pwms;
      const int* upper_bias;
      const int* lower_bias;
   } uhb;
};


void flexarraySetup(const struct FlexarrayModel* aModel);

extern float flexarrayOutputScaleFactor;
#define flexarrayGetOutput(i) \
   (*((float *)plxIoIn.FlexarrayOutput + i) * flexarrayOutputScaleFactor)

int flexarrayGetConductionState(int i);

#define flexarraySetInput(i, u) \
   *((float *)plxIoOut.FlexarrayInput + i) = u

void flexarraySetDcsUpperGate(int i, double g_up);
void flexarraySetDcsLowerGate(int i, double g_lo);

void flexarraySetupOutputToNanostep(uint32_t aNanostepIndex, uint32_t aMeterIdx1, uint32_t aMeterIdx2);
void flexarraySetupInputFromNanostep(uint32_t aNanostepIndex, uint32_t aNanostepMeterIndex, uint32_t aInputIdx);

#ifdef __cplusplus
}
#endif

#endif // FLEXARRAY_H
