/*
   Copyright (c) 2021 by Plexim GmbH
   All rights reserved.

   A free license is granted to anyone to use this software for any legal
   non safety-critical purpose, including commercial applications, provided
   that:
   1) IT IS NOT USED TO DIRECTLY OR INDIRECTLY COMPETE WITH PLEXIM, and
   2) THIS COPYRIGHT NOTICE IS PRESERVED in its entirety.

   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
   OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
   SOFTWARE.
 */

#ifndef UART_H_
#define UART_H_

#include "hw_wrapper.h"
#include <stdbool.h>

/**
 * The RT Box contains two UARTS that may be used for RS232, RS422 and RS485 communication.
 * Each UART has two FIFOs of 64 bytes each for sending and receiving.
 */

typedef enum {
   PLX_UART_1,
   PLX_UART_2,
} PlxUartChannel;

typedef enum {
   PLX_UART_8_BITS,
   PLX_UART_7_BITS,
   PLX_UART_6_BITS
} PlxUartBitSize;

typedef enum {
   PLX_UART_RS232,
   PLX_UART_RS422,
   PLX_UART_RS485 // not supported on hardware rev. 1.1
} PlxUartType;

typedef enum {
   PLX_UART_EVEN_PARITY,
   PLX_UART_ODD_PARITY,
   PLX_UART_SPACE_PARITY,
   PLX_UART_MARK_PARITY,
   PLX_UART_NO_PARITY
} PlxUartParity;

typedef enum {
   PLX_UART_1_STOP_BIT,
   PLX_UART_1_5_STOP_BIT,
   PLX_UART_2_STOP_BIT
} PlxUartStopBits;

typedef enum {
   PLX_UART_TERMINATION_DISABLE,
   PLX_UART_TERMINATION_ENABLE
} PlxUartTermination;

PlxStatus plxUartInit(PlxUartChannel aChannel, PlxUartType aType, uint32_t aBaudrate, PlxUartBitSize aBitSize,
                      PlxUartParity aParity, PlxUartStopBits aStopBits, PlxUartTermination aTermination);

/**
 * @brief plxUartSend sends aDataSize bytes from the buffer aData.
 * @param aChannel  The UART to use.
 * @param aData     The data bufffer containing the data to send.
 * @param aDataSize The number of bytes in the data buffer to send.
 * @return The number of bytes actually sent.
 */
uint32_t plxUartSend(PlxUartChannel aChannel, uint8_t* aData, uint32_t aDataSize);

/**
 * @brief plxUartReceive attempts to read aNumBytes bytes of received data from the receive FIFO.
 * @param aChannel   The UART to use.
 * @param aData      The data buffer to place the data in.
 * @param aDataSize  The number of bytes to read.
 * @return The number of bytes actually read from the receive FIFO.
 */
uint32_t plxUartReceive(PlxUartChannel aChannel, uint8_t* aData, uint32_t aNumBytes);

/**
 * @brief plxUartSetFlowDelay enables automatic flow control. RTS is deactivated when the number
 * of used bytes in the receive FIFO is higher than or equal to aDelay. RTS is activated again if the number
 * of used bytes is four less than aDelay.
 * @param aChannel The UART to use.
 * @param aDelay   The used number of bytes in the receiving FIFO when RTS is deactivated. If aDelay is
 *                 smaller than 4, automatic flow control is deactivated.
 */
void plxUartSetFlowDelay(PlxUartChannel aChannel, uint8_t aDelay);
uint8_t plxUartGetFlowDelay(PlxUartChannel aChannel);

bool plxUartGetCTS(PlxUartChannel aChannel);
bool plxUartIsSending(PlxUartChannel aChannel);

#endif
