/*
   Copyright (c) 2016 by Plexim GmbH
   All rights reserved.

   A free license is granted to anyone to use this software for any legal
   non safety-critical purpose, including commercial applications, provided
   that:
   1) IT IS NOT USED TO DIRECTLY OR INDIRECTLY COMPETE WITH PLEXIM, and
   2) THIS COPYRIGHT NOTICE IS PRESERVED in its entirety.

   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
   OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
   SOFTWARE.
 */

#ifndef DIGITALIN_H_
#define DIGITALIN_H_

#include "plexim/hw_wrapper.h"

#ifdef __cplusplus
extern "C"
{
#endif

/**
* @brief Returns the actual state of digital input n.
 * 
 *	
 * @param n ID of digital input.
 */
#define getDigitalIn(n) ((*plxIoIn.DigitalInput >> (n)) & 1)

/**
* @brief Returns the percentage the digital input n persisted in logically true state during the previous interval.
 * 
 *	
 * @param n ID of digital input. 
 */
#define plxGetDigitalInAveraged(n) ((*(plxIoIn.ActiveCountPWM+(n)) & 0x0fff)*plxPWMCycleRate)


/**
* @brief Returns the number of rising edges that occured at digital input n during the previous interval.
 * 
 *	This can be used to calculate switching losses. 
 *
 * @param n ID of digital input.
 */
#define plxGetRisingEdges(n) (*(plxIoIn.ActiveCountPWM+(n)) >> 12)

/**
* @brief Returns the number of falling edges that occured at digital input n during the previous interval.
 *
 *	This can be used to calculate switching losses.
 *
 * @param n ID of digital input.
 */
#define plxGetFallingEdges(n) ((n & 0x01) ? (*(plxIoIn.PWMFallingEdges+(n/2)) >> 4) : (*(plxIoIn.PWMFallingEdges+(n/2)) & 0x0f))

/**
* @brief Initializes the pwm capture module. 
 * 
 */
void initPWMCapture(void);

/**
* @brief Initializes a channel of the input capture module. 
 * 
 * @param aChannel Defines the channel.
 * @param aPolarity Defines the polarity of a logically true signal.
 */
void plxSetupPWMCaptureChannel(u16 aChannel,PWMPolarity aPolarity, u8 aInputType);

void plxSetupDigitalInput(u16 aChannel, u8 aInputType);

void plxQueryDigitalInputConfig(void);

void plxConfigureAndInput(uint16_t aChannel1, uint16_t aPolarity1, uint16_t aChannel2, uint16_t aPolarity2);

void plxEnableAndAssertion(uint16_t aChannel1, uint16_t aPolarity1, uint16_t aChannel2, uint16_t aPolarity2);

void plxConfigureOrInput(uint16_t aChannel1, uint16_t aPolarity1, uint16_t aChannel2, uint16_t aPolarity2);

void plxEnableTripleAndAssertion(uint16_t aChannel1, uint16_t aPolarity1, uint16_t aChannel2,
                                 uint16_t aPolarity2, uint16_t aChannel3, uint16_t aPolarity3);

#define plxCheckAssertion(aAssertion) ((*plxIoIn.Assertions & (1 << aAssertion)) == 0)

void plxSetPWMPrescaler(int aPWMPrescaler);

double plxGetPWMSampleRate(void);

#ifdef __cplusplus
}
#endif

#endif /* DIGITALIN_H_ */
