local Module = {}

local Utils = require('blocks.BlockUtils')
local ModbusUtils = require('blocks.ModbusUtils')


function Module.getBlock(globals)
  local ModbusClient = require('blocks.block').getBlock(globals)

  function ModbusClient:getDirectFeedthroughCode()
    local InitCode = StringList:new()
    local OutputCode = StringList:new()
    local Require = ResourceList:new()
    local Include = StringList:new()

    local instance = ModbusClient:getInstanceNumber()

    Include:append("plexim/ModbusClient.h")
    
    print(dump(Block.Parameters))
    
    local inputConfig = Block.Parameters['InputSignalConfiguration']
    local outputConfig = Block.Parameters['OutputSignalConfiguration']
    local littleEndianRegisterOrder = (Block.Parameters['RegisterOrder'] == 1)

    local bitInputAddressToConfigMap, regInputAddressToConfigMap = ModbusUtils.getModbusAddressToConfigMaps(inputConfig)
    local bitOutputAddressToConfigMap, regOutputAddressToConfigMap = ModbusUtils.getModbusAddressToConfigMaps(outputConfig)
    local bitInputRanges, numBitInputs = ModbusUtils.getRanges(bitInputAddressToConfigMap)
    local regInputRanges, numRegInputs = ModbusUtils.getRanges(regInputAddressToConfigMap)
    local bitOutputRanges, numBitOutputs = ModbusUtils.getRanges(bitOutputAddressToConfigMap)
    local regOutputRanges, numRegOutputs = ModbusUtils.getRanges(regOutputAddressToConfigMap)
    local totalSize = numBitInputs + numRegInputs + numBitOutputs + numRegOutputs

    local outputSignal = Block:OutputSignal()
    local bitInputMap, regInputBufferOffset = ModbusUtils.getSignalToBufferAddressMap(bitInputAddressToConfigMap)
    regInputBufferOffset = (regInputBufferOffset + 15) // 16
    local regInputMap, bitOutputBufferOffset = ModbusUtils.getSignalToBufferAddressMap(regInputAddressToConfigMap)
    bitOutputBufferOffset = bitOutputBufferOffset + regInputBufferOffset
    local bitOutputMap, regOutputBufferOffset = ModbusUtils.getSignalToBufferAddressMap(bitOutputAddressToConfigMap)
    regOutputBufferOffset = (regOutputBufferOffset + 15) // 16 + bitOutputBufferOffset
    local regOutputMap, vOutputBufferOffset = ModbusUtils.getSignalToBufferAddressMap(regOutputAddressToConfigMap)
    vOutputBufferOffset = vOutputBufferOffset + regOutputBufferOffset
    local sOutputBufferOffset = vOutputBufferOffset + 1

    local bufferOffset = Target.Coder.RegisterModbusDevice(sOutputBufferOffset+1)
    regInputBufferOffset = regInputBufferOffset + bufferOffset
    bitOutputBufferOffset = bitOutputBufferOffset + bufferOffset
    regOutputBufferOffset = regOutputBufferOffset + bufferOffset
    vOutputBufferOffset = vOutputBufferOffset + bufferOffset
    sOutputBufferOffset = sOutputBufferOffset + bufferOffset

    InitCode:append("/* Modbus initialization */\n")
    local bitInputInitCode = ModbusUtils.generateBitBufferAssignmentCode(inputConfig, bufferOffset, ModbusUtils.defaultValueCode)
    local regInputInitCode = ModbusUtils.generateBufferAssignmentCode(inputConfig, regInputMap, regInputBufferOffset, ModbusUtils.defaultValueCode)
    local bitOutputInitCode = ModbusUtils.generateBitBufferAssignmentCode(outputConfig, bitOutputBufferOffset, ModbusUtils.defaultValueCode)
    local regOutputInitCode = ModbusUtils.generateBufferAssignmentCode(outputConfig, regOutputMap, regOutputBufferOffset, ModbusUtils.defaultValueCode)
    InitCode:append("{\n")
    InitCode:append("uint16_t* modbusBuffer = plxGetModbusBuffer();\n")    
    for _, v in pairs(bitInputInitCode) do table.insert(InitCode, v) end
    for _, v in pairs(regInputInitCode) do table.insert(InitCode, v) end
    for _, v in pairs(bitOutputInitCode) do table.insert(InitCode, v) end
    for _, v in pairs(regOutputInitCode) do table.insert(InitCode, v) end
    InitCode:append("}\n")

    InitCode:append("{\n")
    InitCode:append("static bool initialized = false;\n")
    InitCode:append("if (!initialized) {\n")
    InitCode:append("const uint32_t ipAddr = ")

    local ip = Block.Parameters['ServerIpAddress']
    if string.match(ip, "%d+%.%d+%.%d+%.%d+") ~= ip then
      InitCode:append("plxResolveHostname(\"%s\");" % { ip } )
    else
      local idx = 1;
      for num in string.gmatch(ip, "%d+") do
        if tonumber(num) > 255 then
          return "Parameter @param:ServerIpAddress: is not a valid IP address."
        end
        if idx > 1 then
          InitCode:append(" + ");
        end
        InitCode:append("(%s << %i)" % { num, (4-idx) * 8 } )
        idx = idx + 1
      end
      InitCode:append(";\n")
    end

    InitCode:append("#pragma pack(push, 4)\n")
    InitCode:append("const struct MyModbusClientSetupMsg {\n")
    InitCode:append("uint32_t mMsg;\n")
    InitCode:append("uint32_t mMsgLength;\n")
    InitCode:append("uint32_t mClientInstance;\n")
    InitCode:append("uint32_t mBufferOffset;\n")
    InitCode:append("uint32_t mTimeout;\n")
    InitCode:append("uint32_t mRetries;\n")
    InitCode:append("uint32_t mIpAddress;\n")
    InitCode:append("uint16_t mPort;\n")
    InitCode:append("uint16_t mUnitIdentifier;\n")
    InitCode:append("uint16_t mNumBitInputs;\n")
    InitCode:append("uint16_t mNumRegInputs;\n")
    InitCode:append("uint16_t mNumBitOutputs;\n")
    InitCode:append("uint16_t mNumRegOutputs;\n")
    InitCode:append("uint16_t mData[%i];\n" % { 2*totalSize })
    InitCode:append("} modbusClientSetupMsg = {\n")
    InitCode:append(".mMsgLength = %i,\n" % { 5*4 + 6*2 + 4*totalSize })
    InitCode:append(".mClientInstance = %i,\n" % { instance })
    InitCode:append(".mBufferOffset = %i,\n" % { bufferOffset })
    InitCode:append(".mIpAddress = ipAddr,\n")
    InitCode:append(".mPort = %i,\n" % { Block.Parameters['Port'] } )
    InitCode:append(".mUnitIdentifier = %i,\n" % { Block.Parameters['UnitIdentifier'] } )    
    InitCode:append(".mNumBitInputs = %i,\n" % { numBitInputs } )
    InitCode:append(".mNumRegInputs = %i,\n" % { numRegInputs } )
    InitCode:append(".mNumBitOutputs = %i,\n" % { numBitOutputs } )
    InitCode:append(".mNumRegOutputs = %i,\n" % { numRegOutputs } )
    InitCode:append(".mData = {\n")
    for i, c in Utils.pairsByKeys(bitInputRanges) do
      InitCode:append("%i, %i, " % { i-1, c } )
    end
    InitCode:append("\n")
    for i, c in Utils.pairsByKeys(regInputRanges) do
      InitCode:append("%i, %i, " % { i-1, c } )
    end
    InitCode:append("\n")
    for i, c in Utils.pairsByKeys(bitOutputRanges) do
      InitCode:append("%i, %i, " % { i-1, c } )
    end
    InitCode:append("\n")
    for i, c in Utils.pairsByKeys(regOutputRanges) do
      InitCode:append("%i, %i, " % { i-1, c } )
    end
    InitCode:append("\n}\n")
    InitCode:append("};\n")   
    InitCode:append("#pragma pack(pop)\n")
    InitCode:append("  plxSetupModbusClient((struct ModbusClientSetupMsg*)&modbusClientSetupMsg);\n" )
    InitCode:append("  initialized = true;\n")
    InitCode:append("}\n")
    InitCode:append("}\n")
 
    OutputCode:append("{\n")
    OutputCode:append("uint16_t* modbusBuffer = plxGetModbusBuffer();\n")    
    local offset = 0
    for i, c in pairs(outputConfig) do
      local t = c.type
      if t == 0 then -- bool
        for j = 1, c.width do
          local addr = bitOutputMap[i] + j - 1
          local bufOffset = addr // 16 
          OutputCode:append("%s = (*(modbusBuffer+%i)) & 0x%x;" % {
            outputSignal[i][j],
            bufOffset + bitOutputBufferOffset,
            2^(addr - (bufOffset * 16))
          })
        end
      else
        local typeSize = ModbusUtils.typeSizes[t]
        if typeSize > 1 then
          for j = 1, c.width do
            OutputCode:append("{\nuint16_t* pu = (uint16_t*)&%s;" % {
              outputSignal[i][j]
            })
            for l = 1, typeSize do
              OutputCode:append("*(pu + %i) = *((modbusBuffer)+%i);\n"  % {
                littleEndianRegisterOrder and l-1 or typeSize-l,
                regOutputMap[i] + regOutputBufferOffset + j + l - 2,
              })
            end
            OutputCode:append("}\n")
          end
        else
          for j = 1, c.width do
            OutputCode:append("%s = (*(((%s*)modbusBuffer)+%i));" % {
              outputSignal[i][j],
              ModbusUtils.typeNames[t],
              regOutputMap[i] + regOutputBufferOffset + j - 1
            })
          end
        end
      end
    end
    
    OutputCode:append("%s = *(modbusBuffer+%i) & 0x1;" % { outputSignal[#outputSignal-1][1], vOutputBufferOffset })
    OutputCode:append("*(modbusBuffer+%i) = 0;" % { vOutputBufferOffset })
    OutputCode:append("%s = *(modbusBuffer+%i);" % { outputSignal[#outputSignal][1], sOutputBufferOffset })

    OutputCode:append("}\n")
   
    return {
      Include = Include,
      InitCode = InitCode,
      OutputCode = OutputCode,
      Require = Require,
      UserData = {
        inputConfig = inputConfig,
        bufferOffset = bufferOffset,
        regInputBufferOffset = regInputBufferOffset,
        regInputMap = regInputMap,
      }
    }
  end
  
  function ModbusClient:getNonDirectFeedthroughCode()
    local InitCode = StringList:new()
    local UpdateCode = StringList:new()
    local Require = ResourceList:new()
    local Include = StringList:new()
    local inputConfig = Block.UserData.inputConfig
    local regInputBufferOffset = Block.UserData.regInputBufferOffset
    local regInputMap = Block.UserData.regInputMap
    local bufferOffset = Block.UserData.bufferOffset
    UpdateCode:append("{\n")
    UpdateCode:append("uint16_t* modbusBuffer = plxGetModbusBuffer();\n")
    local bitBufferAssignmentCode = ModbusUtils.generateBitBufferAssignmentCode(inputConfig, bufferOffset, ModbusUtils.inputSignalCode)
    for _, v in pairs(bitBufferAssignmentCode) do table.insert(UpdateCode, v) end
    local regBufferAssignmentCode = ModbusUtils.generateBufferAssignmentCode(inputConfig, regInputMap, regInputBufferOffset, ModbusUtils.inputSignalCode)
    for _, v in pairs(regBufferAssignmentCode) do table.insert(UpdateCode, v) end
    
    local execution = Block.Parameters['Execution']
    local triggerType = Block.Parameters['TriggerType']
    
    if execution == 1 then -- regular
      local interval = Block.Parameters['TriggerInterval']
      if (interval < 0.01) then
        return "Parameter @param:TriggerInterval: is too small. The minimum trigger interval is 1e-2."
      end
      local sampleTime = Block.Task.SampleTime[1]
      local numSamples = interval / sampleTime
      if math.floor(numSamples * 100 + 0.5) ~= math.floor(numSamples + 0.5) * 100 then
        return "Parameter @param:TriggerInterval: is not an integer multiple of the block sample time %.9g." %
          { 
            sampleTime
          }
        end
      UpdateCode:append("static const uint32_t numTicks = %d;\n" % math.floor(numSamples + 0.5))
      UpdateCode:append("static uint32_t tickCounter = 0;\n")
      UpdateCode:append("if (tickCounter == 0)\n")
      UpdateCode:append("{\n")
      UpdateCode:append("plxTriggerModbusClient(%i);\n" % { ModbusClient:getInstanceNumber() })
      UpdateCode:append("tickCounter = numTicks;\n")
      UpdateCode:append("}")
      UpdateCode:append("tickCounter--;")
    elseif execution == 2 then -- triggered
      UpdateCode:append("static bool lastTrigger = false;\n")
      UpdateCode:append("bool trigger = %s > 0;\n" % { Block.InputSignal[#Block.InputSignal][1]} )
      if triggerType == 1 then -- rising
        UpdateCode:append("if (trigger && !lastTrigger)\n")
      elseif triggerType == 2 then --falling
        UpdateCode:append("if (!trigger && lastTrigger)\n")
      else -- both
        UpdateCode:append("if (trigger != lastTrigger)\n")
      end
      UpdateCode:append("plxTriggerModbusClient(%i);\n" % { ModbusClient:getInstanceNumber() })
      UpdateCode:append("lastTrigger = trigger;\n")
    end
    UpdateCode:append("}\n")
    
    return {
      Include = Include,
      InitCode = InitCode,
      UpdateCode = UpdateCode,
      Require = Require,
    }
  end
  
  return ModbusClient
end  
return Module
