/*
   Copyright (c) 2020 by Plexim GmbH
   All rights reserved.

   A free license is granted to anyone to use this software for any legal
   non safety-critical purpose, including commercial applications, provided
   that:
   1) IT IS NOT USED TO DIRECTLY OR INDIRECTLY COMPETE WITH PLEXIM, and
   2) THIS COPYRIGHT NOTICE IS PRESERVED in its entirety.

   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
   OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
   SOFTWARE.
 */

#include "plx_canbus.h"

void PLX_CANBUS_sinit()
{
   static bool firstCall = true;
   if(firstCall)
   {
      firstCall = false;
      __HAL_RCC_CAN1_CLK_ENABLE();
   }
}

PLX_CANBUS_Handle_t PLX_CANBUS_init(void *aMemory, const size_t aNumBytes)
{
   PLX_CANBUS_Handle_t handle;

   if(aNumBytes < sizeof(PLX_CANBUS_Obj_t))
   {
      return((PLX_CANBUS_Handle_t)NULL);
   }
   // set handle
   handle = (PLX_CANBUS_Handle_t)aMemory;

   return handle;
}

PLX_CANBUS_TxHeader_t PLX_CANBUS_TxHeader_init(void *aMemory, const size_t aNumBytes)
{
   PLX_CANBUS_TxHeader_t handle;

   if(aNumBytes < sizeof(PLX_CANBUS_TxHeader_Obj_t))
   {
      return((PLX_CANBUS_TxHeader_t)NULL);
   }
   // set handle
   handle = (PLX_CANBUS_TxHeader_t)aMemory;

   return handle;
}

void PLX_CANBUS_configure(PLX_CANBUS_Handle_t aHandle, PLX_CANBUS_Unit_t aUnit, CAN_InitTypeDef *aInitDef)
{
   PLX_CANBUS_Obj_t *obj = (PLX_CANBUS_Obj_t*)aHandle;
   obj->unit = aUnit;

   switch(obj->unit)
   {
      case PLX_CAN_CAN1:
      obj->canHandle.Instance = CAN;
      break;
      default:
      PLX_ASSERT(0);
   }
   obj->canHandle.Init = *aInitDef;
   if (HAL_CAN_Init(&obj->canHandle) != HAL_OK)
   {
      PLX_ASSERT(0);
   }
   obj->lastErrorCode = 0;
}

void PLX_CANBUS_configureTxHeader(PLX_CANBUS_TxHeader_t aHandle, CAN_TxHeaderTypeDef aTypeDef)
{
   PLX_CANBUS_TxHeader_Obj_t *obj = (PLX_CANBUS_TxHeader_Obj_t*)aHandle;
   obj->header = aTypeDef;
}

CAN_HandleTypeDef* PLX_CANBUS_getHandle(PLX_CANBUS_Handle_t aHandle)
{
   PLX_CANBUS_Obj_t *obj = (PLX_CANBUS_Obj_t*)aHandle;
   return &obj->canHandle;
}

void PLX_CANBUS_start(PLX_CANBUS_Handle_t aHandle)
{
   PLX_CANBUS_Obj_t *obj = (PLX_CANBUS_Obj_t*)aHandle;
   CAN_HandleTypeDef *hcan = &obj->canHandle;
   if (hcan->State == HAL_CAN_STATE_READY)
   {
      /* Change CAN peripheral state */
      hcan->State = HAL_CAN_STATE_LISTENING;
      /* Request leave initialisation */
      CLEAR_BIT(hcan->Instance->MCR, CAN_MCR_INRQ);
      /* Reset the CAN ErrorCode */
      hcan->ErrorCode = HAL_CAN_ERROR_NONE;
   }
}

void PLX_CANBUS_setBusOn(PLX_CANBUS_Handle_t aHandle, bool busOn)
{
   PLX_CANBUS_Obj_t *obj = (PLX_CANBUS_Obj_t*)aHandle;
   CAN_HandleTypeDef *hcan = &obj->canHandle;
   if (busOn)
   {
      if (hcan->State == HAL_CAN_STATE_READY)
      {
         hcan->State = HAL_CAN_STATE_LISTENING;
         CLEAR_BIT(hcan->Instance->MCR, CAN_MCR_INRQ);
         /* Reset the CAN ErrorCode */
         hcan->ErrorCode = HAL_CAN_ERROR_NONE;
      }
   }
   else
   {
      if (hcan->State == HAL_CAN_STATE_LISTENING)
      {
         /* Request initialisation */
         SET_BIT(hcan->Instance->MCR, CAN_MCR_INRQ);
         /* Exit from sleep mode */
         CLEAR_BIT(hcan->Instance->MCR, CAN_MCR_SLEEP);
         /* Change CAN peripheral state */
         hcan->State = HAL_CAN_STATE_READY;
      }
   }
}

bool PLX_CANBUS_isBusOn(PLX_CANBUS_Handle_t aHandle, __attribute__ ((unused)) bool aAutoBusOnEnabled)
{
   PLX_CANBUS_Obj_t *obj = (PLX_CANBUS_Obj_t*)aHandle;
   uint32_t statusRegister;
   statusRegister = READ_REG(obj->canHandle.Instance->ESR);
   if ((statusRegister & CAN_ESR_BOFF) >> CAN_ESR_BOFF_Pos)
   {
      return false;
   }
   else
   {
      return true;
   }
}

uint8_t PLX_CANBUS_getLastErrorCode(PLX_CANBUS_Handle_t aHandle)
{
   PLX_CANBUS_Obj_t *obj = (PLX_CANBUS_Obj_t*)aHandle;
   uint32_t statusRegister;
   statusRegister = (READ_REG(obj->canHandle.Instance->ESR) & CAN_ESR_LEC) >> CAN_ESR_LEC_Pos;
   obj->lastErrorCode = (uint8_t)statusRegister;
   return obj->lastErrorCode;
}

bool PLX_CANBUS_isErrorActive(PLX_CANBUS_Handle_t aHandle)
{
   PLX_CANBUS_Obj_t *obj = (PLX_CANBUS_Obj_t*)aHandle;
   uint32_t statusRegister = READ_REG(obj->canHandle.Instance->ESR);

   // Error Active state: no Bus Off and no Error Passive flags
   return !((statusRegister & CAN_ESR_BOFF) || (statusRegister & CAN_ESR_EPVF));
}

extern void PLX_CANBUS_initMsgQueue(PLX_CAN_MSG_QUEUE_Handle_t aQueue, void* aMsgBuffer, uint16_t aBufferSize, PLX_CANBUS_Protocol_t aProtocol)

{
   PLX_CAN_MSG_QUEUE_t* obj = (PLX_CAN_MSG_QUEUE_t*)aQueue;
   obj->bufferSize = aBufferSize;
   obj->tail = 0;
   obj->head = 0;
   obj->length = 0;
   obj->msgBuffer = aMsgBuffer;
}

uint8_t PLX_CANBUS_getReceiveErrorCounter(PLX_CANBUS_Handle_t aHandle)
{
   PLX_CANBUS_Obj_t *obj = (PLX_CANBUS_Obj_t*)aHandle;
   uint32_t counterRegister;
   counterRegister = READ_REG(obj->canHandle.Instance->ESR);

   return (uint8_t)((counterRegister & CAN_ESR_REC) >> CAN_ESR_REC_Pos);
}

uint8_t PLX_CANBUS_getTransmitErrorCounter(PLX_CANBUS_Handle_t aHandle)
{
   PLX_CANBUS_Obj_t *obj = (PLX_CANBUS_Obj_t*)aHandle;
   uint32_t counterRegister;
   counterRegister = READ_REG(obj->canHandle.Instance->ESR);

   return (uint8_t)((counterRegister & CAN_ESR_TEC) >> CAN_ESR_TEC_Pos);
}

uint8_t PLX_CANBUS_getNodeState(PLX_CANBUS_Handle_t aHandle)
{
   PLX_CANBUS_Obj_t *obj = (PLX_CANBUS_Obj_t*)aHandle;
   uint8_t nodeState = 0;
   uint32_t statusRegister;
   statusRegister = READ_REG(obj->canHandle.Instance->ESR);
   if ((statusRegister & CAN_ESR_BOFF) >> CAN_ESR_BOFF_Pos)
   {
      nodeState = 3;
   }
   else
   {
      if ((statusRegister & CAN_ESR_EPVF) >> CAN_ESR_EPVF_Pos)
      {
         nodeState = 2;
      }
      else
      {
         nodeState = 1;
      }
   }
   return nodeState;
}
