--[[
  Copyright (c) 2023 by Plexim GmbH
  All rights reserved.

  A free license is granted to anyone to use this software for any legal
  non safety-critical purpose, including commercial applications, provided
  that:
  1) IT IS NOT USED TO DIRECTLY OR INDIRECTLY COMPETE WITH PLEXIM, and
  2) THIS COPYRIGHT NOTICE IS PRESERVED in its entirety.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
  OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
  AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
  OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
  SOFTWARE.
--]]

local T = { }
local U = require('common.utils')

function T.getChipName()
  return Target.Variables.ChipName
end

function T.getFullChipName()
  return Target.Variables.FullChipName
end

function T.getLuaTargetFileName()
  return string.sub(T.getFullChipName(), 1, -2) .. 'x'
end

local MCU = require('targets.' .. T.getLuaTargetFileName())

function T.getMemorySizeChipName()
  local str = T.getFullChipName()
  return string.sub(str, 1, -3) .. "x" .. string.sub(str, -1)
end

function T.getCpuName(index)
  return T.getTargetParameters()['cpus'][T.getChipName()][tonumber(index)]
end

function T.getNvicAddress(index)
  return T.getTargetParameters()['nvicAddress'][T.getChipName()][tonumber(index)]
end

function T.getPowerSupplySetting()
  local targetName = 'PowerSupply' .. T.getChipName()
  local powerSupplySetting = 1
  if Target.Variables[targetName] then
    powerSupplySetting = Target.Variables[targetName]
  end
  return powerSupplySetting
end

function T.getAllPossibleFamilyPrefixes()
  return {'f3', 'g4', 'h7'}
end
--[[
    Check if the current target matches one in the argument,
    where the arg 'targets' is optionally a string for one target
    or a list of strings for one or more targets.
--]]
function T.targetMatches(targets, should_be_nil)
  if should_be_nil ~= nil then
    error('Multiple targets must be listed in an array for targetMatches.')
  end
  return U.targetMatches(targets, T)
end

function T.targetSupportsVarDeadTime()
  return T.getTargetParameters()['pwm']['supports_var_dead_time']
end

function T.populateResourceTable(resources)
  local generatedResourceTable = MCU['resourceTable']
  for peripheral, units in pairs(generatedResourceTable) do
    for _, unit in pairs(units) do
      resources:add(unit)
    end
  end
end

function T.populateGpioResources(resources)
  local gpios = MCU['gpio_af']
  for gpio, k in pairs(gpios) do
    local port, pin = string.match(gpio, "(%a)(%d+)")
    resources:add("P".. port, tonumber(pin))
  end
end

function T.populateMcuTargetResources(params)

  if params.dacs == nil then
    params.dacs = {}
  end
  for unit, data in pairs(MCU.dacs) do
    if params.dacs[unit] then
      for k, v in pairs(data) do
        params.dacs[unit][k] = v
      end
    else
      params.dacs[unit] = data
    end
  end
  if params.comps == nil then
    params.comps = {}
  end
  for unit, data in pairs(MCU.comps) do
    if params.comps[unit] then
      for k, v in pairs(data) do
        params.comps[unit][k] = v
      end
    else
      params.comps[unit] = data
    end
  end
  if params.opamps == nil then
    params.opamps = {}
  end
  for unit, data in pairs(MCU.opamps) do
    if params.opamps[unit] then
      for k, v in pairs(data) do
        params.opamps[unit][k] = v
      end
    else
      params.opamps[unit] = data
    end
  end
end

function T.configure(resources)
  -- non-ST resources
  resources:add("Model Trigger")  
  resources:add("Powerstage Control")
  resources:add("Base Task Load")
  -- generic GPIOs
  T.populateGpioResources(resources)
  -- populate automatically generated resource table
  T.populateResourceTable(resources)
end

function T.checkAlternateFunctionExists(params)
  U.enforceParamContract(
    params,
    {
      func = U.isString,
      pad = U.isString,
    })

  local pads = MCU.gpio_af
  local afs = pads[params.pad]
  if afs then
    local af = afs[params.func]
    if af then
      return true
    end
  end

  return false
end

function T.getAlternateFunctionOrError(params)
  U.enforceParamContract(
    params,
    {
      func = U.isString,
      pad = U.isString,
      opt_errMsgPrefix = U.isString,
      opt_errArgs = U.isTable,
    })

  local pads = MCU.gpio_af
  local afs = pads[params.pad]
  if afs then
    local af = afs[params.func]
    if af then
      return af
    end
  end

  local defaultErrMsgPrefix = '%s does not provide function %s.'
     % {params.pad, params.func}
  local errMsgPrefix = params.opt_errMsgPrefix or defaultErrMsgPrefix
  U.error(errMsgPrefix..'\n\n'..T.getValidPadsMsg({func = params.func}), params.opt_errArgs or {})
end

function T.getValidPadsMsg(params)
  U.enforceParamContract(
    params,
    {
      func = U.isString,
      opt_userErrorFunc = U.isString,
    })

  local pads = MCU.gpio_af
  local validPads = {}
  for p, f in pairs(pads) do
    for k, _ in pairs(f) do
      -- string.match returns the substring of k that matches the regular
      -- expression params.func. If k is the function we are looking for, that
      -- substring will be identical to k.
      if string.match(k, params.func) == k then
        table.insert(validPads, p)
      end
    end
  end
  if #validPads == 0 then
    return 'The function %s is not available on the selected target.'
       % {params.opt_userErrorFunc or params.func}
  end
  local preamble = 'Valid pins for function %s are:'
     % {params.opt_userErrorFunc or params.func}
  table.insert(validPads, 1, preamble)
  return table.concat(validPads, '\n• P')
end


function T.getPinAlternateFunctionDescription(fun, pad)
  local afs = MCU['gpio_af'][pad]
  if afs == nil then
    return nil
  end
  local afCode
  for name, af in pairs(afs) do
    if af == fun then
      afCode = name
      break
    end
  end
  return afCode
end

function T.getGpioLocation(pad)
  local loc = T.getTargetParameters()['gpio_location'][pad]

  return loc
end

function T.getAdcVRef()
  return U.enforceCO(U.isNonNegativeScalar, 'AdcVRef')
end

function T.getAdcGpio(adc, channel)
  local adcInput = {}
  local adcs = MCU['adcs']
  local internalAdcs = T.getInternalAdcChannels()
  adcInput.pads = adcs['ADC%d' % {adc}]

  if adcInput.pads ~= nil then
    adcInput.gpio = adcInput.pads['IN%d' % {channel}]
    if adcInput.gpio == nil then
      -- check internal channels
      adcInput.pads = internalAdcs['ADC%d' % {adc}]
      if adcInput.pads ~= nil then
        adcInput.gpio = adcInput.pads['IN%d' % {channel}]
      end
    end
  end
  return adcInput
end

function T.getHrtimGpio(hrtim, unit, out)
  local gpio
  local pad 

  local fun = 'HRTIM%d_CH%s%d' % {hrtim, unit, out}
  local padFound = false

  local pads = MCU['gpio_af']
  for p, f in pairs(pads) do
    if padFound then
      break
    end
    for k, afunc in pairs(f) do
      if k == fun then
        pad = p
        padFound = true
        break
      end
    end
  end

  if pad ~= nil then
    local port, pin = pad:match("([A-Za-z]+)(%d+)")
    gpio = {}
    gpio.port = port
    gpio.pin = tonumber(pin)
  end

  return gpio
end

function T.getCleanSysClkHz()
  return U.enforceCO(U.megaHzIsIntegerHertz, 'SysClkMHz') * 1e6
end

function T.getCleanExtClkHz()
  return U.enforceCO(U.megaHzIsIntegerHertz, 'ExtClkMHz') * 1e6
end

function T.getHighResolutionMasterTimerSetupCode(hrtim, params)
  local targetCode = T.ts_getHighResolutionMasterTimerSetupCode(hrtim, params)

  return [[
    %(opt_adcTrigCode)s
    LL_HRTIM_TIM_SetPrescaler(%(m_hrtim)s, LL_HRTIM_TIMER_MASTER, %(m_prescale)s);
    LL_HRTIM_TIM_SetCounterMode(%(m_hrtim)s, LL_HRTIM_TIMER_MASTER, LL_HRTIM_MODE_CONTINUOUS);
    LL_HRTIM_TIM_SetPeriod(%(m_hrtim)s, LL_HRTIM_TIMER_MASTER, %(period)d);
    LL_HRTIM_TIM_SetRepetition(%(m_hrtim)s, LL_HRTIM_TIMER_MASTER, %(rep)d);
    LL_HRTIM_TIM_DisableHalfMode(%(m_hrtim)s, LL_HRTIM_TIMER_MASTER);
    %(opt_interleavedModeCode)s
    LL_HRTIM_TIM_%(startOnSync)sStartOnSync(%(m_hrtim)s, LL_HRTIM_TIMER_MASTER);
    LL_HRTIM_TIM_%(resetOnSync)sResetOnSync(%(m_hrtim)s, LL_HRTIM_TIMER_MASTER);
    LL_HRTIM_TIM_SetDACTrig(%(m_hrtim)s, LL_HRTIM_TIMER_MASTER, LL_HRTIM_DACTRIG_NONE);
    LL_HRTIM_TIM_%(preload)sPreload(%(m_hrtim)s, LL_HRTIM_TIMER_MASTER);
    LL_HRTIM_TIM_SetUpdateGating(%(m_hrtim)s, LL_HRTIM_TIMER_MASTER, LL_HRTIM_UPDATEGATING_INDEPENDENT);
    LL_HRTIM_TIM_SetUpdateTrig(%(m_hrtim)s, LL_HRTIM_TIMER_MASTER, LL_HRTIM_UPDATETRIG_REPETITION);
    LL_HRTIM_TIM_SetBurstModeOption(%(m_hrtim)s, LL_HRTIM_TIMER_MASTER, LL_HRTIM_BURSTMODE_MAINTAINCLOCK);
    LL_HRTIM_ForceUpdate(%(m_hrtim)s, LL_HRTIM_TIMER_MASTER);
    LL_HRTIM_TIM_SetCompare1(%(m_hrtim)s, LL_HRTIM_TIMER_MASTER, %(m_cmp1)s);
    LL_HRTIM_TIM_SetCompare3(%(m_hrtim)s, LL_HRTIM_TIMER_MASTER, %(m_cmp3)s);
    LL_HRTIM_TIM_SetCompare2(%(m_hrtim)s, LL_HRTIM_TIMER_MASTER, %(m_cmp2)s);
    LL_HRTIM_TIM_SetCompare4(%(m_hrtim)s, LL_HRTIM_TIMER_MASTER, %(m_cmp4)s);
    %(opt_enableRepCode)s
  ]] % {
    opt_adcTrigCode = targetCode.opt_adcTrigCode or '',
    opt_interleavedModeCode = targetCode.opt_interleavedModeCode or '',
    opt_enableRepCode = targetCode.opt_enableRepCode or '',
    m_prescale = targetCode.m_prescale,
    rep = params.repetition and math.max(0, params.repetition - 1) or 0,
    preload = (params.mode == 'pcc') and 'Disable' or 'Enable',
    resetOnSync = params.reset_tim_on_sync and 'Enable' or 'Disable',
    startOnSync = params.start_tim_on_sync and 'Enable' or 'Disable',
    period = params.period,
    m_hrtim = 'HRTIM%d' % {hrtim},
    m_cmp1 = params.cmp_init and tostring(params.cmp_init[1]) or targetCode.m_cmpDefault,
    m_cmp2 = params.cmp_init and tostring(params.cmp_init[2]) or targetCode.m_cmpDefault,
    m_cmp3 = params.cmp_init and tostring(params.cmp_init[3]) or targetCode.m_cmpDefault,
    m_cmp4 = params.cmp_init and tostring(params.cmp_init[4]) or targetCode.m_cmpDefault,
  }
end

function T.getHighResolutionSubTimerSetupCode(hrtim, params)
  --[[
    repetition counter similar to advanced PWM tims (does not have updown-challenge as ROM option is provided)
  --]]

  local targetCode = T.ts_getHighResolutionSubTimerSetupCode(hrtim, params)

  local m_hrtim = 'HRTIM%d' % {hrtim}
  local m_subtim = 'LL_HRTIM_TIMER_%s' % {params.subtimer}

  local deadTimeCode
  if params.dead_time.counts then
    local m_deadTimePrescalerMacros = {
      [1] = 'LL_HRTIM_DT_PRESCALER_MUL8',
      [2] = 'LL_HRTIM_DT_PRESCALER_MUL4',
      [4] = 'LL_HRTIM_DT_PRESCALER_MUL2',
      [8] = 'LL_HRTIM_DT_PRESCALER_DIV1',
      [16] = 'LL_HRTIM_DT_PRESCALER_DIV2',
      [32] = 'LL_HRTIM_DT_PRESCALER_DIV4',
      [64] = 'LL_HRTIM_DT_PRESCALER_DIV8',
      [128] = 'LL_HRTIM_DT_PRESCALER_DIV16',
    }

    deadTimeCode = [[
      LL_HRTIM_TIM_EnableDeadTime(%(m_hrtim)s, %(m_subtim)s);
      LL_HRTIM_DT_Config(%(m_hrtim)s, %(m_subtim)s, %(m_dtPrescale)s | LL_HRTIM_DT_RISING_POSITIVE | LL_HRTIM_DT_FALLING_POSITIVE);
      LL_HRTIM_DT_SetRisingValue(%(m_hrtim)s, %(m_subtim)s, %(dtCounts)d);
      LL_HRTIM_DT_SetFallingValue(%(m_hrtim)s, %(m_subtim)s, %(dtCounts)d);
      LL_HRTIM_DT_LockRisingSign(%(m_hrtim)s, %(m_subtim)s);
      LL_HRTIM_DT_LockFallingSign(%(m_hrtim)s, %(m_subtim)s);
    ]] % {
      m_hrtim = m_hrtim,
      dtCounts = params.dead_time.counts,
      m_dtPrescale = m_deadTimePrescalerMacros[params.dead_time.prescale],
      m_subtim = m_subtim,
    }
  else
    deadTimeCode = [[
      LL_HRTIM_TIM_DisableDeadTime(%(m_hrtim)s, %(m_subtim)s);
    ]] % {
      m_hrtim = m_hrtim,
      m_subtim = m_subtim,
    }
  end

  local eventCode = ''
  if params.mode == 'pcc' then
    if params.events then
      for event, conf in pairs(params.events) do
        eventCode = eventCode..[[
          LL_HRTIM_TIM_SetEventFilter(%(m_hrtim)s,
                                      %(m_subtim)s,
                                      LL_HRTIM_EVENT_%(event)d,
                                      LL_HRTIM_EEFLTR_%(filter)s);
          LL_HRTIM_TIM_SetEventLatchStatus(%(m_hrtim)s,
                                           %(m_subtim)s,
                                           LL_HRTIM_EVENT_%(event)d,
                                           LL_HRTIM_EELATCH_%(m_latch)s);
        ]] % {
          m_hrtim = m_hrtim,
          m_subtim = m_subtim,
          event = event,
          filter = conf.filter and tostring(conf.filter) or 'NONE',
          m_latch = conf.latch_enabled and 'ENABLED' or 'DISABLED',
        }
      end
    end
  end

  local triggerCode
  if params.mode == 'single_shot' then
    if params.phase_shift then
      triggerCode = [[
        LL_HRTIM_TIM_SetUpdateTrig(%(m_hrtim)s,
                                   %(m_subtim)s,
                                   LL_HRTIM_UPDATETRIG_RESET);
        LL_HRTIM_TIM_SetResetTrig(%(m_hrtim)s,
                                  %(m_subtim)s,
                                  LL_HRTIM_RESETTRIG_MASTER_CMP%(cmp)d);
      ]] % {
        cmp = params.cmp,
        m_hrtim = m_hrtim,
        m_subtim = m_subtim,
      }
    else
      triggerCode = [[
        LL_HRTIM_TIM_SetUpdateTrig(%(m_hrtim)s,
                                   %(m_subtim)s,
                                   LL_HRTIM_UPDATETRIG_MASTER);
        LL_HRTIM_TIM_SetResetTrig(%(m_hrtim)s,
                                  %(m_subtim)s,
                                  LL_HRTIM_RESETTRIG_MASTER_PER);
      ]] % {
        m_hrtim = m_hrtim,
        m_subtim = m_subtim,
      }
    end
  else
    triggerCode = [[
      LL_HRTIM_TIM_SetUpdateTrig(%(m_hrtim)s,
                                 %(m_subtim)s,
                                 LL_HRTIM_UPDATETRIG_RESET);
      LL_HRTIM_TIM_SetResetTrig(%(m_hrtim)s,
                                %(m_subtim)s,
                                LL_HRTIM_RESETTRIG_NONE);
    ]] % {
      m_hrtim = m_hrtim,
      m_subtim = m_subtim,
    }
  end

  return [[
  {
    %(opt_adcTrigCode)s
    LL_HRTIM_TIM_SetPrescaler(%(m_hrtim)s, %(m_subtim)s, %(m_prescale)s);
    LL_HRTIM_TIM_SetCounterMode(%(m_hrtim)s, %(m_subtim)s, LL_HRTIM_MODE_%(m_counterMode)s);
    LL_HRTIM_TIM_SetPeriod(%(m_hrtim)s, %(m_subtim)s, %(period)d);
    LL_HRTIM_TIM_SetRepetition(%(m_hrtim)s, %(m_subtim)s, %(rep)d);
    %(opt_repRolloverCode)s
    LL_HRTIM_TIM_SetUpdateGating(%(m_hrtim)s, %(m_subtim)s, LL_HRTIM_UPDATEGATING_INDEPENDENT);
    %(opt_setModeCode)s
    LL_HRTIM_TIM_SetDACTrig(%(m_hrtim)s, %(m_subtim)s, LL_HRTIM_DACTRIG_NONE);
    LL_HRTIM_TIM_DisableHalfMode(%(m_hrtim)s, %(m_subtim)s);
    %(opt_interleavedModeCode)s
    LL_HRTIM_TIM_%(startOnSync)sStartOnSync(%(m_hrtim)s, %(m_subtim)s);
    LL_HRTIM_TIM_%(resetOnSync)sResetOnSync(%(m_hrtim)s, %(m_subtim)s);
    LL_HRTIM_TIM_%(preload)sPreload(%(m_hrtim)s, %(m_subtim)s);
    %(opt_disableResyncCode)s
    %(triggerCode)s
    LL_HRTIM_TIM_DisablePushPullMode(%(m_hrtim)s, %(m_subtim)s);
    LL_HRTIM_TIM_SetBurstModeOption(%(m_hrtim)s, %(m_subtim)s, LL_HRTIM_BURSTMODE_MAINTAINCLOCK);
    %(deadTimeCode)s
    %(opt_dacSyncCode)s
    %(eventCode)s
    LL_HRTIM_ForceUpdate(%(m_hrtim)s, %(m_subtim)s);
    %(setCompareCode)s
  }
  ]] % {
    opt_adcTrigCode = targetCode.opt_adcTrigCode or '',
    m_prescale = targetCode.m_prescale,
    opt_repRolloverCode = targetCode.opt_repRolloverCode or '',
    opt_setModeCode = targetCode.opt_setModeCode or '',
    opt_interleavedModeCode = targetCode.opt_interleavedModeCode or '',
    opt_disableResyncCode = targetCode.opt_disableResyncCode or '',
    opt_dacSyncCode = targetCode.opt_dacSyncCode or '',
    setCompareCode = targetCode.setCompareCode,
    rep = targetCode.repetitionPeriod,
    triggerCode = triggerCode,
    deadTimeCode = deadTimeCode,
    eventCode = eventCode,
    m_hrtim = m_hrtim,
    period = params.timing.period_in_timer_counts,
    m_subtim = m_subtim,
    m_counterMode = (params.mode == 'continuous' or params.mode == 'pcc') and 'CONTINUOUS' or 'RETRIGGERABLE',
    resetOnSync = params.reset_tim_on_sync and 'Enable' or 'Disable',
    startOnSync = params.start_tim_on_sync and 'Enable' or 'Disable',
    preload = (params.mode == 'pcc') and 'Disable' or 'Enable',
  }
end

function T.getAchievableTimSampleTime(ts)
  local timing = T.getTimPrescaleAndPeriod16(1/ts)
  return 1/timing.freq
end

function T.getTimPrescaleAndPeriod16(f, pmin)
  local minPeriod = 1
  local errorMsg
  local warningMsg
  if pmin ~= nil then
    -- minimal period so that we may ensure a certain PWM resolution
    minPeriod = pmin 
  end
  local clk = T.getTimerClock()
  local prescale = 1
  local period
  local bestPrescale
  local bestAbsError = f
  -- Initial check if minimal period is reachable
  if math.floor(clk/prescale/f + 0.5) < minPeriod then
    warningMsg = [[
      Unable to achieve the desired PWM resolution:
      - desired minimal counter period value: %d
      - achievable counter period value: %d
                
      Please reduce the carrier frequency setting. You may also adjust the system clock frequency under Coder Options->Target->General.
    ]]
    warningMsg = warningMsg % {minPeriod, math.floor(clk/prescale/f + 0.5)}
    U.warning(warningMsg)
  end
  minPeriod = 1
  while prescale < 0x10000 do
    local period = math.max(1, math.floor(clk/prescale/f + 0.5))
    if (period >= minPeriod) and (period <= 0x10000) then
      local fa = clk/prescale/period
      local abserr = math.abs(fa-f)
      if abserr == 0 then
        bestPrescale = prescale
        break;
      end
      if abserr < bestAbsError then
        bestPrescale = prescale
        bestAbsError = abserr
      end
    end
    prescale = prescale + 1
  end
  if bestPrescale == nil then
    errorMsg = [[   
      Unable to find an appropriate prescaler to achieve the desired PWM frequency.

      Please adjust the carrier frequency setting. 
    ]]
    U.error(errorMsg)
    bestPrescale = 1
  end
  prescale = bestPrescale
  period = math.max(1, math.floor(clk/prescale/f + 0.5))
  local achievableF = clk/prescale/period
  return {
    freq = achievableF,
    prescale = prescale,
    period = period
  }
end

function T.getHrtimPrescaleAndPeriod(f, params)
  local maxPeriod = 0xFFFF
  if params.varFreq ~= nil and params.varFreq == true then
    maxPeriod = 0x3E80 -- ensure that period value is below 16'000 for var. freq operation (to ensure a certain operational range)
  end

  local inputClk = T.getHrtimInputClkHz()
  local min_hrtim_input_freq = T.getTargetParameters()['min_hrtim_input_freq']
  if(inputClk < min_hrtim_input_freq) then
    U.error('HRTIM requires minimal system clock frequency of %d Hz.' % {min_hrtim_input_freq})
  end
    
  -- determine proper prescaling ratio clock ratio for desired period (1/f)
  local hrtim_clk_limits = T.getTargetParameters()['hrtims']['clk_limits']
  local hrtimClk = hrtim_clk_limits.input_clk_mul*inputClk
  local preScale = hrtim_clk_limits.prescale_min
  while(preScale <= hrtim_clk_limits.prescale_max) do
    local period = math.floor(hrtimClk/preScale/f-1 + 0.5)
    if period <= maxPeriod then 
      break;
    end
    preScale = preScale * 2
  end
  if preScale > hrtim_clk_limits.prescale_max then
    local minPwmFreq = hrtimClk / hrtim_clk_limits.prescale_max / (maxPeriod-1)
    U.error("Unable to achieve the desired PWM frequency (%.2f Hz is too low). Lowest achievable PWM frequency is: %.2f" % {f, minPwmFreq})
  end

  local counterPeriod = math.floor(hrtimClk/preScale/f-1 + 0.5)
  local achievablePeriodInTimerCounts = counterPeriod + 1
  local achievableF = hrtimClk/preScale/achievablePeriodInTimerCounts;

  if preScale == 1 and achievablePeriodInTimerCounts < 100 then
    U.error("Unable to achieve the desired PWM frequency (%f Hz is too high)." % {f})
  end

  return {
    freq = achievableF,
    prescale = preScale,
    period = achievablePeriodInTimerCounts,
    clk = hrtimClk/preScale
  }
end

function T.getHrtimBurstModePrescaleAndPeriod(f)
  local inputClk = T.getHrtimInputClkHz()
  local min_hrtim_input_freq = T.getTargetParameters()['min_hrtim_input_freq']
  if(inputClk < min_hrtim_input_freq) then
    U.error('HRTIM requires minimal system clock frequency of %d Hz.' % {min_hrtim_input_freq})
  end
    
  -- determine proper prescaling ratio clock ratio for desired period (1/f)
  local hrtimClk = inputClk
  local preScale = 1
  while(preScale <= 32768) do
    local period = math.floor(hrtimClk/preScale/f-1 + 0.5)
    if period <= 0xFFFF then
      break;
    end
    preScale = preScale * 2
  end
  if preScale > 32768 then
    U.error("Unable to achieve the desired Burst mode frequency (%f Hz is too low)." % {f})
  end

  local counterPeriod = math.floor(hrtimClk/preScale/f-1 + 0.5)
  local achievablePeriodInTimerCounts = counterPeriod + 1
  local achievableF = hrtimClk/preScale/achievablePeriodInTimerCounts;

  if preScale == 1 and achievablePeriodInTimerCounts < 100 then
    U.error("Unable to achieve the desired Burst mode frequency (%f Hz is too high)." % {f})
  end

  return {
    freq = achievableF,
    prescale = preScale,
    period = achievablePeriodInTimerCounts,
    clk = hrtimClk/preScale
  }
end

function T.getHrtimDeadtimePrescaleAndCounts(deadTime)
  local inputClk = T.getHrtimInputClkHz()
  local dtClk = 8*inputClk
  local dTPreScale = 1
  local dTCounts
  while(dTPreScale <= 128) do
    dTCounts = math.floor(dtClk/dTPreScale*deadTime-1 + 0.5)
    if dTCounts < 0 then
      dTCounts = 0
    end
    if dTCounts <= 0x1FF then
      break;
    end
    dTPreScale = dTPreScale * 2
  end
  if dTPreScale > 128 then
    U.error("Unable to achieve the desired deadtime (%f is too large)." % {deadTime})
  end
  return {
    counts  = dTCounts,
    prescale = dTPreScale
  }
end

function T.getBasicTimerSetupCode(params)
  U.enforceParamContract(
    params,
    {
      prescale = U.isNonNegativeIntScalar,
      period = U.isPositiveIntScalar,
      carrierType = U.isString,
      opt_repetition = U.isNonNegativeIntScalar,
    }
  )

  local m_mode, period
  if params.carrierType == 'sawtooth' then
    m_mode = 'LL_TIM_COUNTERMODE_UP'
    period = params.period - 1
  elseif params.carrierType == 'triangle' then
    m_mode = 'LL_TIM_COUNTERMODE_CENTER_UP_DOWN'
    period = params.period / 2
  else
    error('Invalid carrier type.')
  end

  local code = [[
    LL_TIM_InitTypeDef initStruct = {0};
    initStruct.Prescaler =  %(prescale)d;
    initStruct.Autoreload = %(period)d;
    initStruct.CounterMode = %(m_mode)s;
    initStruct.ClockDivision = LL_TIM_CLOCKDIVISION_DIV1;
    initStruct.RepetitionCounter = %(rep)d;
  ]] % {
    prescale = params.prescale - 1,
    rep = params.opt_repetition and math.max(0, params.opt_repetition - 1) or 0,
    period = period,
    m_mode = m_mode,
  }

  return code
end

function T.getTimerTriggerConfigCode(handle, params)
  U.enforceParamContract(
    params,
    {
      opt_isInternallySynced = U.isBoolean,
      opt_isMasterForInternalSync = U.isBoolean,
      opt_itr = U.isNonNegativeIntScalar,
      opt_startOnExternalSync = U.isBoolean,
      opt_resetOnExternalSync = U.isBoolean,
      opt_triggerType = U.isPositiveIntScalar,
      libraryBlockType = U.isString,
    }
  )

  local m_slaveMode = 'LL_TIM_SLAVEMODE_DISABLED'
  local m_trigInput = 'LL_TIM_TS_ITR0'

  if params.opt_isInternallySynced then
    if params.opt_isMasterForInternalSync and params.opt_resetOnExternalSync then
      m_slaveMode = 'LL_TIM_SLAVEMODE_RESET'
      m_trigInput = 'LL_TIM_TS_ETRF'
    else
      m_slaveMode = 'LL_TIM_SLAVEMODE_TRIGGER'
      m_trigInput = 'LL_TIM_TS_ITR%d' % {params.opt_itr}
    end
  else
    if params.opt_resetOnExternalSync then
      m_slaveMode = 'LL_TIM_SLAVEMODE_COMBINED_RESETTRIGGER'
      m_trigInput = 'LL_TIM_TS_ETRF'
    elseif params.opt_startOnExternalSync then
      m_slaveMode = 'LL_TIM_SLAVEMODE_TRIGGER'
      m_trigInput = 'LL_TIM_TS_ETRF'
    end
  end

  local m_trigOut1, m_trigOut2
  if params.libraryBlockType == 'pwm' then
    m_trigOut1 = 'LL_TIM_TRGO_ENABLE'
    m_trigOut2 = 'LL_TIM_TRGO2_UPDATE'
  elseif params.libraryBlockType == 'timer' then
    m_trigOut1 = 'LL_TIM_TRGO_UPDATE'
    m_trigOut2 = 'LL_TIM_TRGO2_RESET'
  else
    error('Invalid "type" passed to getTimerTriggerConfigCode.')
  end

  local opt_etrCode
  if params.opt_startOnExternalSync then
    local etrInitCode = T.getTimerETRSourceInitCode()

    opt_etrCode = [[
      LL_TIM_ConfigETR(handle,
                       LL_TIM_ETR_POLARITY_%(polarity)s,
                       LL_TIM_ETR_PRESCALER_DIV1,
                       LL_TIM_ETR_FILTER_FDIV1_N8);
      %(etrInitCode)s
    ]] % {
      polarity = (params.opt_triggerType == 1) and 'NONINVERTED' or 'INVERTED',
      etrInitCode = etrInitCode,
    }
  end

  local code = [[
    TIM_TypeDef* handle = PLX_TIM_getStmLLHandle(%(handle)s);  
    LL_TIM_SetSlaveMode(handle, %(m_slaveMode)s);
    LL_TIM_SetTriggerInput(handle, %(m_trigInput)s);
    LL_TIM_SetTriggerOutput(handle, %(m_trigOut1)s);
    LL_TIM_SetTriggerOutput2(handle, %(m_trigOut2)s);
    LL_TIM_DisableMasterSlaveMode(handle);
    %(etrCode)s
  ]] % {
    handle = handle,
    m_slaveMode = m_slaveMode,
    m_trigInput = m_trigInput,
    m_trigOut1 = m_trigOut1,
    m_trigOut2 = m_trigOut2,
    etrCode = opt_etrCode or '',
  }

  return code
end

function T.getTimerPwmChannelSetupCode(params)
  local code = [[
    LL_TIM_OC_InitTypeDef sConfig = {0};
    sConfig.OCMode = LL_TIM_OCMODE_%(m_ocMode)s;
    sConfig.OCState = LL_TIM_OCSTATE_DISABLE;
    sConfig.OCNState = LL_TIM_OCSTATE_DISABLE;
    sConfig.CompareValue = 100;
    sConfig.OCPolarity = LL_TIM_OCPOLARITY_%(m_polarity)s;
    sConfig.OCNPolarity = LL_TIM_OCPOLARITY_%(m_polarity)s;
    sConfig.OCIdleState = LL_TIM_OCIDLESTATE_%(m_idleState)s;
    sConfig.OCNIdleState = LL_TIM_OCIDLESTATE_%(m_idleState)s;
  ]] % {
    m_polarity = (params.polarity == 2) and 'LOW' or 'HIGH',
    m_idleState = (params.polarity == 2) and 'HIGH' or 'LOW',
    m_ocMode = (params.has_powerstage and params.is_pwm_light) and 'FORCED_INACTIVE' or 'PWM1',
  }

  return code
end

function T.calcDTGValue(deadTimeInSeconds)
  -- brute for trial of all options
  -- 0xx first
  local clk = T.getTimerClock()
  local dtg_x = math.min(0x7F, math.max(0, math.floor(clk*deadTimeInSeconds + 0.5)))
  local dt = dtg_x/clk
  local dtg = dtg_x
  local abserror = math.abs(deadTimeInSeconds - dt)
  --print('0XX: gtg:0x%X dt: %f us' % {dtg, dt*10^6})
  
  clk = T.getTimerClock()/2
  local dtg_10x_x = math.min(0x7F, math.max(0x40, math.floor(clk*deadTimeInSeconds + 0.5)))
  local dt_10x = dtg_10x_x/clk   
  --print('10X: dtg_x: 0x%X dt: %f us' % {dtg_10x_x, dt_10x*10^6})   
  local abserror_10x = math.abs(deadTimeInSeconds - dt_10x)
  if abserror_10x < abserror then
    abserror = abserror_10x
    dtg = dtg_10x_x + 0x80 - 0x40
    dt = dt_10x
  end
  
  clk = T.getTimerClock()/8
  local dtg_110_x = math.min(0x3F, math.max(0x20, math.floor(clk*deadTimeInSeconds + 0.5)))
  local dt_110 = dtg_110_x/clk 
  --print('110: dtg_x: 0x%X dt: %f us' % {dtg_110_x, dt_110*10^6})  
  local abserror_110 = math.abs(deadTimeInSeconds - dt_110)
  if abserror_110 < abserror then
    abserror = abserror_110
    dtg = dtg_110_x + 0xC0 - 0x20
    dt = dt_110
  end
  
  clk = T.getTimerClock()/16
  local dtg_111_x = math.min(0x3F, math.max(0x20, math.floor(clk*deadTimeInSeconds + 0.5)))
  local dt_111 = dtg_111_x/clk 
  --print('111: dtg_x: 0x%X dt: %f us' % {dtg_111_x, dt_111*10^6})   
  local abserror_111 = math.abs(deadTimeInSeconds - dt_111)
  if abserror_111 < abserror then
    abserror = abserror_111
    dtg = dtg_111_x + 0xE0 - 0x20
    dt = dt_111
  end  
 
  return {
    dtg = dtg,
    dt = dt
  }
end

function T.getHighResolutionTimerChannelSetupCode(hrtim, params)
  local m_hrtim = 'HRTIM%d' % {hrtim}
  local m_timCh = 'LL_HRTIM_OUTPUT_T%s%d' % {params.subtim, params.channel}

  local m_fltState
  if params.flt_state == 1 then
    m_fltState = 'LL_HRTIM_OUT_FAULTSTATE_INACTIVE'
  elseif params.flt_state == 2 then
    m_fltState = 'LL_HRTIM_OUT_FAULTSTATE_ACTIVE'
  elseif params.flt_state == 3 then
    m_fltState = 'LL_HRTIM_OUT_FAULTSTATE_HIGHZ'
  else
    m_fltState = 'LL_HRTIM_OUT_FAULTSTATE_NO_ACTION'
  end

  local outputSetSrcCode = ''
  if params.set then
    local m_setSources = T.getHrtimOutputSetSrcMacros({
      set = params.set,
      opt_cmp = params.cmp,
      carrier = params.carrier,
    })

    outputSetSrcCode = [[
      LL_HRTIM_OUT_SetOutputSetSrc(%(m_hrtim)s, %(m_timCh)s, %(m_setSources)s);
    ]] % {
      m_hrtim = m_hrtim,
      m_timCh = m_timCh,
      m_setSources = m_setSources,
    }
  end

  local outputResetSrcCode = ''
  if params.reset then
    local m_resetSources = T.getHrtimOutputResetSrcMacros({
      reset = params.reset,
    })

    outputResetSrcCode = [[
      LL_HRTIM_OUT_SetOutputResetSrc(%(m_hrtim)s, %(m_timCh)s, %(m_resetSources)s);
    ]] % {
      m_hrtim = m_hrtim,
      m_timCh = m_timCh,
      m_resetSources = m_resetSources,
    }
  end

  local code = [[
    {
      LL_HRTIM_OUT_SetPolarity(%(m_hrtim)s, %(m_timCh)s, LL_HRTIM_OUT_%(m_polarity)s_POLARITY);
      %(outputSetSrcCode)s
      %(outputResetSrcCode)s
      LL_HRTIM_OUT_SetIdleMode(%(m_hrtim)s, %(m_timCh)s, LL_HRTIM_OUT_%(m_idleMode)s);
      LL_HRTIM_OUT_SetIdleLevel(%(m_hrtim)s, %(m_timCh)s, LL_HRTIM_OUT_IDLELEVEL_%(m_idleLvl)s);
      LL_HRTIM_OUT_SetFaultState(%(m_hrtim)s, %(m_timCh)s, %(m_fltState)s);
      LL_HRTIM_OUT_SetChopperMode(%(m_hrtim)s, %(m_timCh)s, LL_HRTIM_OUT_CHOPPERMODE_DISABLED);
      PLX_HRTIM_addPwmChannel(HrtimHandles[%(parentInstance)d], PLX_HRTIM_T%(m_subtim)s%(ch)d, %(isSyncedToMaster)d);
    }
  ]] % {
    m_hrtim = m_hrtim,
    m_timCh = m_timCh,
    outputSetSrcCode = outputSetSrcCode,
    outputResetSrcCode = outputResetSrcCode,
    m_polarity = (not params.polarity or (params.polarity == 0)) and 'POSITIVE' or 'NEGATIVE',
    m_subtim = params.subtim,
    ch = params.channel,
    m_fltState = m_fltState,
    m_idleMode = params.burst_mode and 'IDLE_WHEN_BURST' or 'NO_IDLE',
    m_idleLvl = (params.idle_level == 2) and 'ACTIVE' or 'INACTIVE',
    parentInstance = params.parentInstance,
    isSyncedToMaster = params.isSyncedToMaster,
  }

  return code
end

function T.getHrtimExternalSyncSetupCode(hrtim)
  -- could also support LL_HRTIM_SYNCIN_SRC_TIM_EVENT

  return 'LL_HRTIM_SetSyncInSrc(HRTIM%d, LL_HRTIM_SYNCIN_SRC_EXTERNAL_EVENT);' % {hrtim}
end

function T.getHrtimExternalSyncOutSetupCode(hrtim, params)
  local code = [[
    LL_HRTIM_SetSyncOutSrc(%(m_hrtim)s, LL_HRTIM_SYNCOUT_SRC_%(m_syncSrc)s_START);
    LL_HRTIM_SetSyncOutConfig(%(m_hrtim)s, LL_HRTIM_SYNCOUT_%(m_syncType)s_PULSE);
  ]] % {
    m_hrtim = 'HRTIM%d' % {hrtim},
    m_syncSrc = (params.src_type == 'hrtim_master') and 'MASTER' or 'TIMA',
    m_syncType = (params.behaviour == 'positive') and 'POSITIVE' or 'NEGATIVE',
  }

  return code
end

function T.checkExtSyncHrtimConnection(subtimer)
  if subtimer ~= 'A' then
    U.error("Only HRTIM Timing unit 'A' can be connected to the External Sync block.")
  end
end

function T.getHrtimSubtimerFaultEnableCode(hrtim, params)
  local code = 'LL_HRTIM_TIM_EnableFault(%(m_hrtim)s, %(m_subtim)s,' % {
    m_hrtim = 'HRTIM%d' % {hrtim},
    m_subtim = 'LL_HRTIM_TIMER_%s' % {params.subtimer},
  }
  local faultCodes = {}
  for _, flt_line in ipairs(params.flt_lines) do
    table.insert(faultCodes, 'LL_HRTIM_FAULT_%d' % {flt_line})
  end
  code = code .. table.concat(faultCodes, ' | ') .. ');'

  return code
end

function T.getBurstModeSetupCode(hrtim, params)
  local code = [[
    {
      LL_HRTIM_BM_SetMode(HRTIM%(unit)d, LL_HRTIM_BM_MODE_CONTINOUS);
      LL_HRTIM_BM_SetClockSrc(HRTIM%(unit)d, LL_HRTIM_BM_CLKSRC_%(m_clkSrc)s);
      LL_HRTIM_BM_SetTrig(HRTIM%(unit)d, LL_HRTIM_BM_TRIG_MASTER_RESET);
      LL_HRTIM_BM_SetPeriod(HRTIM%(unit)d, %(period)d);
      LL_HRTIM_BM_SetPrescaler(HRTIM%(unit)d, LL_HRTIM_BM_PRESCALER_DIV%(prescaler)d);
      LL_HRTIM_BM_SetCompare(HRTIM%(unit)d, %(compare)d);
      LL_HRTIM_BM_EnablePreload(HRTIM%(unit)d);
    }
  ]] % {
    unit = hrtim,
    period = params.period - 1,
    compare = 0,
    prescaler = params.prescaler,
    m_clkSrc = (params.clk_source == 'master') and 'MASTER' or 'FHRTIM',
  }

  return code
end

function T.getSpiClockPrescaler(spi, clk)
  local prescaler = 256
  local apb = T.getTargetParameters()['spis']['SPI%d' % {spi}]['apb']
  local spiInputClock
  if (apb == 1) then
    spiInputClock = T.getAPB1Clock()
  elseif (apb == 2) then
    spiInputClock = T.getAPB2Clock()
  elseif (apb == 4) then
    spiInputClock = T.getAPB4Clock()
  end
  local keepSearching = true

  while (spiInputClock / prescaler < clk) and (prescaler >= 2) do
      prescaler = prescaler / 2
  end
  return prescaler
end

function T.getSpiConfigurationCode(spi, params)
  local FIFOThresholdCode = T.getSpiFIFOThresholdCode(spi, params)

  local code = [[
    LL_SPI_InitTypeDef SPI_InitStruct = {0};

    SPI_InitStruct.TransferDirection = LL_SPI_FULL_DUPLEX;
    SPI_InitStruct.Mode = LL_SPI_MODE_%(m_mode)s;
    SPI_InitStruct.DataWidth = LL_SPI_DATAWIDTH_%(dataWidth)dBIT;
    SPI_InitStruct.ClockPolarity = LL_SPI_POLARITY_%(m_polarity)s;
    SPI_InitStruct.ClockPhase = LL_SPI_PHASE_%(m_phase)s;
    SPI_InitStruct.NSS = LL_SPI_NSS_%(m_nss)s;
    SPI_InitStruct.BaudRate = LL_SPI_BAUDRATEPRESCALER_DIV%(prescaler)d;
    SPI_InitStruct.BitOrder = LL_SPI_MSB_FIRST;
    SPI_InitStruct.CRCCalculation = LL_SPI_CRCCALCULATION_DISABLE;
    LL_SPI_Init(SPI%(spi)d, &SPI_InitStruct);
    LL_SPI_SetStandard(SPI%(spi)d, LL_SPI_PROTOCOL_MOTOROLA);
    LL_SPI_DisableNSSPulseMgt(SPI%(spi)d);
    %(FIFOThresholdCode)s
  ]] % {
    spi = spi,
    m_mode = params.isMaster and 'MASTER' or 'SLAVE',
    m_nss = params.isMaster and 'SOFT' or 'HARD_OUTPUT',
    dataWidth = params.dataWidth,
    m_polarity = params.polarity and 'HIGH' or 'LOW',
    m_phase = params.phase and '2EDGE' or '1EDGE',
    prescaler = params.prescaler,
    FIFOThresholdCode = FIFOThresholdCode,
  }

  return code
end

function T.checkSpiClockIsAchievable(spi, params)

  local apb = T.getTargetParameters()['spis']['SPI%d' % {spi}]['apb']
  local spiInputClock
  if (apb == 1) then
    spiInputClock = T.getAPB1Clock()
  else
    spiInputClock = T.getAPB2Clock()
  end
  local maxClk = 75 * 1e6 
  local minClk = math.ceil(spiInputClock / (0xFF + 1))  -- maximum prescaler for baudrate is 256
  if spiInputClock/params.prescaler > maxClk then
    local errorMsg = [[
      A SPI clock rate of %.0f MHz and APB1 clock frequency of %.0f MHz requires a prescaler of %.0f. 
      The resulting SPI clock (%.0f MHz) is more than the maximum allowed clock rate of %.0f MHz.

      Please reduce the SPI clock frequency.
    ]] % {params.baudrate/1e6, spiInputClock/1e6, params.prescaler, math.floor(spiInputClock/params.prescaler)/1e6, maxClk/1e6}
    U.error(errorMsg)
  end
end

function T.getActualSpiClockFrequency(spi, params)
  local apb = T.getTargetParameters()['spis']['SPI%d' % {spi}]['apb']
  local spiInputClock
  if (apb == 1) then
    spiInputClock = T.getAPB1Clock()
  else
    spiInputClock = T.getAPB2Clock()
  end
  return spiInputClock/params.prescaler
end

function T.getMaxSciBaudRate()
  -- assuming 8N1. 1.5 characters per poll
  local maxRate = 1/Target.Variables.SAMPLE_TIME * 15
  return math.min(115200, maxRate)
end

function T.getJLinkOptions(serial_number)
  local sn = ''
  if serial_number ~= nil then
    sn = sn .. '=%s' % {serial_number}
  end

  return '-select USB%(serial_number)s -if swd -device STM32%(chip_name)s -endian little -speed 1000 -port 3333 -swoport 2332 -telnetport 2333 -vd -ir -localhostonly 1 -singlerun -strict -timeout 0 -nogui -silent' % {
    serial_number = sn,
    chip_name = T.getFullChipName()
  }
end

function T.getNonIntrusiveJLinkOptions(serial_number, port)
  local sn = ''
  if serial_number ~= nil then
    sn = sn .. '=%s' % {serial_number}
  end

  return '-select USB%(serial_number)s -device STM32%(chip_name)s -endian little -if SWD -speed 1000 -noir -nohalt -silent -port %(port)s' % {
    serial_number = sn,
    chip_name = T.getFullChipName(),
    port = port
  }
end

return T
