--[[
  Copyright (c) 2021 by Plexim GmbH
  All rights reserved.

  A free license is granted to anyone to use this software for any legal
  non safety-critical purpose, including commercial applications, provided
  that:
  1) IT IS NOT USED TO DIRECTLY OR INDIRECTLY COMPETE WITH PLEXIM, and
  2) THIS COPYRIGHT NOTICE IS PRESERVED in its entirety.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
  OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
  AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
  OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
  SOFTWARE.
--]]

local T = require('STM32')
local U = require('common.utils')

function T.getFamilyPrefix()
  return 'g4'
end

function T.getNucleoBoardPinCount()
  return 64
end

function T.getTargetParameters()
	local params = {
		timebase_tims = {
      G431xx = {
        3, 4
      },
      G441xx = {
        3, 4
      },
      G473xx = {
        3, 4
      },
      G474xx = {
        3, 4
      },
      G483xx = {
        3, 4
      },
      G484xx = {
        3, 4
      },
      G491xx = {
        3, 4
      },
    },
    cpus = {
      G431xx = { 'CM4' },
      G441xx = { 'CM4' },
      G473xx = { 'CM4' },
      G474xx = { 'CM4' },
      G483xx = { 'CM4' },
      G484xx = { 'CM4' },
      G491xx = { 'CM4' },
    },
    nvicAddress = {
      G431xx = { '0x08000000' },
      G441xx = { '0x08000000' },
      G473xx = { '0x08000000' },
      G474xx = { '0x08000000' },
      G483xx = { '0x08000000' },
      G484xx = { '0x08000000' },
      G491xx = { '0x08000000' },
    },
		min_hrtim_input_freq = 100000000,
		usarts = {
      tx_gpio = {A9=1, B6=1, C4=1, E0=1, G9=1, A2=2, A14=2, B3=2, D5=2, B9=3, B10=3, C10=3, D8=3, E15=3},
      rx_gpio = {A10=1, B7=1, C5=1, E1=1, A3=2, A15=2, B4=2, D6=2, B8=3, B11=3, C11=3, D9=3},
		},
    max_tasks = {
      G431xx = {
        rtos = 16,
        baremetal = 6
      },
      G441xx = {
        rtos = 16,
        baremetal = 6
      },
      G473xx = {
       rtos = 16,
       baremetal = 6
      },
      G474xx = {
       rtos = 16,
       baremetal = 6
      },
      G483xx = {
       rtos = 16,
       baremetal = 6
      },
      G484xx = {
       rtos = 16,
       baremetal = 6
      },
      G491xx = {
       rtos = 16,
       baremetal = 6
      },
    },
    dma = {
      irq = 'Channel',
      G431xx = {
        ADC1 = {
          unit = 1,
          channel = 1
        },
        ADC2 = {
          unit = 1,
          channel = 2
        }
      },
      G441xx = {
        ADC1 = {
          unit = 1,
          channel = 1
        },
        ADC2 = {
          unit = 1,
          channel = 2
        }
      },
      G473xx = {
        ADC1 = {
          unit = 1,
          channel = 1
        },
        ADC2 = {
          unit = 1,
          channel = 2
        },
        ADC3 = {
          unit = 1,
          channel = 3
        },
        ADC4 = {
          unit = 1,
          channel = 4
        },
        ADC5 = {
          unit = 1,
          channel = 5
        }
      },
      G474xx = {
        ADC1 = {
          unit = 1,
          channel = 1
        },
        ADC2 = {
          unit = 1,
          channel = 2
        },
        ADC3 = {
          unit = 1,
          channel = 3
        },
        ADC4 = {
          unit = 1,
          channel = 4
        },
        ADC5 = {
          unit = 1,
          channel = 5
        }
      },
      G483xx = {
        ADC1 = {
          unit = 1,
          channel = 1
        },
        ADC2 = {
          unit = 1,
          channel = 2
        },
        ADC3 = {
          unit = 1,
          channel = 3
        },
        ADC4 = {
          unit = 1,
          channel = 4
        },
        ADC5 = {
          unit = 1,
          channel = 5
        }
      },
      G484xx = {
        ADC1 = {
          unit = 1,
          channel = 1
        },
        ADC2 = {
          unit = 1,
          channel = 2
        },
        ADC3 = {
          unit = 1,
          channel = 3
        },
        ADC4 = {
          unit = 1,
          channel = 4
        },
        ADC5 = {
          unit = 1,
          channel = 5
        }
      },
      G491xx = {
        ADC1 = {
          unit = 1,
          channel = 1
        },
        ADC2 = {
          unit = 1,
          channel = 2
        },
        ADC3 = {
          unit = 1,
          channel = 3
        },
        ADC4 = {
          unit = 1,
          channel = 4
        },
        ADC5 = {
          unit = 1,
          channel = 5
        }
      },
    },
    internalAdcChannels = {
      ADC1 = {
        IN13 = {port = ''}, -- Vopamp1
        IN16 = {port = ''}, -- Vts
        IN17 = {port = ''}, -- Vbat/3
        IN18 = {port = ''} -- Vrefint
      },
      ADC2 = {
        IN16 = {port = ''}, -- Vopamp2
        IN18 = {port = ''}, -- Vopamp3
      },
      ADC3 = {
        IN13 = {port = ''}, -- Vopamp3
        IN18 = {port = ''}, -- Vrefint
      },
      ADC4 = {
        IN17 = {port = ''}, -- Vopamp6
        IN18 = {port = ''}, -- Vrefint
      },
      ADC5 = {
        IN3 = {port = ''}, -- Vopamp5
        IN4 = {port = ''}, -- Vts
        IN5 = {port = ''}, -- Vopamp4
        IN17 = {port = ''}, -- Vbat/3
        IN18 = {port = ''}, -- Vrefint
      }
    },
    adcs = {
      injChLimit = 4,
      regChLimit = 16,
      m_injIrqName = {
        -- [ADC unit] = IRQ macro
        [1] = 'ADC1_2_IRQ',
        [2] = 'ADC1_2_IRQ',
        [3] = 'ADC3_IRQ',
        [4] = 'ADC4_IRQ',
        [5] = 'ADC5_IRQ',
      },
    },
		dacs = {
		  DAC1 = {
		    max_samples_per_second = 3000000, 
		  },
      DAC2 = {
        max_samples_per_second = 3000000, 
      },
      DAC3 = {
        max_samples_per_second = 15000000, 
      },
      DAC4 = {
        max_samples_per_second = 15000000, 
      }
		},
    comps = {
      COMP1 = {
        dac_in_n = {DAC3_CH1=4, DAC1_CH1=5}, 
        hrtim_eev = {4, 6},
        hrtim_flt = 4
      },
      COMP2 = {
        dac_in_n = {DAC3_CH2=4, DAC1_CH2=5},
        hrtim_eev = {1, 6},
        hrtim_flt = 1
      },
      COMP3 = {
        dac_in_n = {DAC3_CH1=4, DAC1_CH1=5},
        hrtim_eev = {5, 8},
        hrtim_flt = 5
      },
      COMP4 = {
        dac_in_n = {DAC3_CH2=4, DAC1_CH1=5},
        hrtim_eev = {2, 7, 9},
        hrtim_flt = 2
      },
      COMP5 = {
        dac_in_n = {DAC4_CH1=4, DAC1_CH2=5},
        hrtim_eev = {9, 4},
        hrtim_flt = 6
      },
      COMP6 = {
        dac_in_n = {DAC4_CH2=4, DAC2_CH1=5},
        hrtim_eev = {3, 8},
        hrtim_flt = 3
      },      
      COMP7 = {
        dac_in_n = {DAC4_CH1=4, DAC2_CH1=5},
        hrtim_eev = {10, 5}
      },
    },
    opamp = {
      OPAMP1 = {
        dac_in_p = {DAC3_CH1=1}, 
      },
      OPAMP2 = {
        dac_in_p = {},
      },
      OPAMP3 = {
        dac_in_p = {DAC3_CH2=1},
      },
      OPAMP4 = {
        dac_in_p = {DAC4_CH1=1},
      },
      OPAMP5 = {
        dac_in_p = {DAC4_CH2=1},
      },
      OPAMP6 = {
        dac_in_p = {DAC3_CH1=1},
      }, 
    },
    can = {
      nominal = {
        prescaler = 512,
        sjw = 128,
        tseg1Min = 2,
        tseg1Max = 256,
        tseg2Min = 2,
        tseg2Max = 128
      },
      data = {
        prescaler = 32,
        sjw = 16,
        tseg1Min = 1,
        tseg1Max = 32,
        tseg2Min = 1,
        tseg2Max = 16
      },
      ssp = {
        offset = 127,
        filter = 127
      }
    },
    pwm = {
      supports_var_dead_time = true,
      irq = {
        [1] = 'TIM1_UP_TIM16',
        [2] = 'TIM2',
        [3] = 'TIM3',
        [4] = 'TIM4',
        [5] = 'TIM5',
        [8] = 'TIM8_UP',
        [20] = 'TIM20_UP'
      },
      master_priority = {1, 8, 20},
      TIM1 = {
        itr = 0
      },
      TIM8 = {
        itr = 5
      },
      TIM20 = {
        itr = 9
      }
    },
    hrtims = {
      carrier = {
        'sawtooth', 'symmetrical'
      },
      clk_limits = {
        input_clk_mul = 32,
        prescale_min = 1,
        prescale_max = 128
      },
      fault_inputs = {
        eev1 = 1,
        eev2 = 2,
        eev3 = 3,
        eev4 = 4,
        eev5 = 5,
        eev6 = 6
      },
    },
    spis = {
      SPI1 = {
        fifo_depth = 2,
        apb = 2,
        --sck_gpio = { A5 = 0, B3 = 1, G2 = 2 },
        --miso_gpio = { A6 = 0, B4 = 1, G3 = 2 },
        --mosi_gpio = { A7 = 0, B5 = 1, G4 = 2 },
        --nss_gpio = { A4 = 0, A15 = 1, G5 = 2 },
      },
      SPI2 = {
        fifo_depth = 2,
        apb = 1,
        --sck_gpio = { B13 = 0, F1 = 1, F9 = 2, F10 = 3 },
        --miso_gpio = { A10 = 0, B14 = 1 },
        --mosi_gpio = { A11 = 0, B15 = 1 },
        --nss_gpio = { B12 = 0, D15 = 1, F0 = 2 },
      },
      SPI3 = {
        fifo_depth = 2,
        apb = 1,
        --sck_gpio = { B3 = 0, C10 = 1, G9 = 2 },
        --miso_gpio = { B4 = 0, C11 = 1 },
        --mosi_gpio = { B5 = 0, C12 = 1 },
        --nss_gpio = {A4 = 0, A15 = 1 },
      },
      SPI4 = {
        fifo_depth = 2,
        apb = 2,
        --sck_gpio = { E2 = 0, E12 = 1 },
        --miso_gpio = { E5 = 0, E13 = 1 },
        --mosi_gpio = { E6 = 0, E14 = 1 },
        --nss_gpio = { E3 = 0, E4 = 1, E11 = 2 },
      }
    },
    gpio_location = {
      A0 = { pin = 28, con = 'CN7' },
      A1 = { pin = 30, con = 'CN7' },
      A2 = { pin = 35, con = 'CN10' },
      A3 = { pin = 37, con = 'CN10' },
      A4 = { pin = 32, con = 'CN7' },
      A5 = { pin = 11, con = 'CN10' },   
      A6 = { pin = 13, con = 'CN10' },
      A7 = { pin = 15, con = 'CN10' },
      A8 = { pin = 23, con = 'CN10' }, 
      A9 = { pin = 21, con = 'CN10' }, 
      A10 = { pin = 33, con = 'CN10' }, 
      A11 = { pin = 14, con = 'CN10' }, 
      A12 = { pin = 12, con = 'CN10' }, 
      A13 = { pin = 13, con = 'CN7' },
      A14 = { pin = 15, con = 'CN7' },
      A15 = { pin = 17, con = 'CN7' },
     
      B0 = { pin = 34, con = 'CN7' },
      B1 = { pin = 24, con = 'CN10' },
      B2 = { pin = 22, con = 'CN10' },
      B3 = { pin = 31, con = 'CN10' },
      B4 = { pin = 27, con = 'CN10' },
      B5 = { pin = 29, con = 'CN10' },
      B6 = { pin = 17, con = 'CN10' },
      B7 = { pin = 21, con = 'CN7' },
      B8 = { pin = 3, con = 'CN10' }, 
      B9 = { pin = 5, con = 'CN10' }, 
      B10 = { pin = 25, con = 'CN10' },
      B11 = { pin = 18, con = 'CN10' },
      B12 = { pin = 16, con = 'CN10' }, 
      B13 = { pin = 30, con = 'CN10' },
      B14 = { pin = 28, con = 'CN10' }, 
      B15 = { pin = 26, con = 'CN10' },
    
      C0 = { pin = 38, con = 'CN7' },
      C1 = { pin = 36, con = 'CN7' },
      C2 = { pin = 35, con = 'CN7' },
      C3 = { pin = 37, con = 'CN7' },
      C4 = { pin = 34, con = 'CN10' },
      C5 = { pin = 6, con = 'CN10' },
      C6 = { pin = 4, con = 'CN10' }, 
      C7 = { pin = 19, con = 'CN10' },
      C8 = { pin = 2, con = 'CN10' }, 
      C9 = { pin = 1, con = 'CN10' }, 
      C10 = { pin = 1, con = 'CN7' },
      C11 = { pin = 2, con = 'CN7' },
      C12 = { pin = 3, con = 'CN7' },
      C13 = { pin = 23, con = 'CN7' },
      C14 = { pin = 25, con = 'CN7' },
      C15 = { pin = 27, con = 'CN7' },
      
      D0 = {  },
      D1 = {  },
      D2 = { pin = 4, con = 'CN7' },
      D3 = {  },
      D4 = {  },
      D5 = {  },
      D6 = {  },
      D7 = {  },
      D8 = {  },
      D9 = {  },
      D10 = {  },
      D11 = {  },
      D12 = {  },
      D13 = {  },
      D14 = {  },
      D15 = {  },
      
      E0 = {  },
      E1 = {  },
      E2 = {  },
      E3 = {  },
      E4 = {  },
      E5 = {  },
      E6 = {  },
      E7 = {  },
      E8 = {  },
      E9 = {  },
      E10 = {  },
      E11 = {  },
      E12 = {  },
      E13 = {  },
      E14 = {  },
      E15 = {  },
      
      F0 = { pin = 29, con = 'CN7' },
      F1 = { pin = 31, con = 'CN7' }, 
    }
	}

  T.populateMcuTargetResources(params)
	return params
end

function T.addStaticInformationToPinmap(pinmap)
  pinmap[3][1] = 'static:VDD'
  pinmap[4][1] = 'static:BOOT0'
  pinmap[5][1] = 'static:NC'
  pinmap[6][1] = 'static:NC'
  pinmap[10][1] = 'static:GND'
  pinmap[17][1] = 'static:VBAT'

  pinmap[3][2] = 'static:E5V'
  pinmap[4][2] = 'static:GND'
  pinmap[5][2] = 'static:NC'
  pinmap[6][2] = 'static:IOREF'
  pinmap[7][2] = 'static:RESET'
  pinmap[8][2] = 'static:+3V3'
  pinmap[9][2] = 'static:+5V'
  pinmap[10][2] = 'static:GND'
  pinmap[11][2] = 'static:GND'
  pinmap[12][2] = 'static:VIN'
  pinmap[13][2] = 'static:NC'

  pinmap[4][3] = 'static:AVDD'
  pinmap[5][3] = 'static:GND'

  pinmap[4][4] = 'static:U5V'
  pinmap[5][4] = 'static:NC'
  pinmap[10][4] = 'static:GND'
  pinmap[16][4] = 'static:AGND'
  pinmap[18][4] = 'static:NC'
  pinmap[19][4] = 'static:NC'
end

function T.getInternalAdcChannels()
  return T.getTargetParameters()['internalAdcChannels']
end

function T.getAvailableInjAdcTrg(hrtim)
  local trg = {5, 6, 7, 8, 9, 10}
  return trg[hrtim]
end

function T.getAvailableRegAdcTrg(hrtim)
  local trg = {5, 6, 7, 8, 9, 10}
  return trg[hrtim]
end

function T.getTimerETRSourceInitCode()
  return 'LL_TIM_SetETRSource(handle, LL_TIM_TIM1_ETRSOURCE_GPIO);'
end

function T.getTimerToRegAdcTrigString(tim, adc)
  return 'LL_ADC_REG_TRIG_EXT_TIM%s_TRGO' % {tim}
end

function T.getTimerToInjAdcTrigString(tim, adc)
  return 'LL_ADC_INJ_TRIG_EXT_TIM%s_TRGO' % {tim}
end

function T.adcGetRealAcqTime(targetAcqTime)
  local targetClkCycles = targetAcqTime * T.getAdcClock()
  local clkCycles = 640.5
  local m_clkCycles = '640CYCLES_5'
  if targetClkCycles <= 2.5 then
    clkCycles = 2.5
    m_clkCycles = '2CYCLES_5'
  elseif targetClkCycles <= 6.5 then
    clkCycles = 6.5
    m_clkCycles = '6CYCLES_5'
  elseif targetClkCycles <= 12.5 then
    clkCycles = 12.5
    m_clkCycles = '12CYCLES_5'
  elseif targetClkCycles <= 24.5 then
    clkCycles = 24.5
    m_clkCycles = '24CYCLES_5'
  elseif targetClkCycles <= 47.5 then
    clkCycles = 47.5
    m_clkCycles = '47CYCLES_5'
  elseif targetClkCycles <= 92.5 then
    clkCycles = 92.5
    m_clkCycles = '92CYCLES_5'
  elseif targetClkCycles <= 247.5 then
    clkCycles = 247.5
    m_clkCycles = '247CYCLES_5'
  end
  return {
    clkCycles = clkCycles,
    m_clkCycles = m_clkCycles
  }
end

function T.getHrtimInputClkHz()
  return T.getCleanSysClkHz()
end

function T.getDacUpdateRate(dac, slope, senseGain, clk, period, tol)
  local dvdt = slope * senseGain
  local minSteps = 10 -- we require at least 10 points per period
  local counts_dt = 0x10000/Target.Variables.AdcVRef * dvdt
  local maxRate = T.getTargetParameters().dacs['DAC%d' % {dac}].max_samples_per_second
  local compVal = math.ceil(clk/maxRate)
  if compVal == 0 then
    compVal = 1
  elseif compVal >= (period/minSteps) then
    
    U.error('DAC %d too slow for ramp generation.' % dac)
  end
  
  local rate, step, reler
  while compVal < (period/minSteps) do
    rate = clk/compVal  
    step = math.floor(counts_dt/rate+0.5)
    local dv_dt_actual = step*Target.Variables.AdcVRef/0x10000*rate
    local dv_dt_error = math.abs(dvdt - dv_dt_actual)
    reler = (dv_dt_error/dvdt)
    if reler < tol then
      break
    end
    compVal = compVal + 1
  end
  if reler > tol then
    U.error('Rate %f, counts_dt %f, step %d, error %f' % {rate, counts_dt, step, reler})
  end
  return rate
end

function T.getBreakInputConfigCode(handle, params)
  local code = [[
    TIM_TypeDef* handle = PLX_TIM_getStmLLHandle(%(handle)s);  
    LL_TIM_SetBreakInputSourcePolarity(handle, LL_TIM_BREAK_INPUT_BKIN, LL_TIM_BKIN_SOURCE_BKIN, LL_TIM_BKIN_POLARITY_%(m_polarity)s);
  ]] % {
    handle = handle,
    m_polarity = (params.polarity == 1) and 'HIGH' or 'LOW',
  }
  return code   
end

function T.checkBreakInputPolarity(polarity)
  -- dummy implementation since this check is needed on other targets
  return
end

function T.getBreakAndDeadtimeConfigCode(handle, params)
  local analogFltSignalCode = ''
  if params.bk2 then
    for _, flt in ipairs(params.analogFltSignals) do
      analogFltSignalCode = [[
        LL_TIM_EnableBreakInputSource(handle, LL_TIM_BREAK_INPUT_BKIN2, LL_TIM_BKIN_SOURCE_BKCOMP%d);
      ]] % {flt.comp_unit}
    end
  end

  if not params.dead_time then
    params.dead_time = 0.0
  end
  local effectiveDeadTime = T.calcDTGValue(params.dead_time)

  local code = [[    
    TIM_TypeDef* handle = PLX_TIM_getStmLLHandle(%(handle)s);

    LL_TIM_BDTR_InitTypeDef initStruct = {0};
    initStruct.OSSRState = LL_TIM_OSSR_ENABLE;
    initStruct.OSSIState = LL_TIM_OSSI_ENABLE;
    initStruct.LockLevel = LL_TIM_LOCKLEVEL_OFF;
    initStruct.DeadTime = 0x%(dtg)X; // %(dtg_us)f us
    initStruct.BreakState = LL_TIM_BREAK_%(m_bk1)s;
    initStruct.BreakPolarity = LL_TIM_BREAK_POLARITY_HIGH;
    initStruct.BreakFilter = LL_TIM_BREAK_FILTER_FDIV1;
    initStruct.BreakAFMode = LL_TIM_BREAK_AFMODE_INPUT;
    initStruct.Break2State = LL_TIM_BREAK2_%(m_bk2)s;
    initStruct.Break2Polarity = LL_TIM_BREAK2_POLARITY_HIGH;
    initStruct.Break2Filter = LL_TIM_BREAK2_FILTER_FDIV1;
    initStruct.Break2AFMode = LL_TIM_BREAK_AFMODE_INPUT;
    initStruct.AutomaticOutput = LL_TIM_AUTOMATICOUTPUT_DISABLE;
    LL_TIM_BDTR_Init(handle, &initStruct);

    %(analogFltSignalCode)s
  ]] % {
    handle = handle,
    dtg = effectiveDeadTime.dtg,
    dtg_us = effectiveDeadTime.dt * 1e6,
    m_bk1 = params.bk and 'ENABLE' or 'DISABLE',
    m_bk2 = params.bk2 and 'ENABLE' or 'DISABLE',
    analogFltSignalCode = analogFltSignalCode,
  }

  if params.dead_time_is_variable then
    code = code .. [[
      LL_TIM_EnableDeadTimePreload(handle);
      LL_TIM_EnableAsymmetricalDeadTime(handle);
    ]]
    if params.const_dead_time.rising then
      local dtg = T.calcDTGValue(params.const_dead_time.rising).dtg
      code = code .. 'LL_TIM_OC_SetDeadTime(handle, 0x%X);' % {dtg}
    end
    if params.const_dead_time.falling then
      local dtg = T.calcDTGValue(params.const_dead_time.falling).dtg
      code = code .. 'LL_TIM_SetFallingDeadTime(handle, 0x%X);' % {dtg}
    end
  end

  return code   
end

function T.getHrtimExternalEventSetupCode(hrtim, params)
  local code = U.CodeLines:new()
  
  code:append([[
    LL_HRTIM_EE_SetPrescaler(%(m_hrtim)s, LL_HRTIM_EE_PRESCALER_DIV1);
  ]] % {
    m_hrtim = 'HRTIM%d' % {hrtim},
  })

  for event, conf in pairs(params.events) do
    code:append([[
      LL_HRTIM_EE_SetSrc(%(m_hrtim)s, LL_HRTIM_EVENT_%(event)d, LL_HRTIM_EEV%(event)dSRC_%(m_evtSrc)s);
      LL_HRTIM_EE_SetPolarity(%(m_hrtim)s, LL_HRTIM_EVENT_%(event)d, LL_HRTIM_EE_POLARITY_%(m_polarity)s);
      LL_HRTIM_EE_SetSensitivity(%(m_hrtim)s, LL_HRTIM_EVENT_%(event)d, LL_HRTIM_EE_SENSITIVITY_%(m_sensitivity)s);
      LL_HRTIM_EE_SetFastMode(%(m_hrtim)s, LL_HRTIM_EVENT_%(event)d, LL_HRTIM_EE_FASTMODE_%(m_fastMode)s);
    ]] % {
      m_hrtim = 'HRTIM%d' % {hrtim},
      event = event,
      m_evtSrc = conf.src,
      m_sensitivity = conf.sensitivity,
      m_fastMode = conf.enable_fastmode and 'ENABLE' or 'DISABLE',
      m_polarity = (conf.polarity == 0) and 'LOW' or 'HIGH',
    })
  end

  return code
end

-- Provides target-specific code for the function T.getHighResolutionMasterTimerSetupCode(hrtim, params)
function T.ts_getHighResolutionMasterTimerSetupCode(hrtim, params)
  local ret = {}

  if params.adctrig then
    local m_trigEvt
    if params.adctrigType == 'inj' then
      m_trigEvt = 'LL_HRTIM_ADCTRIG_SRC24_MPER' -- trigger for injected conversions
    else 
      m_trigEvt = 'LL_HRTIM_ADCTRIG_SRC13_MPER' -- trigger for regular conversions
    end

    ret.opt_adcTrigCode = [[
      LL_HRTIM_ConfigADCTrig(%(m_hrtim)s, %(m_adcTrig)s, LL_HRTIM_ADCTRIG_UPDATE_MASTER, %(m_trigEvt)s);
      LL_HRTIM_SetADCPostScaler(%(m_hrtim)s, %(m_adcTrig)s, %(rep)d);
    ]] % {
      m_hrtim = 'HRTIM%d' % {hrtim},
      m_adcTrig = (params.adctrigType == 'inj') and 'LL_HRTIM_ADCTRIG_2' or 'LL_HRTIM_ADCTRIG_1', -- injected or regular conversions
      rep = params.repetition and math.max(0, params.repetition-1) or 0,
      m_trigEvt = m_trigEvt,
    }
  end

  if params.mode == 'pcc' then
    if params.repetition then
      ret.opt_enableRepCode = [[
        LL_HRTIM_EnableIT_REP(HRTIM%d, LL_HRTIM_TIMER_MASTER);
      ]] % {hrtim}
    end
  end

  local m_prescaleMacros = {
    [1] = 'LL_HRTIM_PRESCALERRATIO_MUL32',
    [2] = 'LL_HRTIM_PRESCALERRATIO_MUL16',
    [4] = 'LL_HRTIM_PRESCALERRATIO_MUL8',
    [8] = 'LL_HRTIM_PRESCALERRATIO_MUL4',
    [16] = 'LL_HRTIM_PRESCALERRATIO_MUL2',
    [32] = 'LL_HRTIM_PRESCALERRATIO_DIV1',
    [64] = 'LL_HRTIM_PRESCALERRATIO_DIV2',
    [128] = 'LL_HRTIM_PRESCALERRATIO_DIV4',
  }
  ret.m_prescale = m_prescaleMacros[params.prescale]

  ret.opt_interleavedModeCode = [[
    LL_HRTIM_TIM_SetInterleavedMode(HRTIM%d, LL_HRTIM_TIMER_MASTER, LL_HRTIM_INTERLEAVED_MODE_DISABLED);
  ]] % {hrtim}

  ret.m_cmpDefault = '0x60'

  return U.contractLockTable(ret)
end

function T.ts_getHighResolutionSubTimerSetupCode(hrtim, params)
  local ret = {}

  local m_hrtim = 'HRTIM%d' % {hrtim}
  local m_subtim = 'LL_HRTIM_TIMER_%s' % {params.subtimer}

  if not params.repetition or not params.repetition.period or (params.repetition.period == 0) then
    ret.repetitionPeriod = 0
  else
    if params.repetition.m_rollover == 'BOTH' or params.carrier == 'sawtooth' then
      ret.repetitionPeriod = params.repetition.period - 1
    else
      ret.repetitionPeriod = params.repetition.period / 2 - 1
    end
  end

  if params.trigger.is_adc_trigger then
    local adcTrig = 10
    if params.trigger.adc_trigger_unit then
      if params.trigger.adc_trig_type == 'inj' then
        adcTrig = T.getAvailableInjAdcTrg(params.subtimer_enum)
      else 
        adcTrig = T.getAvailableRegAdcTrg(params.subtimer_enum)
      end
    end

    local m_trigSrc
    if (adcTrig == 5) or (adcTrig == 7) or (adcTrig == 9) then
      m_trigSrc = 'SRC579'
    elseif (adcTrig == 6) or (adcTrig == 8) or (adcTrig == 10) then
      m_trigSrc = 'SRC6810'
    else
      error('Invalid trigger source.')
    end

    local m_samplingPoint
    if params.carrier == 'symmetrical' then
      m_samplingPoint = 'PER'
    else
      if params.trigger.sampling_point == 'period' then
        m_samplingPoint = 'PER'
      else
        m_samplingPoint = 'CMP4'
      end
    end

    local opt_adcRolloverCode
    if params.carrier == 'symmetrical' then
      opt_adcRolloverCode = [[
        LL_HRTIM_TIM_SetADCRollOverMode(%(m_hrtim)s,
                                        %(m_subtim)s,
                                        LL_HRTIM_ROLLOVER_MODE_%(m_rolloverMode)s);
      ]] % {
        m_hrtim = m_hrtim,
        m_subtim = m_subtim,
        m_rolloverMode = params.repetition.m_rollover,
      }
    end

    ret.opt_adcTrigCode = [[
      LL_HRTIM_ConfigADCTrig(%(m_hrtim)s,
                             LL_HRTIM_ADCTRIG_%(adcTrig)d,
                             LL_HRTIM_ADCTRIG_UPDATE_TIMER_%(m_subtim)s,
                             LL_HRTIM_ADCTRIG_%(m_trigSrc)s_TIM%(m_subtim)s_%(m_samplingPoint)s);
      LL_HRTIM_SetADCPostScaler(%(m_hrtim)s,
                                LL_HRTIM_ADCTRIG_%(adcTrig)d,
                                %(rep)d);
      %(opt_adcRolloverCode)s
    ]] % {
      m_hrtim = m_hrtim,
      rep = ret.repetitionPeriod,
      adcTrig = adcTrig,
      m_subtim = params.subtimer,
      m_trigSrc = m_trigSrc,
      m_samplingPoint = m_samplingPoint,
      opt_adcRolloverCode = opt_adcRolloverCode or '',
    }
  end

  local m_prescaleMacros = {
    [1] = 'LL_HRTIM_PRESCALERRATIO_MUL32',
    [2] = 'LL_HRTIM_PRESCALERRATIO_MUL16',
    [4] = 'LL_HRTIM_PRESCALERRATIO_MUL8',
    [8] = 'LL_HRTIM_PRESCALERRATIO_MUL4',
    [16] = 'LL_HRTIM_PRESCALERRATIO_MUL2',
    [32] = 'LL_HRTIM_PRESCALERRATIO_DIV1',
    [64] = 'LL_HRTIM_PRESCALERRATIO_DIV2',
    [128] = 'LL_HRTIM_PRESCALERRATIO_DIV4',
  }
  ret.m_prescale = m_prescaleMacros[params.timing.prescale]

  if params.carrier == 'symmetrical' then
    ret.opt_repRolloverCode = [[
      LL_HRTIM_TIM_SetRollOverMode(%(m_hrtim)s,
                                   %(m_subtim)s,
                                   LL_HRTIM_ROLLOVER_MODE_%(m_rolloverMode)s);
    ]] % {
      m_hrtim = m_hrtim,
      m_subtim = m_subtim,
      m_rolloverMode = params.repetition.m_rollover,
    }
  end

  ret.opt_setModeCode = [[
    LL_HRTIM_TIM_SetCountingMode(%(m_hrtim)s, %(m_subtim)s, LL_HRTIM_COUNTING_MODE_%(carrier)s);
    LL_HRTIM_TIM_SetTriggeredHalfMode(%(m_hrtim)s, %(m_subtim)s, LL_HRTIM_TRIGHALF_DISABLED);
  ]] % {
    m_hrtim = m_hrtim,
    m_subtim = m_subtim,
    carrier = (params.carrier == 'sawtooth') and 'UP' or (params.carrier == 'symmetrical') and 'UP_DOWN',
  }

  ret.opt_interleavedModeCode = [[
    LL_HRTIM_TIM_SetInterleavedMode(%(m_hrtim)s, %(m_subtim)s, LL_HRTIM_INTERLEAVED_MODE_DISABLED);
  ]] % {
    m_hrtim = m_hrtim,
    m_subtim = m_subtim,
  }

  ret.opt_disableResyncCode = [[
    LL_HRTIM_TIM_DisableResyncUpdate(%(m_hrtim)s, %(m_subtim)s);
  ]] % {
    m_hrtim = m_hrtim,
    m_subtim = m_subtim,
  }

  if params.mode == 'pcc' then
    ret.opt_dacSyncCode = [[
      LL_HRTIM_TIM_SetDualDacResetTrigger(%(m_hrtim)s,
                                          %(m_subtim)s,
                                          LL_HRTIM_DCDR_COUNTER);
      LL_HRTIM_TIM_SetDualDacStepTrigger(%(m_hrtim)s,
                                         %(m_subtim)s,
                                         LL_HRTIM_DCDS_CMP2); // CMP2 in special mode!
      LL_HRTIM_TIM_EnableDualDacTrigger(%(m_hrtim)s,
                                        %(m_subtim)s);
    ]] % {
      m_hrtim = m_hrtim,
      m_subtim = m_subtim,
    }
  end

  ret.setCompareCode = [[
    LL_HRTIM_TIM_SetComp1Mode(%(m_hrtim)s, %(m_subtim)s, LL_HRTIM_GTCMP1_EQUAL);
    LL_HRTIM_TIM_SetCompare1(%(m_hrtim)s, %(m_subtim)s, %(m_cmp1)s);
    LL_HRTIM_TIM_SetComp3Mode(%(m_hrtim)s, %(m_subtim)s, LL_HRTIM_GTCMP3_EQUAL);
    LL_HRTIM_TIM_SetCompare3(%(m_hrtim)s, %(m_subtim)s, %(m_cmp3)s);
    LL_HRTIM_TIM_SetCompare2(%(m_hrtim)s, %(m_subtim)s, %(m_cmp2)s);
    LL_HRTIM_TIM_SetCompare4(%(m_hrtim)s, %(m_subtim)s, %(m_cmp4)s);
  ]] % {
    m_hrtim = m_hrtim,
    m_subtim = m_subtim,
    m_cmp1 = params.cmp_init and tostring(params.cmp_init[1]) or '0',
    m_cmp2 = params.cmp_init and tostring(params.cmp_init[2]) or '0',
    m_cmp3 = params.cmp_init and tostring(params.cmp_init[3]) or '0',
    m_cmp4 = params.cmp_init and tostring(params.cmp_init[4]) or '0x60',
  }

  return U.contractLockTable(ret)
end

function T.getHrtimOutputSetSrcMacros(params)
  U.enforceParamContract(
    params,
    {
      set = {U.isArrayOf, U.isString},
      opt_cmp = U.isNonNegativeIntScalar,
      carrier = U.isString,
    })

  local m_setSourcesList = {}
  for _, s in ipairs(params.set) do
    local m_setSource = 'LL_HRTIM_OUTPUTSET_%s' % {s}
    if params.opt_cmp and (params.carrier == 'sawtooth') then
      m_setSource = m_setSource .. tostring(params.opt_cmp)
    end
    table.insert(m_setSourcesList, m_setSource)
  end

  return table.concat(m_setSourcesList, ' | ')
end

function T.getHrtimOutputResetSrcMacros(params)
  U.enforceParamContract(
    params,
    {
      reset = {U.isArrayOf, U.isString},
    })

  local m_resetSourcesList = {}
  for _, r in ipairs(params.reset) do
    table.insert(m_resetSourcesList, 'LL_HRTIM_OUTPUTRESET_%s' % {r})
  end

  return table.concat(m_resetSourcesList, ' | ')
end

function T.getHrtimFaultLineSetupCode(hrtim, params)
  local code = [[
    LL_HRTIM_FLT_SetCounterThreshold(%(m_hrtim)s, LL_HRTIM_FAULT_%(flt_line)d, 0);
    LL_HRTIM_FLT_SetResetMode(%(m_hrtim)s, LL_HRTIM_FAULT_%(flt_line)d, LL_HRTIM_FLT_COUNTERRST_CONDITIONAL);
    LL_HRTIM_FLT_DisableBlanking(%(m_hrtim)s, LL_HRTIM_FAULT_%(flt_line)d);
    LL_HRTIM_FLT_SetSrc(%(m_hrtim)s, LL_HRTIM_FAULT_%(flt_line)d, LL_HRTIM_FLT_SRC_%(m_fltSrc)s);
    LL_HRTIM_FLT_SetPolarity(%(m_hrtim)s, LL_HRTIM_FAULT_%(flt_line)d, LL_HRTIM_FLT_POLARITY_%(m_fltPolarity)s);
    LL_HRTIM_FLT_SetFilter(%(m_hrtim)s, LL_HRTIM_FAULT_%(flt_line)d, LL_HRTIM_FLT_FILTER_NONE);
    LL_HRTIM_FLT_Enable(%(m_hrtim)s, LL_HRTIM_FAULT_%(flt_line)d);
  ]] % {
    m_hrtim = 'HRTIM%d' % {hrtim},
    flt_line = params.flt_line,
    m_fltPolarity = (params.polarity == 1) and 'LOW' or 'HIGH',
    m_fltSrc = (params.src == 'on_chip') and 'INTERNAL' or 'EEVINPUT',
  }

  return code
end

function T.getAdcUnitSetupCode(adcUnit, params)
  U.enforceParamContract(
    params,
    {
      injChannels = U.isTable,
      nRegInputCh = U.isNonNegativeIntScalar,
      m_injTrgSrc = U.isString,
      m_regTrgSrc = U.isString,
    })

  local injChLimit = T.getTargetParameters().adcs.injChLimit
  if #params.injChannels > injChLimit then
    error('The number of injected channels surpasses the limit.')
  end

  local m_injSeqRanks
  if #params.injChannels <= 1 then
    m_injSeqRanks = 'LL_ADC_INJ_SEQ_SCAN_DISABLE'
  else
    m_injSeqRanks = 'LL_ADC_INJ_SEQ_SCAN_ENABLE_%dRANKS' % {#params.injChannels}
  end

  local m_reqSeqRanks
  if params.nRegInputCh <= 1 then
    m_regSeqRanks = 'LL_ADC_REG_SEQ_SCAN_DISABLE'
  else
    m_regSeqRanks = 'LL_ADC_REG_SEQ_SCAN_ENABLE_%dRANKS' % {params.nRegInputCh}
  end

  local code = [[
    LL_ADC_InitTypeDef adcInitStruct = {0};
    LL_ADC_REG_InitTypeDef adcRegInitStruct = {0};
    LL_ADC_INJ_InitTypeDef adcInjInitStruct = {0};
    
    adcInitStruct.Resolution = LL_ADC_RESOLUTION_12B;
    adcInitStruct.DataAlignment = LL_ADC_DATA_ALIGN_RIGHT;
    adcInitStruct.LowPowerMode = LL_ADC_LP_MODE_NONE;
    LL_ADC_Init(ADC%(adcUnit)d, &adcInitStruct);
    
    adcRegInitStruct.TriggerSource = %(m_regTrgSrc)s;
    adcRegInitStruct.SequencerLength = %(m_regSeqRanks)s;
    adcRegInitStruct.SequencerDiscont = LL_ADC_REG_SEQ_DISCONT_DISABLE;
    adcRegInitStruct.ContinuousMode = LL_ADC_REG_CONV_%(m_regConvMode)s;
    adcRegInitStruct.DMATransfer = LL_ADC_REG_DMA_TRANSFER_UNLIMITED;
    adcRegInitStruct.Overrun = LL_ADC_REG_OVR_DATA_OVERWRITTEN;
    LL_ADC_REG_Init(ADC%(adcUnit)d, &adcRegInitStruct);
    
    LL_ADC_SetGainCompensation(ADC%(adcUnit)d, 0);
    
    LL_ADC_SetOverSamplingScope(ADC%(adcUnit)d, LL_ADC_OVS_DISABLE); // LL_ADC_OVS_GRP_INJECTED
    
    adcInjInitStruct.TriggerSource = %(m_injTrigSrc)s;
    adcInjInitStruct.SequencerLength = %(m_injSeqRanks)s;
    adcInjInitStruct.SequencerDiscont = LL_ADC_INJ_SEQ_DISCONT_DISABLE;
    adcInjInitStruct.TrigAuto = LL_ADC_INJ_TRIG_INDEPENDENT;
    LL_ADC_INJ_Init(ADC%(adcUnit)d, &adcInjInitStruct);
    LL_ADC_INJ_SetQueueMode(ADC%(adcUnit)d, LL_ADC_INJ_QUEUE_DISABLE);

    LL_ADC_DisableDeepPowerDown(ADC%(adcUnit)d);
    LL_ADC_EnableInternalRegulator(ADC%(adcUnit)d);
    {
      uint32_t wait_loop_index;
      wait_loop_index = ((LL_ADC_DELAY_INTERNAL_REGUL_STAB_US * (SystemCoreClock / (100000 * 2))) / 10);
      while(wait_loop_index != 0)
      {
        wait_loop_index--;
      }
    }
  ]] % {
    adcUnit = adcUnit,
    m_regTrgSrc = params.m_regTrgSrc,
    m_injTrigSrc = params.m_injTrgSrc,
    m_injSeqRanks = m_injSeqRanks,
    m_regSeqRanks = m_regSeqRanks,
    m_regConvMode = (params.m_regTrgSrc == 'LL_ADC_REG_TRIG_SOFTWARE') and 'CONTINUOUS' or 'SINGLE',
  }

  return code
end

function T.getAdcRegChSetupCode(adcUnit, channel, params)
  local code = [[
    LL_ADC_REG_SetSequencerRanks(ADC%(adcUnit)d, LL_ADC_REG_RANK_%(rank)d, LL_ADC_CHANNEL_%(ch)d);
    LL_ADC_SetChannelSamplingTime(ADC%(adcUnit)d, LL_ADC_CHANNEL_%(ch)d, LL_ADC_SAMPLINGTIME_%(m_tsConfig)s);
    LL_ADC_SetChannelSingleDiff(ADC%(adcUnit)d, LL_ADC_CHANNEL_%(ch)d, LL_ADC_SINGLE_ENDED);
  ]] % {
    adcUnit = adcUnit,
    ch = channel,
    rank = params.rank,
    m_tsConfig = params.m_acqTimeClkCycles,
  }

  return code
end

function T.getAdcInjChSetupCode(adcUnit, channel, params)
  local code = [[
    LL_ADC_INJ_SetSequencerRanks(ADC%(adcUnit)d, LL_ADC_INJ_RANK_%(rank)d, LL_ADC_CHANNEL_%(ch)d);
    LL_ADC_SetChannelSamplingTime(ADC%(adcUnit)d, LL_ADC_CHANNEL_%(ch)d, LL_ADC_SAMPLINGTIME_%(m_tsConfig)s);
    LL_ADC_SetChannelSingleDiff(ADC%(adcUnit)d, LL_ADC_CHANNEL_%(ch)d, LL_ADC_SINGLE_ENDED);
  ]] % {
    adcUnit = adcUnit,
    ch = channel,
    rank = params.rank,
    m_tsConfig = params.m_acqTimeClkCycles,
  }

  return code
end

function T.getAdcInjIrqCode(params)
  U.enforceParamContract(
    params,
    {
      isModTrig = U.isBoolean,
      adcUnit = U.isPositiveIntScalar,
      adcBlockInstance = U.isNonNegativeIntScalar,
      adcTriggerBlockType = U.isString,
      opt_adcPostScaler = U.isNonNegativeIntScalar,
    })

  local opt_modTrigCode
  if params.isModTrig then
    opt_modTrigCode = 'DISPR_dispatch();'
  end

  local code = [=[
    if(LL_ADC_IsActiveFlag_JEOS(ADC%(adcUnit)d) != 0)
    {
      LL_ADC_ClearFlag_JEOS(ADC%(adcUnit)d);
      PLX_ADC_updateInjected(AdcUnitHandles[AdcUnitHandleLookup[%(adcBlockInstance)d]]);
      %(opt_modTrigCode)s
    }
  ]=] % {
    adcUnit = params.adcUnit,
    adcBlockInstance = params.adcBlockInstance,
    opt_modTrigCode = opt_modTrigCode or '',
  }

  return code
end

function T.getDmaBufferDeclaration()
  return '#define DMA_BUFFER'
end

function T.getDacChannelSetupCode(dac, channel, params)
  local waveConfigCode = ''
  if params.waveform_gen then
    local waveParams = params.waveform_gen

    waveConfigCode = [[
      initStruct.WaveAutoGeneration = LL_DAC_WAVE_AUTO_GENERATION_SAWTOOTH;
      initStruct.WaveAutoGenerationConfig = __LL_DAC_FORMAT_SAWTOOTHWAVECONFIG(
        LL_DAC_SAWTOOTH_POLARITY_%(m_polarity)s,
        %(bias)s,
        %(increment)d); 
    ]] % {
      bias = waveParams.bias,
      m_polarity = (waveParams.increment < 0) and 'DECREMENT' or 'INCREMENT',
      increment = math.abs(waveParams.increment),
    }
  end

  local code = [[
    LL_DAC_SetSignedFormat(DAC%(dac)d, LL_DAC_CHANNEL_%(ch)d, LL_DAC_SIGNED_FORMAT_DISABLE);
    
    LL_DAC_InitTypeDef initStruct = {0};
    initStruct.TriggerSource = %(m_trigSrc1)s; 
    initStruct.TriggerSource2 = %(m_trigSrc2)s;

    %(waveConfigCode)s
    initStruct.OutputBuffer = LL_DAC_OUTPUT_BUFFER_%(m_connectGpio)s;
    initStruct.OutputConnection = LL_DAC_OUTPUT_CONNECT_%(m_outputRouting)s;
    initStruct.OutputMode = LL_DAC_OUTPUT_MODE_NORMAL;

    LL_DAC_Init(DAC%(dac)d, LL_DAC_CHANNEL_%(ch)d, &initStruct);
    LL_DAC_EnableTrigger(DAC%(dac)d, LL_DAC_CHANNEL_%(ch)d);
    LL_DAC_DisableDMADoubleDataMode(DAC%(dac)d, LL_DAC_CHANNEL_%(ch)d);
  ]] % {
    dac = dac,
    ch = channel,
    m_trigSrc1 = params.trig_src,
    m_trigSrc2 = params.trig_src2,
    waveConfigCode = waveConfigCode,
    m_connectGpio = params.connect_gpio and 'ENABLE' or 'DISABLE',
    m_outputRouting = params.connect_gpio and 'GPIO' or 'INTERNAL',
  }

  return code
end

function T.getOpampSetupCode(opamp, params)
  local opt_pgaCode
  if params.mode == 'PGA' then
    local m_gainMacros = {
      [2] = 'LL_OPAMP_PGA_GAIN_2_OR_MINUS_1',
      [4] = 'LL_OPAMP_PGA_GAIN_4_OR_MINUS_3',
      [8] = 'LL_OPAMP_PGA_GAIN_8_OR_MINUS_7',
      [16] = 'LL_OPAMP_PGA_GAIN_16_OR_MINUS_15',
      [32] = 'LL_OPAMP_PGA_GAIN_32_OR_MINUS_31',
      [64] = 'LL_OPAMP_PGA_GAIN_64_OR_MINUS_63',
    }
    if not m_gainMacros[params.pga_gain] then
      U.error('PGA gain of %d not available on this target.' % {params.pga_gain})
    end

    opt_pgaCode = [[
      LL_OPAMP_SetPGAGain(OPAMP%(opamp)d, %(m_pgaGain)s);
    ]] % {
      opamp = opamp,
      m_pgaGain = m_gainMacros[params.pga_gain],
    }
  end

  local code = [[
  {
    LL_OPAMP_SetPowerMode(OPAMP%(opamp)d, LL_OPAMP_POWERMODE_NORMAL);
    LL_OPAMP_SetMode(OPAMP%(opamp)d, LL_OPAMP_MODE_FUNCTIONAL);
    LL_OPAMP_SetFunctionalMode(OPAMP%(opamp)d, LL_OPAMP_MODE_%(m_mode)s);
    %(opt_pgaCode)s
    LL_OPAMP_SetInputNonInverting(OPAMP%(opamp)d, LL_OPAMP_INPUT_NONINVERT_%(m_nonInvert)s);
    LL_OPAMP_SetInputInverting(OPAMP%(opamp)d, LL_OPAMP_INPUT_INVERT_%(m_invert)s);
    LL_OPAMP_SetTrimmingMode(OPAMP%(opamp)d, LL_OPAMP_TRIMMING_FACTORY);
    LL_OPAMP_SetInternalOutput(OPAMP%(opamp)d, LL_OPAMP_INTERNAL_OUPUT_%(m_internalOutput)s);
    LL_OPAMP_Enable(OPAMP%(opamp)d);
    // Delay for OPAMP stabilization
    __IO uint32_t wait_loop_index = 0;
    wait_loop_index = ((LL_OPAMP_DELAY_STARTUP_US * (SystemCoreClock / (100000 * 2))) / 10);
    while(wait_loop_index != 0)
    {
      wait_loop_index--;
    }
  }
  ]] % {
    m_invert = params.inn or 'CONNECT_NO',
    m_nonInvert = params.inp or 'CONNECT_NO',
    opamp = opamp,
    m_mode = params.mode,
    m_internalOutput = params.enable_out and 'DISABLED' or 'ENABLED',
    opt_pgaCode = opt_pgaCode or '',
  }

  return code
end

function T.getCompSetupCode(comp, params)
  local code = [[
    LL_COMP_InitTypeDef initStruct = {0};
  
    initStruct.InputPlus = LL_COMP_INPUT_PLUS_%(m_inp)s;
    initStruct.InputMinus = LL_COMP_INPUT_MINUS_%(m_inm)s;
    initStruct.InputHysteresis = LL_COMP_HYSTERESIS_%(m_hyst)s;
    initStruct.OutputPolarity = LL_COMP_OUTPUTPOL_%(m_pol)s;
    initStruct.OutputBlankingSource = LL_COMP_BLANKINGSRC_%(m_source)s;
    LL_COMP_Init(COMP%(comp)d, &initStruct);
    LL_COMP_Enable(COMP%(comp)d);
  ]] % {
    comp = comp,
    m_inm = params.inm,
    m_inp = params.inp,
    m_pol = (params.pol == 1) and 'INVERTED' or 'NONINVERTED',
    m_hyst = params.hyst or 'NONE',
    m_source = params.blanking_src and (params.blanking_src .. '_COMP%d' % {comp}) or 'NONE'
  }

  return code
end

function T.getSysClkPllSettings(clkInHz)
    local use_hsi = false
    local sysClkHz = T.getCleanSysClkHz()
    if clkInHz == nil then
      clkInHz = 16*1e6 -- HSI
      use_hsi = true
    end
    
    if (sysClkHz < 8*1e6) or (sysClkHz > 170*1e6) then
      U.error("The system clock has to be between 8 .. 170 MHz.")
    end 
    
    if (clkInHz < 4*1e6) or (clkInHz > 48*1e6) then
      U.error("The input clock has to be between 4 .. 48 MHz.")
    end 
    
    local m_div = 1
    local n_mul = 8
    local r_div = 2
    local apb1_div = 1  -- For G4 series APB+ prescaler is always 1
    
    local keepSearching = true
    
    m_div = 1
    while (m_div <= 16) and (keepSearching == true) do
      --print('m-div: %d' % m_div)
      local clkM = clkInHz / m_div
      if clkM < 2.66*1e6 then
        keepSearching = false
      elseif clkM <= 16*1e6 then
        n_mul = 8
        while (n_mul <= 127) and (keepSearching == true) do
          --print(' n-mul: %d' % n_mul)
          local clkN = clkM * n_mul
          if clkN > 344*1e6 then
            break
          elseif clkN >= 96*1e6 then
            r_div = 2
            while (r_div <= 8) and (keepSearching == true) do
              --print('  r-div: %d' % r_div)
              local clkR = clkN/r_div
              if clkR == sysClkHz then
                keepSearching = false
              else
                r_div = r_div + 2
              end
            end
          end
          if keepSearching == true then
            n_mul = n_mul + 1
          end
        end
      end
      if keepSearching == true then
        m_div = m_div + 1
      end
    end
    
    local clkM = clkInHz / m_div
    local clkN = clkM * n_mul
    local clkR = clkN/r_div
    
    if clkR ~= sysClkHz then
      U.error("Unable to configure the PLL for the desired system clock.")
    end
    if (clkM < 2.66*1e6) or (clkM > 16*1e6) then
      U.error("PLL algorithm failed (M).")
    end
    if (clkN < 96*1e6) or (clkN > 344*1e6) then
      U.error("PLL algorithm failed (N).")
    end
        
    -- determine flash latency (assumes boost mode above 150 MHz)
    local flash_latency = 4
    if sysClkHz <= 30*1e6 then
      flash_latency = 0
    elseif sysClkHz <= 60*1e6 then
      flash_latency = 1
    elseif sysClkHz <= 90*1e6 then
      flash_latency = 2
    elseif sysClkHz <= 120*1e6 then
      flash_latency = 3
    end
    
    return {
      flash_latency = flash_latency,
      m_div = m_div,
      n_mul = n_mul,
      r_div = r_div,
      apb1_div = apb1_div,
      use_hsi = use_hsi
    }
end

function T.getCpuClock()
  return T.getSysClock()
end

function T.getAPB1Clock()
  return T.getCleanSysClkHz() -- RCC_HCLK_DIV1 hardcoded in the clock configuration code
end

function T.getAPB2Clock()
  return T.getCleanSysClkHz() -- RCC_HCLK_DIV1 hardcoded in the clock configuration code
end

function T.getSysClock()
  return T.getCleanSysClkHz()
end

function T.getTimerClock()
  return T.getCleanSysClkHz()
end

function T.getAdcClock()
  return T.getCleanSysClkHz()
end

function T.getClockConfigurationCode(params)
  local clkEnableCode = ''
  if T.getCleanSysClkHz() > 150e6 then
    clkEnableCode = [[
      // enable voltage range 1 boost mode for frequency above 150 Mhz
      __HAL_RCC_PWR_CLK_ENABLE();
      HAL_PWREx_ControlVoltageScaling(PWR_REGULATOR_VOLTAGE_SCALE1_BOOST);
      __HAL_RCC_PWR_CLK_DISABLE();
    ]]
  end

  local pllActivationCode
  if params.pll_settings.use_hsi then
    pllActivationCode = [[
      // activate PLL with HSI as source
      RCC_OscInitStruct.OscillatorType      = RCC_OSCILLATORTYPE_HSI;
      RCC_OscInitStruct.HSICalibrationValue = RCC_HSICALIBRATION_DEFAULT;
      RCC_OscInitStruct.HSIState            = RCC_HSI_ON;
      RCC_OscInitStruct.PLL.PLLState        = RCC_PLL_ON;
      RCC_OscInitStruct.PLL.PLLSource       = RCC_PLLSOURCE_HSI;
    ]]
  else
    pllActivationCode = [[
      // activate PLL with HSE as source
      RCC_OscInitStruct.OscillatorType = RCC_OSCILLATORTYPE_HSE;
      RCC_OscInitStruct.HSEState = RCC_HSE_ON;
      RCC_OscInitStruct.PLL.PLLState = RCC_PLL_ON;
      RCC_OscInitStruct.PLL.PLLSource = RCC_PLLSOURCE_HSE;
    ]]
  end

  local m_periphClkSelect
  local adcClkSelectCode
  if T.getChipName() == 'G431xx' or T.getChipName() == 'G441xx' then
    adcClkSelectCode = [[
      PeriphClkInit.Adc12ClockSelection = RCC_ADC12CLKSOURCE_PLL;
    ]]
    m_periphClkSelect = 'RCC_PERIPHCLK_RTC|RCC_PERIPHCLK_ADC12|RCC_PERIPHCLK_FDCAN'

  elseif T.getChipName() == 'G473xx' or T.getChipName() == 'G483xx' then
    adcClkSelectCode = [[
      PeriphClkInit.Adc12ClockSelection = RCC_ADC12CLKSOURCE_PLL;
      PeriphClkInit.Adc345ClockSelection = RCC_ADC345CLKSOURCE_PLL;
    ]]
    m_periphClkSelect = 'RCC_PERIPHCLK_RTC|RCC_PERIPHCLK_ADC12|RCC_PERIPHCLK_ADC345|RCC_PERIPHCLK_FDCAN'

  elseif T.getChipName() == 'G474xx' or T.getChipName() == 'G484xx' then
    adcClkSelectCode = [[
      PeriphClkInit.Adc12ClockSelection = RCC_ADC12CLKSOURCE_PLL;
      PeriphClkInit.Adc345ClockSelection = RCC_ADC345CLKSOURCE_PLL;
    ]]
    m_periphClkSelect = 'RCC_PERIPHCLK_RTC|RCC_PERIPHCLK_ADC12|RCC_PERIPHCLK_ADC345|RCC_PERIPHCLK_FDCAN'

  elseif T.getChipName() == 'G491xx' then
    adcClkSelectCode = [[
      PeriphClkInit.Adc12ClockSelection = RCC_ADC12CLKSOURCE_PLL;
      PeriphClkInit.Adc345ClockSelection = RCC_ADC345CLKSOURCE_PLL;
    ]]
    m_periphClkSelect = 'RCC_PERIPHCLK_RTC|RCC_PERIPHCLK_ADC12|RCC_PERIPHCLK_ADC345|RCC_PERIPHCLK_FDCAN'

  else
    error('Target not implemented.')
  end

  local code = [[
    RCC_ClkInitTypeDef RCC_ClkInitStruct = {0};
    RCC_OscInitTypeDef RCC_OscInitStruct = {0};
    RCC_PeriphCLKInitTypeDef PeriphClkInit = {0};

    %(clkEnableCode)s
    %(pllActivationCode)s

    RCC_OscInitStruct.PLL.PLLM            = RCC_PLLM_DIV%(mDiv)d;
    RCC_OscInitStruct.PLL.PLLN            = %(nMul)d;
    RCC_OscInitStruct.PLL.PLLP            = RCC_PLLP_DIV%(pDiv)d;
    RCC_OscInitStruct.PLL.PLLQ            = RCC_PLLQ_DIV%(qDiv)d;
    RCC_OscInitStruct.PLL.PLLR            = RCC_PLLR_DIV%(rDiv)d;
    if(HAL_RCC_OscConfig(&RCC_OscInitStruct) != HAL_OK)
    {
      PLX_ASSERT(0);
    }

    // select PLL as system clock source and configure the HCLK, PCLK1 and PCLK2 clocks dividers */
    RCC_ClkInitStruct.ClockType           = (RCC_CLOCKTYPE_SYSCLK | RCC_CLOCKTYPE_HCLK | \
                                            RCC_CLOCKTYPE_PCLK1 | RCC_CLOCKTYPE_PCLK2);
    RCC_ClkInitStruct.SYSCLKSource        = RCC_SYSCLKSOURCE_PLLCLK;
    RCC_ClkInitStruct.AHBCLKDivider       = RCC_SYSCLK_DIV1;
    RCC_ClkInitStruct.APB1CLKDivider      = RCC_HCLK_DIV1;
    RCC_ClkInitStruct.APB2CLKDivider      = RCC_HCLK_DIV1;
    if(HAL_RCC_ClockConfig(&RCC_ClkInitStruct, FLASH_LATENCY_%(latency)d) != HAL_OK)
    {
      PLX_ASSERT(0);
    }

    // initialize the peripherals clocks
    PeriphClkInit.PeriphClockSelection = %(m_periphClkSelect)s;
    %(adcClkSelectCode)s
    PeriphClkInit.RTCClockSelection = RCC_RTCCLKSOURCE_LSI;
    PeriphClkInit.FdcanClockSelection = RCC_FDCANCLKSOURCE_PCLK1;
    if (HAL_RCCEx_PeriphCLKConfig(&PeriphClkInit) != HAL_OK)
    {
      PLX_ASSERT(0);
    }
  ]] % {
    clkEnableCode = clkEnableCode,
    pllActivationCode = pllActivationCode,
    mDiv = params.pll_settings.m_div,
    nMul = params.pll_settings.n_mul,
    pDiv = params.pll_settings.r_div,
    qDiv = params.pll_settings.r_div,
    rDiv = params.pll_settings.r_div,
    latency = params.pll_settings.flash_latency,
    m_periphClkSelect = m_periphClkSelect,
    adcClkSelectCode = adcClkSelectCode,
  }

  return code
end

function T.getSpiFIFOThresholdCode(spi, params)
  local code = [[
    LL_SPI_SetRxFIFOThreshold(SPI%(spi)d, LL_SPI_RX_FIFO_TH_%(m_rxth)s);
  ]] % {
    spi = spi,
    m_rxth = (params.dataWidth > 8) and 'HALF' or 'QUARTER',
  }

  return code
end

function T.getUartObjectInitializationCode(params)
  local code = [[
    UartHandle = PLX_UART_init(&UartObj, sizeof(UartObj));
    PLX_UART_configure(UartHandle, PLX_USART%(uart)d);
    (void)PLX_UART_setupPort(UartHandle, %(baud)d);
  ]] % {
    uart = params.uart,
    baud = params.baudrate,
  }

  return code
end

function T.getPilFrameworkConfigurationCode(params)
  local opt_jtagCode
  if params.ext_mode == 'jtag' then
    opt_jtagCode = [[
      PIL_configureParallelCom(PilHandle,
                               PARALLEL_COM_PROTOCOL,
                               PARALLEL_COM_BUF_ADDR,
                               PARALLEL_COM_BUF_LEN);
    ]]
  end

  local opt_serialCode
  if params.ext_mode == 'serial' then
    opt_serialCode = [[
      PIL_setSerialComCallback(PilHandle,
                               (PIL_CommCallbackPtr_t)SciPoll);
    ]]
  end

  local preInitCode = [[
    PilHandle = PIL_init(&PilObj, sizeof(PilObj));
    PIL_setGuid(PilHandle, PIL_GUID_PTR);
    PIL_setChecksum(PilHandle, %(baseName)s_checksum);
    PIL_setAndConfigScopeBuffer(PilHandle, (uint16_t *)&ScopeBuffer, %(scopeBufferSize)d, %(scopeMaxTraceWidth)d);
    %(opt_jtagCode)s
    %(opt_serialCode)s
  ]] % {
    baseName = params.base_name,
    scopeBufferSize = params.scope_buffer_size,
    scopeMaxTraceWidth = params.scope_max_trace_width,
    opt_jtagCode = opt_jtagCode or '',
    opt_serialCode = opt_serialCode or '',
  }

  local declarationsCode = ''
  local interruptEnableCode = ''
  return preInitCode, declarationsCode, interruptEnableCode
end

function T.getCanSetupCode(instance, params)
  local timingContract = {
    brp = U.isPositiveIntScalar,
    tseg1 = U.isPositiveIntScalar,
    tseg2 = U.isPositiveIntScalar,
    sjw = U.isPositiveIntScalar,
    baud = U.isPositiveIntScalar,
    samplePoint = U.isPositiveScalar,
  }
  U.enforceParamContract(
    params,
    {
      can = U.isNonNegativeIntScalar,
      isBrsEnabled = U.isBoolean,
      autoBusOnEnabled = U.isBoolean,
      isCanFd = U.isBoolean,
      numStdRxMessages = U.isNonNegativeIntScalar,
      numExtRxMessages = U.isNonNegativeIntScalar,
      timing = {
        nominal = timingContract,
        opt_data = timingContract,
      }
    }
  )

  local initCode = [[
      FDCAN_InitTypeDef hfdcanInit = {
        0
      };
      hfdcanInit.ClockDivider = FDCAN_CLOCK_DIV1;
      hfdcanInit.FrameFormat = %(frameFormat)s;
      hfdcanInit.Mode = FDCAN_MODE_NORMAL;
      hfdcanInit.AutoRetransmission = DISABLE;
      hfdcanInit.TransmitPause = DISABLE;
      hfdcanInit.ProtocolException = DISABLE;
      hfdcanInit.NominalPrescaler = %(nominalPrescaler)d;
      hfdcanInit.NominalSyncJumpWidth = %(nominalSjw)d;
      hfdcanInit.NominalTimeSeg1 = %(nominalTseg1)d;
      hfdcanInit.NominalTimeSeg2 = %(nominalTseg2)d;
      hfdcanInit.StdFiltersNbr = %(numStdMsg)d;
      hfdcanInit.ExtFiltersNbr = %(numExtMsg)d;
      hfdcanInit.TxFifoQueueMode = FDCAN_TX_FIFO_OPERATION;
  ]] % {
    frameFormat = params.isCanFd and (params.isBrsEnabled and 'FDCAN_FRAME_FD_BRS' or 'FDCAN_FRAME_FD_NO_BRS') or 'FDCAN_FRAME_CLASSIC',
    nominalPrescaler = params.timing.nominal.brp,
    nominalSjw = params.timing.nominal.sjw,
    nominalTseg1 = params.timing.nominal.tseg1,
    nominalTseg2 = params.timing.nominal.tseg2,
    numStdMsg = params.numStdRxMessages,
    numExtMsg = params.numExtRxMessages,
  }
  if params.isBrsEnabled then
    local dataInitCode = [[
        hfdcanInit.DataPrescaler = %(dataPrescaler)d;
        hfdcanInit.DataSyncJumpWidth = %(dataSjw)d;
        hfdcanInit.DataTimeSeg1 = %(dataTseg1)d;
        hfdcanInit.DataTimeSeg2 = %(dataTseg2)d;
    ]] % {
      dataPrescaler = params.timing.opt_data.brp,
      dataSjw = params.timing.opt_data.sjw,
      dataTseg1 = params.timing.opt_data.tseg1,
      dataTseg2 = params.timing.opt_data.tseg2,
    }
    initCode = initCode .. dataInitCode
  end

  initCode = initCode .. 'PLX_CANBUS_configure(CanHandles[%d], PLX_CAN_CAN%d, &hfdcanInit);' % {instance, params.can}
  return '{' .. initCode .. '}'
end

function T.getCanTxHeaderSetupCode(canId, params)
  U.enforceParamContract(
    params,
    {
      isExtId = U.isBoolean,
      width = U.isNonNegativeIntScalar,
      isCanFd = U.isBoolean,
      isBrsEnabled = U.isBoolean,
    }
  )
  local initCode = [[
    FDCAN_TxHeaderTypeDef TxHeader = {
      0
    };

    TxHeader.Identifier = %(txId)d;
    TxHeader.IdType = %(frameFormat)s;
    TxHeader.TxFrameType = FDCAN_DATA_FRAME;
    TxHeader.DataLength = FDCAN_DLC_BYTES_%(dataLength)d;
    TxHeader.ErrorStateIndicator = FDCAN_ESI_ACTIVE;
    TxHeader.BitRateSwitch = %(brs)s;
    TxHeader.FDFormat = %(protocol)s;
    TxHeader.TxEventFifoControl = FDCAN_NO_TX_EVENTS;
    TxHeader.MessageMarker = 0;
  ]] % {
    txId = canId,
    frameFormat = params.isExtId and 'FDCAN_EXTENDED_ID' or 'FDCAN_STANDARD_ID',
    dataLength = T.getCanDLC(params.width),
    brs = params.isBrsEnabled and 'FDCAN_BRS_ON' or 'FDCAN_BRS_OFF',
    protocol = params.isCanFd and 'FDCAN_FD_CAN' or 'FDCAN_CLASSIC_CAN',
  }
  return initCode
end

function T.getCanDLC(width)
  local dlc = {12, 16, 20, 24, 32, 48, 64}

  if width <= 8 then
    return width
  else
    for idx = 1, #dlc do
      if width <= dlc[idx] then
        return dlc[idx]
      end
    end
  end
end

function T.getCanTxDelaySetupCode(can, params)
  return [[
    if (HAL_FDCAN_ConfigTxDelayCompensation(PLX_CANBUS_getHandle(CanHandles[%(handle)d]), %(offset)d, %(filter)d) != HAL_OK) {
      PLX_ASSERT(0);
    }
    if (HAL_FDCAN_EnableTxDelayCompensation(PLX_CANBUS_getHandle(CanHandles[%(handle)d])) != HAL_OK) {
      PLX_ASSERT(0);
    }
  ]] % {
    handle = can,
    offset = params.ssp.offset,
    filter = params.ssp.filter,
  } 
end

function T.getCanRxIrqName(can)
  return T.getCanIrqName(can)
end

function T.getCanTxIrqName(can)
  return T.getCanIrqName(can)
end

function T.getCanIrqName(can)
 return 'FDCAN%(can)d_IT0' % {
    can = can,
  }
end

function T.getSendNextMessageFromQueueCode(params)
  return [[
    bool PLX_sendNextMessageFromQueue%(can)d() {
      bool success = false;
      if (!PLX_CANBUS_msgQueueIsEmpty(&canTxMsgQueue%(can)d)) {
        if (HAL_FDCAN_GetTxFifoFreeLevel((FDCAN_HandleTypeDef*)PLX_CANBUS_getHandle(CanHandles[%(instance)d])) > 0) {
          PLX_CAN%(m_canFd)s_TxMessage_Obj_t msg;
          if (PLX_CANBUS_dequeueMsg(&canTxMsgQueue%(can)d, &msg)) {
            PLX_CANBUS_TxHeader_t txHeader;
            txHeader = TxHandles[msg.txHandleIndex];
            if (HAL_FDCAN_AddMessageToTxFifoQ((FDCAN_HandleTypeDef*)PLX_CANBUS_getHandle(CanHandles[%(instance)d]), &txHeader->header, msg.data) == HAL_OK) {
              success = true;
            }
          }
        }
      }
      return success;
    }
  ]] % {
    can = params.can,
    instance = params.instance,
    m_canFd = params.m_canFd,
  }
end

function T.getTxBufferCallbackCode(params)
  return [[
    HAL_FDCAN_RegisterCallback(PLX_CANBUS_getHandle(CanHandles[%(instance)d]), HAL_FDCAN_TX_FIFO_EMPTY_CB_ID, PLX_CANBUS_TxBufferCompleteCallback%(can)d);
    HAL_FDCAN_ActivateNotification(PLX_CANBUS_getHandle(CanHandles[%(instance)d]), FDCAN_IT_TX_FIFO_EMPTY, FDCAN_TX_BUFFER0);
  ]] % {
    can = params.can,
    instance = params.instance,
  }
end

function T.getCanTxCompleteCallbackPrototype(can)
  return 'void PLX_CANBUS_TxBufferCompleteCallback%(can)d(FDCAN_HandleTypeDef* aHandle)' % {
    can = can,
  }
end

function T.getRxFifoCallbackCode(params)
  return [[
    HAL_FDCAN_RegisterRxFifo0Callback(PLX_CANBUS_getHandle(CanHandles[%(instance)d]), PLX_CANBUS_RxCallback%(can)d);
    HAL_FDCAN_ActivateNotification(PLX_CANBUS_getHandle(CanHandles[%(instance)d]), FDCAN_IT_RX_FIFO0_NEW_MESSAGE, 0);
  ]] % {
    can = params.can,
    instance = params.instance,
  }
end

function T.getRxCallbackCode(params)
  return [[
    void PLX_CANBUS_RxCallback%(can)d(FDCAN_HandleTypeDef* aHandle, uint32_t aRxFifoIt);
    void PLX_CANBUS_RxCallback%(can)d(FDCAN_HandleTypeDef* aHandle, uint32_t aRxFifoIt) {
      FDCAN_RxHeaderTypeDef rxHeader;
      uint8_t rxData[%(rxData)d];
      if ((aRxFifoIt & FDCAN_IT_RX_FIFO0_NEW_MESSAGE ) != 0) {
        if (HAL_FDCAN_GetRxMessage(aHandle, FDCAN_RX_FIFO0, &rxHeader, rxData) == HAL_OK) {
          int index = PLX_getIndexFromCanId%(can)d(rxHeader.Identifier);
          if (index != -1) {
            uint32_t interruptState = enterCriticalSection(); // disable interrupts
            if (canMessageArray%(can)d[index].valid) {
              canMessageArray%(can)d[index].overwritten = 1;
            }
            // Store the message in the message array
            canMessageArray%(can)d[index].id = rxHeader.Identifier;
            canMessageArray%(can)d[index].length = PLX_CAN_dlcToBytes(rxHeader.DataLength);
            memcpy(canMessageArray%(can)d[index].data, rxData, canMessageArray%(can)d[index].length);
            canMessageArray%(can)d[index].valid = 1;

            exitCriticalSection(interruptState);
          }
        }
      }
    }
  ]] % {
    can = params.can,
    rxData = params.rxData,
  }
end

function T.getRxFilterConfigurationCode(instance, params)
  U.enforceParamContract(
    params,
    {
      canId = U.isNonNegativeIntScalar,
      rxFifo = U.isNonNegativeIntScalar,
      filterIndex = U.isNonNegativeIntScalar,
      isExtId = U.isBoolean,
      mbox = U.isNonNegativeIntScalar,
    }
  )

  local initCode = [[
    {
      // Configure reception filter for message with CAN ID %(canId)d
      FDCAN_FilterTypeDef sFilterConfig = {
        0
      };
      sFilterConfig.IdType = %(frameFormat)s;
      sFilterConfig.FilterIndex = %(filterIndex)d;
      sFilterConfig.FilterType = FDCAN_FILTER_MASK;
      sFilterConfig.FilterConfig = %(rxFifo)s;
      sFilterConfig.FilterID1 = %(canId)d;
      sFilterConfig.FilterID2 = %(filterMask)s;

      HAL_FDCAN_ConfigFilter(PLX_CANBUS_getHandle(CanHandles[%(instance)d]), &sFilterConfig);
    }
  ]] % {
    instance = instance,
    canId = params.canId,
    rxFifo = (params.rxFifo == 0) and 'FDCAN_FILTER_TO_RXFIFO0' or 'FDCAN_FILTER_TO_RXFIFO1',
    filterIndex = params.filterIndex,
    frameFormat = params.isExtId and 'FDCAN_EXTENDED_ID' or 'FDCAN_STANDARD_ID',
    filterMask = params.isExtId and '0x1FFFFFFF' or '0x7FF',
  }
  return initCode
end

function T.getCanClk()
  -- TODO: use T.getAPB1Clock to get peripheral clk
  return T.getCleanSysClkHz()
end

function T.getCanPeripheralType()
  return 'FDCAN'
end

function T.getExtiIrqHandlerString(line)
  return 'EXTI%d_IRQ' % {line}
end

function T.getTimIrqHandlerString(unit)
  return T.getTargetParameters()['pwm']['irq'][unit]
end

function T.getOpenOcdOptions(opt_path, opt_serial_number)
  local options = '-f board/st_nucleo_g4.cfg'
  if opt_path then
    options = '-f '..U.shellQuote(opt_path)
  end
  if opt_serial_number then
    options = options..' -c '..U.shellQuote('hla_serial '..opt_serial_number)
  end
  return options
end

function T.getNonIntrusiveOpenOcdOptions(opt_serial_number, port)
  local options = '-f board/st_nucleo_g4_nonintrusive.cfg'
  if opt_serial_number then
    options = options..' -c '..U.shellQuote('hla_serial '..opt_serial_number)
  end
  options = options ..  ' -c "gdb_port %s"' % {port}
  return options
end

return T
