--[[
  Copyright (c) 2021, 2025 by Plexim GmbH
  All rights reserved.

  A free license is granted to anyone to use this software for any legal
  non safety-critical purpose, including commercial applications, provided
  that:
  1) IT IS NOT USED TO DIRECTLY OR INDIRECTLY COMPETE WITH PLEXIM, and
  2) THIS COPYRIGHT NOTICE IS PRESERVED in its entirety.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
  OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
  AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
  OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
  SOFTWARE.
--]]

-- TODO:
--  LL_ADC_SetCommonPathInternalCh(__LL_ADC_COMMON_INSTANCE(|<ADC>|), LL_ADC_PATH_INTERNAL_VREFINT);

local U = require('common.utils')

local static = {}

local Module = {}
function Module.getBlock(globals, cpu)
  local Adc = require('common.block').getBlock(globals, cpu)
  if not static[cpu] then
    static[cpu] = {
      numAdcBlockInstances = 0,
      numAdcUnitInstances = 0,
      adcBlockBIDs = {}, -- ADC block instances grouped by ADC unit
      finalized = false,
      dmaBufferRequired = false,
      adcUnitData = {
        --[[
          Registry of data for use by all ADC blocks, indexed by adcUnit. It has the following members:
          
            adcUnitInstance,  -- Indexes the ADC units used by ADC blocks in
                              -- the PLECS model. This is used to index the ADC
                              -- objects in the generated C code.

            analogInputs = {      -- All analog inputs for a particular ADC unit
                          input,
                          port,
                          pin,
            },

            regChannelCtr = 0,    -- Number of regular channels, not counting
                                  -- oversampled channels.

            opt_regChannels = {   -- List of all regular channels associated with
              input,              -- a given ADC unit containing user input data
              m_acqTimeClkCycles, -- particular to that channel. Oversampled
              scale,              -- channels are listed multiple times.
              offset,
            },

            regChRankCtr = 0,     -- Multiple ADC blocks can use regular
                                  -- channels on the same ADC unit, but each
                                  -- channel must have a unique rank.

            injAdcIsTaskTrigger,  -- Boolean variable (nil or true) indicating
                                  -- whether or not the base task is triggered
                                  -- by an injected conversion on this ADC unit.

            opt_injChannels = CopyOf(self.opt_injChannels),
            {                     -- List of all injected channels associated
              input,              -- with a given ADC unit containing user input
              m_acqTimeClkCycles, -- data particular to that channel.
              scale,
              offset,
            },

            opt_oversamplingRatio,  -- Oversampling ratio defined for triggered ADCs.
        --]]
      },
    }
  end
  Adc.adcBlockInstance = static[cpu].numAdcBlockInstances
  static[cpu].numAdcBlockInstances = static[cpu].numAdcBlockInstances + 1

  function Adc:checkMaskParameters()
    -- Main tab
    self.adcUnit = U.enforceMask_newComboBox(
      'AdcUnit', {'1', '2', '3', '4', '5'}).asCx()

    if Block.Mask.TrigSrc then -- Analog In (Triggered) block
      self.opt_cbx_trigSrc = U.enforceMask_newComboBox(
        'TrigSrc',
        {
          [2] = 'explicit', -- Called 'Show trigger port' in mask
          [1] = 'implicit', -- Called 'Determine automatically' in mask
        })
    end

    self.inputs = U.enforceMask(U.isNonNegativeIntScalarOrVector, 'Input') -- constraints: integer, nonEmpty, nonNegative
    self.scales = U.enforceMask(U.isScalarOrVector, 'Scale') -- constraints: finite, nonEmpty, vectorized in MaskInit
    self.offsets = U.enforceMask(U.isScalarOrVector, 'Offset') -- constraints: finite, nonEmpty, vectorized in MaskInit

    if U.comboEnabled(Block.Mask.AcqTimeSel) then
      self.opt_targetAcqTimes = U.enforceMask(U.isNonNegativeScalarOrVector, 'AcqTime') -- constraints: finite, nonEmpty, nonNegative, vectorized in MaskInit
    end

    -- Oversampling tab
    if Block.Mask.OverSampRatio then -- Exists in the Analog In (Triggered) block mask but not the Analog In block mask
      self.oversampling = U.enforceMask(U.isPositiveIntScalar, 'OverSampRatio') -- constraints: integer, positive, scalar

      self.opt_cbx_overSampMode = U.enforceMask_newComboBox(
        'OverSampMode',
        {
          'Output all samples',
          'Mean',
          'Max',
          'Min',
        })
    else
      self.oversampling = 1 -- default for modes without oversampling
    end
  end

  function Adc:getSyscfgForMatchingAdcUnit(adcUnit)
    -- Only one syscfg entry exists for each ADC unit.
    local adcs = globals.syscfg:get().adc
    if adcs then
      for _, adc in pairs(adcs) do
        if adc.unit == adcUnit then
          return adc
        end
      end
    end
  end

  function Adc:p_getDirectFeedthroughCode()
    local Require = ResourceList:new()
    local InitCode = U.CodeLines:new()
    local AdcOutputSignal = StringList:new()
    local TriggerOutputSignal = StringList:new()

    --[[
      An ADC unit can be used by multiple Analog In and Analog In (Triggered)
      blocks in a model, subject to the following constraints:

       - An ADC unit can be used by at most one Analog In (Triggered) block, and

       - An ADC unit can be used by one or more Analog In blocks, provided that
         it is not used by an Analog In (Triggered) block using regular
         conversions. Analog In (Triggered) blocks use regular conversions if
         they use oversampling or more channels than the available number of
         injected channels.
    --]]
    if not static[self.cpu].adcBlockBIDs[self.adcUnit] then
      static[self.cpu].adcBlockBIDs[self.adcUnit] = {}
    end
    table.insert(static[self.cpu].adcBlockBIDs[self.adcUnit], self.bid)

    if not static[self.cpu].adcUnitData[self.adcUnit] then
      static[self.cpu].adcUnitData[self.adcUnit] = {
        adcUnitInstance = static[self.cpu].numAdcUnitInstances,
        analogInputs = {},
        regChannelCtr = 0,
        regChRankCtr = 0, -- Ranks start at 1, 0 implies that no regular channel has been added yet
      }

      static[self.cpu].numAdcUnitInstances = static[self.cpu].numAdcUnitInstances + 1

      -- Multiple blocks can use the same adcUnit, but the resource only needs
      -- to be claimed by the first block to use it.
      Require:add('ADC%d' % {self.adcUnit})
    end

    local adcUnitData = static[self.cpu].adcUnitData[self.adcUnit]

    -- Optional Oversampling configured by Triggered ADC
    if self:isTriggered() then
      --[[
        If adcUnitData.opt_oversamplingRatio is not nil at this point, that
        means that there are multiple triggered ADC blocks using the same ADC
        unit. This is detected in the p_getNonDirectFeedthroughCode() function
        and a user error is thrown in that case.
      --]]
      adcUnitData.opt_oversamplingRatio = self.oversampling
    end

    for i, input in ipairs(self.inputs) do
      local adcInput = globals.target.getAdcGpio(self.adcUnit, input)
      if not adcInput.pads then
        U.error('ADC unit %d not available on this target.' % {self.adcUnit})
      elseif not adcInput.gpio then
        U.error('ADC channel %d not available for ADC unit %d on this target.' %
          {input, self.adcUnit})
      end

      local port, pin
      port = adcInput.gpio.port
      if port ~= '' then
        pin = adcInput.gpio.pin

        -- ADCs can use the same pin as other peripherals (e.g. DACs or other
        -- ADCs). The pins only need to be required once to make sure the target
        -- provides them.
        local pinIsFree = globals.syscfg:claimResourceIfFree(
          'P%s%d' % {port, pin})

        if pinIsFree then
          Require:add('P%s' % {port}, pin)
        end

        local analogInput = {
          input = input,
          port = port,
          pin = pin,
        }

        if not U.isInSet(analogInput, adcUnitData.analogInputs) then
          table.insert(adcUnitData.analogInputs, U.copyTable(analogInput))
        end
      end

      local targetAcqTime = 0
      if self.opt_targetAcqTimes then
        targetAcqTime = U.getFromScalarOrVector(self.opt_targetAcqTimes, i)
      end
      local realAcqTime = globals.target.adcGetRealAcqTime(targetAcqTime)

      if self:usesInjectedConversions() then

        if not self.opt_injChannels then
          self.opt_injChannels = {}
        end

        self.opt_injChannels[i] = {
          input = input,
          m_acqTimeClkCycles = realAcqTime.m_clkCycles,
          scale = U.getFromScalarOrVector(self.scales, i),
          offset = U.getFromScalarOrVector(self.offsets, i),
        }
        AdcOutputSignal:append('PLXHAL_ADC_getInjIn(%d, %d)' % {self.adcBlockInstance, i - 1})

      elseif self:usesRegularConversions() then

        if not adcUnitData.opt_regChannels then
          adcUnitData.opt_regChannels = {}
        end

        static[self.cpu].dmaBufferRequired = true

        for j = 0, (self.oversampling - 1) do
          local oversamplingOffset = j * #self.inputs

          adcUnitData.opt_regChannels[adcUnitData.regChannelCtr + oversamplingOffset + i] = {
            input = input,
            m_acqTimeClkCycles = realAcqTime.m_clkCycles,
            scale = U.getFromScalarOrVector(self.scales, i),
            offset = U.getFromScalarOrVector(self.offsets, i),
          }

          if self.oversampling == 1
          or self.opt_cbx_overSampMode.equals('Output all samples') then
            AdcOutputSignal:append('PLXHAL_ADC_getRegIn(%d, %d)'
              % {self.adcBlockInstance, oversamplingOffset + (i - 1)})
          end
        end

        if self.oversampling > 1 then
          if self.opt_cbx_overSampMode.equals('Mean') then
            AdcOutputSignal:append('PLXHAL_ADC_getMean(%d, %d)'
              % {self.adcBlockInstance, (i - 1)})

          elseif self.opt_cbx_overSampMode.equals('Max') then
            AdcOutputSignal:append('PLXHAL_ADC_getMax(%d, %d)'
              % {self.adcBlockInstance, (i - 1)})

          elseif self.opt_cbx_overSampMode.equals('Min') then
            AdcOutputSignal:append('PLXHAL_ADC_getMin(%d, %d)'
              % {self.adcBlockInstance, (i - 1)})
          end
        end
      end
    end

    if self:usesInjectedConversions() then
      self.nInjCh = #self.inputs
    elseif self:usesRegularConversions() then
      local adcUnitData = static[self.cpu].adcUnitData[self.adcUnit]

      self.nOversampRegCh = self.oversampling * #self.inputs

      -- Multiple Analog In blocks can read from channels on the same ADC unit.
      -- The channels are stored in the same array in the same C object
      -- associated with an ADC unit. self.regChannelsStartIdx is used to create
      -- lookup tables that allow a given ADC block to read data from the right
      -- channel using a PLXHAL getter function.
      self.regChannelsStartIdx = adcUnitData.regChannelCtr

      adcUnitData.regChannelCtr = adcUnitData.regChannelCtr + #self.inputs

      local regChLimit = globals.target.getTargetParameters().adcs.regChLimit
      if adcUnitData.regChannelCtr > regChLimit then
        -- Regular channels can be used by either one Analog In (Triggered)
        -- block or multiple Analog In blocks.
        if self:isTriggered() then -- Analog In (Triggered)
          U.error([[
            The maximum number of @param:Input: is %d.
          ]] % {regChLimit})
        else -- Analog In
          U.error([[
            The maximum cumulative number of @param:Input: across all Analog In blocks using the same @param:AdcUnit: is %d.
          ]] % {regChLimit})
        end
      end
    end

    if self:isTriggered() then
      TriggerOutputSignal:append('{modtrig = {bid = %d}}' % {Adc:getId()})
      OutputSignal = {AdcOutputSignal, TriggerOutputSignal}
    else
      OutputSignal = {AdcOutputSignal}
    end

    return {
      InitCode = InitCode,
      OutputSignal = OutputSignal,
      Require = Require,
      UserData = {bid = self:getId()}
    }
  end

  function Adc:p_getNonDirectFeedthroughCode()
    if not self:isTriggered() then
      -- No trigger source exists because an Analog In block is calling this
      -- function. This condition is also checked in the getTriggerSampleTime(),
      -- propagateTriggerSampleTime(), and requestImplicitTrigger() functions to
      -- avoid assigning an implicit trigger in this case.
    else
      local trig = Block.InputSignal[1][1]

      if self.opt_cbx_trigSrc then -- This is never nil if self:isTriggered() is true
        if self.opt_cbx_trigSrc.equals('implicit') or string.find(trig, 'UNCONNECTED') then
          -- an implicit trigger will be assigned using the requestImplicitTrigger() function
        else
          -- explicit triggering with connected port

          trig = trig:gsub('%s+', '')  -- remove whitespace
          if trig:sub(1, #'{adctrig') ~= '{adctrig' then
            U.error('Trigger port must be connected to ADC Trigger source.')
          end

          self.trigExpression = trig
        end
      else
        error('The code is broken: The trigger source ComboBox variable is nil even though block is triggered.')
      end
    end

    local adcUnitData = static[self.cpu].adcUnitData[self.adcUnit]

    local opt_adcUnitSyscfg = self:getSyscfgForMatchingAdcUnit(self.adcUnit)
    if not opt_adcUnitSyscfg then
      globals.syscfg:addEntry('adc', {
        unit = self.adcUnit,
        analogInputs = adcUnitData.analogInputs,
        path = self:getName(), -- Note: multiple ADC blocks can use the same ADC unit, but only one block path is passed
      })
    end

    if adcUnitData.regChannelCtr > 0 then
      if self.opt_injChannels then
        -- Add dummy conversion with minimum sampling time to handle silicon bug in revision Y chips
        table.insert(self.opt_injChannels, 1,
                     {
                       input = self.opt_injChannels[1].input,
                       m_acqTimeClkCycles = '2CYCLES_5',
                       scale = 1.0,
                       offset = 0.0,
                     })

        self.nInjCh = self.nInjCh + 1

        self.needsDummyConversion = true -- Add dummy conversion to handle silicon bug in revision Y chips

        local injChLimit = globals.target.getTargetParameters().adcs.injChLimit
        if #self.opt_injChannels > injChLimit then
          U.error(
            'In models containing Analog In and Analog In (Triggered) blocks using the same @param:AdcUnit:, the maximum number of @param:Input: is %d.' %
            {injChLimit - 1}) -- subtract 1 to account for a dummy conversion introduced to handle a silicon bug in revision Y chips
        end
      end
    else
      -- TODO: this check will be obsolete once we release oversampling and automatically switch to regular conversions if this limit is exceeded
      local injChLimit = globals.target.getTargetParameters().adcs.injChLimit
      if #(self.opt_injChannels) > injChLimit then
        U.error('The maximum number of @param:Input: is %d.' % {
          injChLimit
        })
      end
    end

    -- We do not need to verify that adcUnitData.opt_injChannels is nil to avoid
    -- overwriting data because there is a maximum of one ADC block with
    -- injected channels per ADC unit.
    if self.opt_injChannels then
      adcUnitData.opt_injChannels = U.copyTable(self.opt_injChannels)
    end

    -- Check if there is a resource conflict between Analog In (Triggered) or
    -- Analog In blocks using the same ADC unit.
    for _, bid in pairs(static[self.cpu].adcBlockBIDs[self.adcUnit]) do
      local adc = globals.instances[bid]

      if adc:getCpu() == self.cpu then
        if adc ~= self then
          if self:isTriggered() and adc:isTriggered() then
            U.error([[
              Each Analog In (Triggered) block must use a unique ADC unit. The @param:AdcUnit: of @1 is identical to the @param:AdcUnit:3: of @3.
            ]], {adc:getPath()})
          end

          -- TODO: It is not possible to trigger this error message until oversampling is released.
          -- The oversampling error message could also be adapted to highlight multiple blocks.
          if self:isTriggered() and self:usesRegularConversions() then
            local injChLimit = globals.target.getTargetParameters().adcs.injChLimit

            U.error([[
              If an Analog In (Triggered) block uses @param:OverSampRatio: > 1 or more than %(injChLimit)d @param:Input:, it cannot use the same @param:AdcUnit: as any Analog In blocks in the model.

              These settings cause the Analog In (Triggered) block to switch from injected to regular conversions, which causes a resource conflict with the Analog In blocks.
            ]] % {injChLimit = injChLimit - 1})
          end
        end
      end
    end

    return {}
  end

  function Adc:isTriggered()
    return self.opt_cbx_trigSrc and true or false
  end

  -- Each ADC block uses either injected OR regular conversions.
  function Adc:usesInjectedConversions()
    return not self:usesRegularConversions()
  end

  function Adc:usesRegularConversions()
    if not self.opt_cbx_trigSrc then
      return true
    end
    --[[ TODO: replace the code above with the commented code below when unlocking oversampling
    -- Regular conversions are used by Analog In blocks which don't have a
    -- trigger source and by Analog In (Triggered) blocks which use oversampling
    -- or use more input channels than the limit given for injected channels.

    local injChLimit = globals.target.getTargetParameters().adcs.injChLimit

    if not self.opt_cbx_trigSrc
    or self.opt_cbx_overSampMode
    or #(self.inputs + 1) > injChLimit then -- add 1 in case a dummy conversion is needed to handle a silicon bug in revision Y chips
      return true
    end]]
    return false
  end

  function Adc:getAdcInjIrqCode()
    return globals.target.getAdcInjIrqCode({
      isModTrig = self.is_mod_trigger,
      adcUnit = self.adcUnit,
      adcBlockInstance = self.adcBlockInstance,
      adcTriggerBlockType = self.triggerBlockType,
      opt_adcPostScaler = self.opt_adcPostScaler,
    })
  end

  function Adc:setImplicitTriggerSource(bid)
    self.trig_base_task_exp = '{adctrig = {bid = %d}}' % {bid}
  end

  function Adc:setSinkForTriggerSource(sink)
    if not self:isTriggered() then
      -- this block is not triggered and does not provide a trigger
      return
    end

    if not self.trigExpression then
      self.trigExpression = self.trig_base_task_exp
    end

    local trig = eval(self.trigExpression).adctrig
    local triggerBlock = globals.instances[trig.bid]
    self.trig = triggerBlock
    self.triggerBlockType = self.trig:getType()

    self.opt_adcPostScaler = triggerBlock:getAdcPostScaler()

    if sink then
      if not self[sink.type] then
        self[sink.type] = {}
      end
      table.insert(self[sink.type], globals.instances[sink.bid])
    end

    if not self.downstreamConnectionsPropagated then
      -- top of chain
      if self.trig then
        self.trig:setSinkForTriggerSource({
          type = 'adctrig',
          bid = self.bid,
        })
      end
      self.downstreamConnectionsPropagated = true;
    end
  end

  function Adc:propagateTriggerSampleTime(ts)
    if not self:isTriggered() then
      -- this block is not triggered and does not provide a trigger
      return
    end

    if ts then
      self.ts = ts * self.oversampling
      if self.modtrig then
        for _, b in ipairs(self.modtrig) do
          local f = b:propagateTriggerSampleTime(self.ts)
        end
      end
    end
  end

  function Adc:requestImplicitTrigger(ts)
    if not self:isTriggered() then
      -- this block is not triggered and does not provide a trigger
      return
    end

    if not self.trigExpression then
      return Target.Variables.SAMPLE_TIME
    else
      local trig = eval(self.trigExpression).adctrig
      local triggerBlock = globals.instances[trig.bid]
      local opt_triggerPeriod = triggerBlock:requestImplicitTrigger()
      if opt_triggerPeriod then
        return opt_triggerPeriod * self.oversampling
      end
    end
  end

  function Adc:finalizeThis(c)
    local adcUnitData = static[self.cpu].adcUnitData[self.adcUnit]

    self.is_mod_trigger = false
    -- determine if this block is supplying the base task trigger
    if self.modtrig then
      for _, b in ipairs(self.modtrig) do
        if b:blockMatches('tasktrigger') then
          self.is_mod_trigger = true
          if self:usesInjectedConversions() then
            adcUnitData.injAdcIsTaskTrigger = true
          end
          break
        end
      end
    end

    -- Set up the interrupt service routine for ADC blocks operating in the
    -- triggered regular mode.
    if self:isTriggered() and self:usesRegularConversions() then

      local dma = globals.target.getTargetParameters().dma[globals.target.getChipName()]['ADC%d' % {self.adcUnit}]
      if dma then
        local dmaIrqName = globals.target.getTargetParameters().dma.irq

        local opt_doubleBufferCode
        local opt_clearDoubleBufferCode
        if (self.oversampling > 1) then
          opt_clearDoubleBufferCode = [=[
            if (LL_DMA_IsActiveFlag_HT%(channel)d(DMA%(unit)d))
            {
              LL_DMA_ClearFlag_HT%(channel)d(DMA%(unit)d);
              PLX_ADC_setDoubleBufferIndex(AdcUnitHandles[AdcUnitHandleLookup[%(adcBlockInstance)d]], 0);
              %(opt_dispCode)s
            }
          ]=] % {
            adcBlockInstance = self.adcBlockInstance,
            channel = dma.channel,
            unit = dma.unit,
            opt_dispCode = self.is_mod_trigger and 'DISPR_dispatch();' or '',
          }

          opt_doubleBufferCode = [=[
            PLX_ADC_setDoubleBufferIndex(AdcUnitHandles[AdcUnitHandleLookup[%(adcBlockInstance)d]],
                                         %(nRegChannels)d);
          ]=] % {
            adcBlockInstance = self.adcBlockInstance,
            nRegChannels = #adcUnitData.opt_regChannels,
          }
        end

        local opt_checkOverrunCode
        if self.is_mod_trigger then
          opt_checkOverrunCode = [[
            bool %(baseName)s_checkOverrun() {
              return HAL_NVIC_GetPendingIRQ(DMA%(dma)d_%(irqName)s%(channel)d_IRQn);
            }
          ]] % {
            baseName = Target.Variables.BASE_NAME,
            dma = dma.unit,
            channel = dma.channel,
            irqName = dmaIrqName,
          }
        end

        c.Declarations:append([[
          void DMA%(unit)d_%(dmaIrqName)s%(channel)d_IRQHandler(void) {
            %(opt_clearDoubleBufferCode)s
            if (LL_DMA_IsActiveFlag_TC%(channel)d(DMA%(unit)d)) {
              LL_DMA_ClearFlag_TC%(channel)d(DMA%(unit)d);
              %(opt_doubleBufferCode)s
              %(opt_dispCode)s
            }
          }
          %(opt_checkOverrunCode)s
        ]] % {
          unit = dma.unit,
          dmaIrqName = dmaIrqName,
          channel = dma.channel,
          opt_dispCode = self.is_mod_trigger and 'DISPR_dispatch();' or '',
          opt_checkOverrunCode = opt_checkOverrunCode or '',
          opt_clearDoubleBufferCode = opt_clearDoubleBufferCode or '',
          opt_doubleBufferCode = opt_doubleBufferCode or '',
        })

        -- The highest interrupt priority is reserved for the event providing
        -- the control task trigger, which could be either a regular ADC
        -- conversion or an injected ADC conversion.
        local irqPriority
        if self.is_mod_trigger then
          irqPriority = 'configLIBRARY_MAX_SYSCALL_INTERRUPT_PRIORITY'
        else
          irqPriority = 'configLIBRARY_MAX_SYSCALL_INTERRUPT_PRIORITY - 1'
        end
        c.InterruptEnableCode:append([=[
          HAL_NVIC_SetPriority(DMA%(unit)d_%(dmaIrqName)s%(channel)d_IRQn, %(irqPriority)s, 0);
          HAL_NVIC_EnableIRQ(DMA%(unit)d_%(dmaIrqName)s%(channel)d_IRQn);
          PLX_ADC_enableRegularAdcInterrupt(AdcUnitHandles[AdcUnitHandleLookup[%(instance)d]]);
        ]=] % {
          unit = dma.unit,
          dmaIrqName = dmaIrqName,
          channel = dma.channel,
          irqPriority = irqPriority,
          instance = self.adcBlockInstance,
        })
      end
    end
  end

  function Adc:finalize(c)
    if static[self.cpu].finalized then
      return
    end

    c.Include:append('plx_adc.h')

    --[[
    typedef struct PLX_ADC_OBJ
{
   ADC_TypeDef *adcInstance;
   uint16_t numRegularConversions;
   uint16_t oversamplingRatio;
   uint16_t numRegularChannels;
   uint16_t numInjectedConversions;
   uint16_t injResults[4];
   float injScale[4];
   float injOffset[4];
   uint16_t* results;
   float* scale;
   float* offset;
   volatile uint16_t doubleBufferOffset;
} PLX_ADC_Obj_t;
    ]]
    c.Declarations:append([[
      // PLX_ADC_Obj for each ADC unit in use
      PLX_ADC_Handle_t AdcUnitHandles[%(numInst)d];
      PLX_ADC_Obj_t AdcUnitObj[%(numInst)d];
    ]] % {
      numInst = static[self.cpu].numAdcUnitInstances,
    })

    if static[self.cpu].dmaBufferRequired then
      c.Declarations:append(globals.target.getDmaBufferDeclaration())
    end

    --[[
      In the generated C code, we create an ADC object for every ADC unit that
      is used by ADC blocks in the model. The ADC object has the following two
      members storing the results from injected and regular ADC measurements:

       - uint16_t injResults[4]
       - uint16_t* results

      We also create two lookup tables to allow the PLXHAL functions to access
      the correct result given an ADC block instance and input channel. To do so,
      we create the following two tables:

       - The AdcChannelLookup table maps input channels to results array indices
         for each ADC block. For each ADC unit channels are indexed separately for

          - Regular channels (stored in injResults array) and
          - Injected Channels (stored in results array)

          Indexing starts at zero, except when a dummy conversion is used. (In the
          case where a dummy conversion is required, it is assigned index 0 for the
          injected channels and the user channels start indexing at 1.)

       - The AdcUnitHandleLookup table maps each block to the index corresponding
        to the ADC unit.
    --]]
    local adcChLookupRows = {}
    local adcUnitHandleLookupRows = {}
    local maxNumChannels = 0
    for adcUnit, adcBIDs in pairs(static[self.cpu].adcBlockBIDs) do
      local adcUnitData = static[self.cpu].adcUnitData[adcUnit]
      for _, bid in pairs(adcBIDs) do
        local adc = globals.instances[bid]

        local channels = {}
        if adc:usesInjectedConversions() then
          local nInjChannels = adc:getParameter('nInjCh')

          if adc:getParameter('needsDummyConversion') then
            for i = 1, nInjChannels - 1 do
              table.insert(channels, i)
            end

            maxNumChannels = math.max(maxNumChannels, nInjChannels - 1)
          else
            for i = 1, nInjChannels do
              table.insert(channels, i - 1)
            end

            maxNumChannels = math.max(maxNumChannels, nInjChannels)
          end
        elseif adc:usesRegularConversions() then
          local nRegChannels = adc:getParameter('nOversampRegCh')

          for i = 1, nRegChannels do
            table.insert(channels, adc:getParameter('regChannelsStartIdx') + i - 1)
          end

          maxNumChannels = math.max(maxNumChannels, nRegChannels)
        end

        local adcBlockInstance = adc:getParameter('adcBlockInstance')
        adcChLookupRows[adcBlockInstance + 1] = 
           '{'..
           table.concat(channels, ',')..
           '}, /* %(blockName)s: ADC Unit %(unit)d, conversion type: %(type)s, ch: %(channels)s */'
           % {
             blockName = adc:getName(),
             unit = adc.adcUnit,
             type = adc:usesInjectedConversions() and 'injected' or 'regular',
             channels = U.inputsToStringList(adc.inputs),
           }

        local adcUnitInstance = adcUnitData.adcUnitInstance
        adcUnitHandleLookupRows[adcBlockInstance + 1] = adcUnitInstance
      end
    end

    c.Declarations:append([=[
      const uint16_t AdcChannelLookup[%(nAdcBlockInstances)d][%(nChannels)d] = %(adcChLookupTable)s;
      const uint16_t AdcUnitHandleLookup[%(nAdcBlockInstances)d] = %(adcUnitHandleLookup)s;

      float PLXHAL_ADC_getInjIn(uint16_t aHandle, uint16_t aChannel) {
        return PLX_ADC_getIn(AdcUnitHandles[AdcUnitHandleLookup[aHandle]],
                             AdcChannelLookup[aHandle][aChannel], true);
      }

      float PLXHAL_ADC_getRegIn(uint16_t aHandle, uint16_t aChannel) {
        return PLX_ADC_getIn(AdcUnitHandles[AdcUnitHandleLookup[aHandle]],
                             AdcChannelLookup[aHandle][aChannel], false);
      }

      float PLXHAL_ADC_getMean(uint16_t aHandle, uint8_t aOffset) {
        return PLX_ADC_getMean(AdcUnitHandles[AdcUnitHandleLookup[aHandle]], aOffset);
      }

      float PLXHAL_ADC_getMax(uint16_t aHandle, uint8_t aOffset) {
        return PLX_ADC_getMax(AdcUnitHandles[AdcUnitHandleLookup[aHandle]], aOffset);
      }

      float PLXHAL_ADC_getMin(uint16_t aHandle, uint8_t aOffset) {
        return PLX_ADC_getMin(AdcUnitHandles[AdcUnitHandleLookup[aHandle]], aOffset);
      }
    ]=] % {
      nAdcBlockInstances = static[self.cpu].numAdcBlockInstances,
      nChannels = maxNumChannels,
      adcChLookupTable = '{\n'..table.concat(adcChLookupRows, '\n')..'\n}', -- support special comments formatting above
      adcUnitHandleLookup = '{'..table.concat(adcUnitHandleLookupRows, ',')..'}',
    })

    c.PreInitCode:append([[
      PLX_ADC_sinit(%(vref)f);
      for (int i = 0; i < %(numAdcUnitInstances)d; i++) {
        AdcUnitHandles[i] = PLX_ADC_init(&AdcUnitObj[i], sizeof(AdcUnitObj[i]));
      }
    ]] % {
      vref = Target.Variables.AdcVRef,
      numAdcUnitInstances = static[self.cpu].numAdcUnitInstances,
    })

    -- Generate code that must be generated once for each ADC block
    for _, adcBIDs in pairs(static[self.cpu].adcBlockBIDs) do
      for _, bid in pairs(adcBIDs) do
        local adc = globals.instances[bid]
        adc:finalizeThis(c)
      end
    end

    -- Generate code that must be generated once for each ADC unit
    for adcUnit, adcBIDs in pairs(static[self.cpu].adcBlockBIDs) do
      local adcUnitData = static[self.cpu].adcUnitData[adcUnit]

      -- Determine the macros indicating the trigger source for triggered
      -- injected and triggered regular ADC conversions, if they exist.
      local opt_m_injTrgSrc, opt_m_regTrgSrc
      for _, bid in pairs(adcBIDs) do
        local adc = globals.instances[bid]

        local opt_triggerBlock = adc:getParameter('trig')

        if opt_triggerBlock then
          local triggerBlock = opt_triggerBlock

          if adc:usesInjectedConversions() then
            opt_m_injTrgSrc = triggerBlock:getInjAdcTrgSrcMacro(adcUnit)

          elseif adc:usesRegularConversions() then
            opt_m_regTrgSrc = triggerBlock:getRegAdcTrgSrcMacro(adcUnit)

          else
            error('The code should never be reached. All ADCs should have either regular or injected conversions.')
          end
        else
          if adc:isTriggered() then
            error('The code related to the trigger chain is broken. The ADC block is triggered but has no trigger block.')
          end
        end
      end

      local adcUnitSetupCode = self.globals.target.getAdcUnitSetupCode(
        adcUnit,
        {
          injChannels = adcUnitData.opt_injChannels or {},
          nRegInputCh = adcUnitData.regChannelCtr,
          m_injTrgSrc = opt_m_injTrgSrc or 'LL_ADC_INJ_TRIG_SOFTWARE',
          m_regTrgSrc = opt_m_regTrgSrc or 'LL_ADC_REG_TRIG_SOFTWARE',
        })

      local optionalInjAdcSetupCode = ''
      if adcUnitData.opt_injChannels then
        local rank = 1
        for _, injChannel in ipairs(adcUnitData.opt_injChannels) do
          optionalInjAdcSetupCode = optionalInjAdcSetupCode..globals.target.getAdcInjChSetupCode(
            adcUnit, injChannel.input, {
              rank = rank,
              m_acqTimeClkCycles = injChannel.m_acqTimeClkCycles,
            })

          optionalInjAdcSetupCode = optionalInjAdcSetupCode..[[
            PLX_ADC_addInjectedChannel(AdcUnitHandles[%(adcUnitInstance)d],
                                      %(scale)f,
                                      %(offset)f);
          ]] % {
            adcUnitInstance = adcUnitData.adcUnitInstance,
            scale = injChannel.scale,
            offset = injChannel.offset,
          }

          rank = rank + 1
        end
      end

      local opt_regAdcSetupCode
      if adcUnitData.opt_regChannels then
        local resultArrayLength = #(adcUnitData.opt_regChannels)
        if self:isTriggered() and self:usesRegularConversions() then
          resultArrayLength = resultArrayLength * 2
        end

        local regChannelSetupCode = ''
        for i, regChannel in ipairs(adcUnitData.opt_regChannels) do
          local opt_adcChannelSetupCode
          -- Downsample channels by oversampling ratio if applicable. Note: If oversampling is enabled, there is only one ADC block per ADC unit.
          if math.fmod(i - 1, adcUnitData.opt_oversamplingRatio or 1) == 0 then
            -- Each regular channel on a given ADC unit has a unique rank, even if multiple ADC blocks use the same ADC unit.
            adcUnitData.regChRankCtr = adcUnitData.regChRankCtr + 1

            opt_adcChannelSetupCode = globals.target.getAdcRegChSetupCode(
              adcUnit, regChannel.input, {
                rank = adcUnitData.regChRankCtr,
                m_acqTimeClkCycles = regChannel.m_acqTimeClkCycles,
              })
          end

          regChannelSetupCode = regChannelSetupCode..[[
            %(opt_adcChannelSetupCode)s

            PLX_ADC_addChannel(AdcUnitHandles[%(adcUnitInstance)d], %(scale)f, %(offset)f);
          ]] % {
            opt_adcChannelSetupCode = opt_adcChannelSetupCode or '',
            adcUnitInstance = adcUnitData.adcUnitInstance,
            scale = regChannel.scale,
            offset = regChannel.offset,
          }
        end

        opt_regAdcSetupCode = [[
          {
            DMA_BUFFER static uint16_t results[%(resLength)d];
            static float scale[%(length)d];
            static float offset[%(length)d];
            PLX_ADC_setupRegAdcBuffer(AdcUnitHandles[%(handle)d], &results[0], &scale[0], &offset[0]);
          }

          %(regChannelSetupCode)s
        ]] % {
          resLength = resultArrayLength,
          length = #(adcUnitData.opt_regChannels),
          handle = adcUnitData.adcUnitInstance,
          regChannelSetupCode = regChannelSetupCode,
        }
      end

      c.PreInitCode:append([[
        {
          PLX_ADC_setup(AdcUnitHandles[%(adcUnitInstance)d],
                        PLX_ADC%(adcUnit)d,
                        %(oversamplingRatio)d,
                        %(nRegInputCh)d);
          %(adcUnitSetupCode)s
          %(optionalInjAdcSetupCode)s
          %(opt_regAdcSetupCode)s
        }
      ]] % {
        adcUnitInstance = adcUnitData.adcUnitInstance,
        adcUnit = adcUnit,
        oversamplingRatio = adcUnitData.opt_oversamplingRatio or 1,
        nRegInputCh = adcUnitData.regChannelCtr,
        adcUnitSetupCode = adcUnitSetupCode,
        optionalInjAdcSetupCode = optionalInjAdcSetupCode,
        opt_regAdcSetupCode = opt_regAdcSetupCode or '',
      })

      c.PostInitCode:append([[
        PLX_ADC_start(AdcUnitHandles[%(adcUnitInstance)d]);
      ]] % {
        adcUnitInstance = adcUnitData.adcUnitInstance,
      })
    end

    -- Set up the interrupt service routine for injected ADCs.

    -- List of ADC blocks associated with unit 1 or unit 2. The interrupt
    -- service routine is combined for these two ADC units, which makes them the
    -- exception.
    local exceptionAdcUnits = {1, 2}
    local exceptionAdcBIDs = {}
    for adcUnit, adcBIDs in pairs(static[self.cpu].adcBlockBIDs) do
      if adcUnit == 1 or adcUnit == 2 then
        for _, bid in pairs(adcBIDs) do
          table.insert(exceptionAdcBIDs, bid)
        end
      end
    end

    local exceptionAdcsAreConfigured = false
    local interruptPriorityIndex = 1
    for adcUnit, adcBIDs in pairs(static[self.cpu].adcBlockBIDs) do
      local adcUnitData = static[self.cpu].adcUnitData[adcUnit]

      if adcUnitData.opt_injChannels then
        if U.arrayContainsValue(exceptionAdcUnits, adcUnit)
        and exceptionAdcsAreConfigured then
          -- Do nothing
        else
          local injAdcIsTaskTrigger = false
          local BIDs
          if U.arrayContainsValue(exceptionAdcUnits, adcUnit) then
            BIDs = exceptionAdcBIDs
            exceptionAdcsAreConfigured = true
            for _, exceptionAdcUnit in pairs(exceptionAdcUnits) do
              local opt_data = static[self.cpu].adcUnitData[exceptionAdcUnit]
              if opt_data and opt_data.injAdcIsTaskTrigger then
                injAdcIsTaskTrigger = true
              end
            end
          else
            BIDs = adcBIDs
            if adcUnitData.injAdcIsTaskTrigger then
              injAdcIsTaskTrigger = true
            end
          end

          local adcInjIrqCode = ''
          for _, bid in pairs(BIDs) do
            local adc = globals.instances[bid]

            if adc:usesInjectedConversions() then
              adcInjIrqCode = adcInjIrqCode..adc:getAdcInjIrqCode()
            end
          end

          local m_injIrqName = globals.target.getTargetParameters().adcs.m_injIrqName[adcUnit]

          local opt_overrunCode
          if injAdcIsTaskTrigger then
            opt_overrunCode = [[
              bool %(baseName)s_checkOverrun() {
                return HAL_NVIC_GetPendingIRQ(%(m_injIrqName)sn);
              }
            ]] % {
              baseName = Target.Variables.BASE_NAME,
              m_injIrqName = m_injIrqName,
            }
          end

          c.Declarations:append([[
            void %(m_injIrqName)sHandler(void) {
              %(adcInjIrqCode)s
            }

            %(opt_overrunCode)s
          ]] % {
            m_injIrqName = m_injIrqName,
            adcInjIrqCode = adcInjIrqCode,
            opt_overrunCode = opt_overrunCode or '',
          })

          local m_interruptPriority
          if injAdcIsTaskTrigger then
            m_interruptPriority = 'configLIBRARY_MAX_SYSCALL_INTERRUPT_PRIORITY'
          else
            m_interruptPriority =
               'configLIBRARY_MAX_SYSCALL_INTERRUPT_PRIORITY - %d' %
               {interruptPriorityIndex}

            interruptPriorityIndex = interruptPriorityIndex + 1
          end

          c.InterruptEnableCode:append([[
            HAL_NVIC_SetPriority(%(m_injIrqName)sn, %(priority)s, 0);
            HAL_NVIC_EnableIRQ(%(m_injIrqName)sn);
          ]] % {
            m_injIrqName = m_injIrqName,
            priority = m_interruptPriority,
          })
        end
      end
    end

    static[self.cpu].finalized = true
  end

  function Adc:getTriggerSampleTime()
    if not self:isTriggered() then
      -- this block is not triggered and does not provide a trigger
      return 0
    end
    return self.ts
  end

  function Adc:getTotalConversionClkCycles()
    local totalClkCycles = 0
    for i, _ in ipairs(self.inputs) do
      local targetAcqTime = 0
      if self.opt_targetAcqTimes then
        targetAcqTime = U.getFromScalarOrVector(self.opt_targetAcqTimes, i)
      end
      local realAcqTime = globals.target.adcGetRealAcqTime(targetAcqTime)
      totalClkCycles = totalClkCycles + realAcqTime.clkCycles
    end
    return totalClkCycles
  end

  return Adc
end

return Module
