--[[
  Copyright (c) 2021 by Plexim GmbH
  All rights reserved.

  A free license is granted to anyone to use this software for any legal
  non safety-critical purpose, including commercial applications, provided
  that:
  1) IT IS NOT USED TO DIRECTLY OR INDIRECTLY COMPETE WITH PLEXIM, and
  2) THIS COPYRIGHT NOTICE IS PRESERVED in its entirety.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
  OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
  AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
  OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
  SOFTWARE.
--]]

local Module = {}
local U = require('common.utils')

local static = {}

function Module.getBlock(globals, cpu)

  local ExtMode = require('common.block').getBlock(globals, cpu)
  if static[cpu] == nil then
    static[cpu] = {
      numInstances = 0,
      finalized  = false,
    }
  end
  ExtMode["instance"] = static[cpu].numInstances
  static[cpu].numInstances = static[cpu].numInstances + 1

  function ExtMode:createImplicit(req)

    local extModeCombo = {'off', 'jtag', 'serial'}
    self.extMode = extModeCombo[Target.Variables.EXTERNAL_MODE+1]

    if self.extMode == 'serial' then
      -- determine UART pinset
      local pinset
      local rxPort = string.char(65+Target.Variables.ExtModeUartRxPort-1)
      local rxPin = Target.Variables.ExtModeUartRxPin
      local txPort = string.char(65+Target.Variables.ExtModeUartTxPort-1)
      local txPin = Target.Variables.ExtModeUartTxPin
      
      -- determine baud rate
      local baudrate
      local maxRate = globals.target.getMaxSciBaudRate()
      local supportedRates = {256000, 128000, 115200, 57600, 38400, 19200, 14400, 9600, 4800}
      for _, rate in ipairs(supportedRates) do
        if maxRate >= rate then
          baudrate = rate
          break
        end
      end
      if baudrate == nil then
        U.error("The control task execution rate is too low to support external mode communication.")
      end

      self.baudrate = baudrate

      local uartObj = self:makeBlock('uart', self.cpu)
      
      uartObj:createImplicit({
        rxPort = rxPort,
        rxPin = rxPin,
        txPort = txPort,
        txPin = txPin,
        baudrate = baudrate,
        path = 'External mode over Serial'
      }, req)
    end
  end

  function ExtMode:p_getDirectFeedthroughCode()
    U.error("Explicit use of EXTMODE via target block not supported.")
  end
  
  function ExtMode:finalize(f)
    if static[self.cpu].finalized then
      return
    end
    if static[self.cpu].numInstances ~= 1 then
      U.error('There should be only one (implicit) instance of the ExtMode block.')
    end
    
    f.Include:append('pil.h')
    f.Include:append('%s.h' % {Target.Variables.BASE_NAME})

    f.Declarations:append('PIL_Obj_t PilObj;')
    f.Declarations:append('PIL_Handle_t PilHandle = 0;')
    
    if (self.extMode ~= 'off') and (Target.Variables.SAMPLE_TIME > 1e-3) then
      U.error('Discretizaton step size too large to support external mode communications.')
    end
    
    if self.extMode == 'off' then
      self:logLine('External mode disabled.')
      return
    elseif self.extMode == 'jtag' then
      self:logLine('Configuring external mode over JTAG.')
    else
      self:logLine('Configuring external mode over UART.')
    end
    
    if self.extMode == 'serial' then
       f.Include:append('plx_uart.h')
       f.Declarations:append('PLX_UART_Obj_t UartObj;')
       f.Declarations:append('PLX_UART_Handle_t UartHandle;')
    end

    -- determine scope buffer size
    local extModeSignalSizeInBytes
    if Target.Variables.FLOAT_TYPE == 'float' then
      extModeSignalSizeInBytes = 4
    elseif Target.Variables.FLOAT_TYPE == 'double' then
      extModeSignalSizeInBytes = 8
    else
      U.error('Unsupported external mode data type (%s).' % {Target.Variables.FLOAT_TYPE})
    end
    
    local scopeMaxTraceWidthInWords = Model.NumExtModeSignals * extModeSignalSizeInBytes/2
    local addressPtrSizeInWords = 2
    local scopeBufSizeInWords = Target.Variables.ExtModeBufferSize + scopeMaxTraceWidthInWords*addressPtrSizeInWords

    self:logLine('Allocating %i bytes for external mode buffer.' % {2*scopeBufSizeInWords})
        
    f.Declarations:append('uint16_t ScopeBuffer[%i] /*__attribute__((aligned(16)))*/;' % {scopeBufSizeInWords})
    f.Declarations:append('extern void PIL_setAndConfigScopeBuffer(PIL_Handle_t aPilHandle, uint16_t* aBufPtr, uint16_t aBufSize, uint16_t aMaxTraceWidthInWords);')
    f.Declarations:append('extern const char * const %s_checksum;\n' % {Target.Variables.BASE_NAME})
    
    -- external mode helper symbols
    f.Declarations:append('// external mode helper symbols\n')
    f.Declarations:append('PIL_CONFIG_DEF(uint32_t, ExtMode_targetFloat_Size, sizeof(%s_FloatType));\n' % {Target.Variables.BASE_NAME})
    f.Declarations:append('PIL_CONFIG_DEF(uint32_t, ExtMode_targetPointer_Size, sizeof(%s_FloatType*));\n' % {Target.Variables.BASE_NAME})
    -- note: the following assumes that the base-task sample time is the first value of the _sampleTime array
    f.Declarations:append('PIL_CONFIG_DEF(uint32_t, ExtMode_sampleTime_Ptr, (uint32_t)&%s_sampleTime);\n' % {Target.Variables.BASE_NAME})
    f.Declarations:append('PIL_CONFIG_DEF(uint32_t, ExtMode_checksum_Ptr, (uint32_t)&%s_checksum);\n' % {Target.Variables.BASE_NAME})
    f.Declarations:append('#if defined(%s_NumTunableParameters) && (%s_NumTunableParameters >0)\n' % {Target.Variables.BASE_NAME, Target.Variables.BASE_NAME})
    f.Declarations:append('PIL_CONFIG_DEF(uint32_t, ExtMode_P_Ptr, (uint32_t)&%s_P);\n' % {Target.Variables.BASE_NAME})
    f.Declarations:append('PIL_CONFIG_DEF(uint32_t, ExtMode_P_Size, (uint32_t)%s_NumTunableParameters);\n' % {Target.Variables.BASE_NAME})
    f.Declarations:append('#endif\n')
    f.Declarations:append('#if defined(%s_NumExtModeSignals) && (%s_NumExtModeSignals > 0)\n' % {Target.Variables.BASE_NAME, Target.Variables.BASE_NAME})
    f.Declarations:append('PIL_CONFIG_DEF(uint32_t, ExtMode_ExtModeSignals_Ptr, (uint32_t)&%s_ExtModeSignals[0]);\n' % {Target.Variables.BASE_NAME})
    f.Declarations:append('PIL_CONFIG_DEF(uint32_t, ExtMode_ExtModeSignals_Size, (uint32_t)%s_NumExtModeSignals);\n' % {Target.Variables.BASE_NAME})
    f.Declarations:append('#endif\n\n')
    
    f.Declarations:append('#define CODE_GUID %s;' % {U.guid()})
    f.Declarations:append('PIL_CONST_DEF(unsigned char, Guid[], %s);' % {'CODE_GUID'})
    f.Declarations:append('PIL_CONST_DEF(unsigned char, CompiledDate[], "%s");' % {os.date('%m/%d/%Y %I:%M %p')})
    f.Declarations:append('PIL_CONST_DEF(unsigned char, CompiledBy[], "%s");' % {'PLECS Coder'})
    f.Declarations:append('PIL_CONST_DEF(uint16_t, FrameworkVersion, PIL_FRAMEWORK_VERSION);')
    f.Declarations:append('PIL_CONST_DEF(char, FirmwareDescription[], "STM32 Project");')
    f.Declarations:append('PIL_CONST_DEF(uint16_t, StationAddress, 0);')
    
    if self.extMode == 'jtag' then
      f.Declarations:append('#define PARALLEL_COM_PROTOCOL %i' % {3})
      f.Declarations:append('PIL_CONST_DEF(uint16_t, ParallelComProtocol, PARALLEL_COM_PROTOCOL);')
      f.Declarations:append('PIL_CONST_DEF(uint32_t, ParallelComBufferAddress, PARALLEL_COM_BUF_ADDR);')
      f.Declarations:append('PIL_CONST_DEF(uint16_t, ParallelComBufferLength, PARALLEL_COM_BUF_LEN);')
      f.Declarations:append('PIL_CONST_DEF(uint16_t, ParallelComTimeoutMs, 1000);')
      f.Declarations:append('PIL_CONST_DEF(uint16_t, ExtendedComTimingMs, 2000);')
    else

      f.Declarations:append('PIL_CONST_DEF(uint32_t, BaudRate, %i);' % {self.baudrate})

      -- generate UART polling code
      local code = [[
      static void SciPoll(PIL_Handle_t aHandle)
      {
          if(PLX_UART_breakOccurred(UartHandle)){
              PLX_UART_reset(UartHandle);
          }
          while(PLX_UART_rxReady(UartHandle))
          {
              // assuming that there will be a "break" when FIFO is empty
              PIL_SERIAL_IN(aHandle, (int16_t)PLX_UART_getChar(UartHandle));
          }
          int16_t ch;
          if(!PLX_UART_txIsBusy(UartHandle)){
              if(PIL_SERIAL_OUT(aHandle, &ch))
              {
                  PLX_UART_putChar(UartHandle, ch);
              }
          }
      }
      ]]
      f.Declarations:append(code)
    end
  
    -- configure PIL framework
    local preInitCode, declarationsCode, interruptEnableCode = self.globals.target.getPilFrameworkConfigurationCode({
      base_name = Target.Variables.BASE_NAME,
      scope_buffer_size = scopeBufSizeInWords,
      scope_max_trace_width = scopeMaxTraceWidthInWords,
      ext_mode = self.extMode,
    })

    f.PreInitCode:append(preInitCode)
    f.Declarations:append(declarationsCode)
    f.InterruptEnableCode:append(interruptEnableCode)
    static[self.cpu].finalized = true
  end

  return ExtMode
end

return Module
