--[[
  Copyright (c) 2021 by Plexim GmbH
  All rights reserved.

  A free license is granted to anyone to use this software for any legal
  non safety-critical purpose, including commercial applications, provided
  that:
  1) IT IS NOT USED TO DIRECTLY OR INDIRECTLY COMPETE WITH PLEXIM, and
  2) THIS COPYRIGHT NOTICE IS PRESERVED in its entirety.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
  OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
  AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
  OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
  SOFTWARE.
--]]--
local Module = {}
local U = require('common.utils')

local static = {}

local inputType = {
  "UP",
  "DOWN",
  "NO",
}

local outputType = {
  "PUSHPULL",
  "OPENDRAIN"
}

function Module.getBlock(globals, cpu)

  local ExtSync = require('common.block').getBlock(globals, cpu)
  if static[cpu] == nil then
    static[cpu] = {
      numInstances = 0,
      instances = {},
      finalized = false,
    }
  end
  ExtSync["instance"] = static[cpu].numInstances
  static[cpu].numInstances = static[cpu].numInstances + 1

  function ExtSync:checkMaskParameters()
    if not U.isNonNegativeIntScalar(Block.Mask.Pin) then
      U.error('Pin must be a positive integer scalar.')
    end
  end

  function ExtSync:p_getDirectFeedthroughCode()
    local Require = ResourceList:new()
    local OutputSignal = StringList:new()

    table.insert(static[self.cpu].instances, self.bid)

    self.port = string.char(65+Block.Mask.Port-1)
    self.pin = Block.Mask.Pin
    self.sync_dir = Block.Mask.SyncDir
    if self.sync_dir == 1 then
      self.input_type = inputType[Block.Mask.InputType]
      self.sync_type_in = Block.Mask.SyncBehaviourIn
    elseif  self.sync_dir == 2 then
      local out_behaviour = {
        "positive",
        "negative"
      }
      self.output_type = outputType[Block.Mask.OutputType]
      self.sync_type_out = out_behaviour[Block.Mask.SyncBehaviourOut]
    end

    self.ext_sync = {}
    
    self.path = self:getName()

    Require:add('P%s'% {self.port}, self.pin) 
    
    local sync_signal = {}
    if self.sync_dir == 1 then
      sync_signal[1] = "{ext_synco = {bid = %i, type = %i}}" % {self:getId(), self.sync_type_in} --dummy: needed for offline implementations
    else
      sync_signal[1] = "{}" --dummy
    end
    OutputSignal:append(sync_signal)

    self.sinkBIDs = {}

    return {
      Require = Require, 
      OutputSignal = OutputSignal,
      UserData = {bid = self:getId()}
    }
  end

  function ExtSync:p_getNonDirectFeedthroughCode()
    local Require = ResourceList:new()
    if self.sync_dir == 2 then -- the sync block act as master --> provides sync signal to a slave
      local synci_terminal = Block.InputSignal[1][1]
      synci_terminal = synci_terminal:gsub("%s+", "") -- remove whitespace
      if synci_terminal:sub(1, #"{synco") ~= "{synco" then
        U.error("'Sync' inport must be connected to a HRTIM Master or HRTIM Timing unit block.")
      end
      local bid = eval(synci_terminal)["synco"]["bid"]
      if bid == nil then
        U.error("'Sync' inport must be connected to a HRTIM Master or HRTIM Timing unit block.")
      end
      local extSyncSourceBlock = globals.instances[bid]
      local fun
      if extSyncSourceBlock:blockMatches('hrtim_master') then
        self.syncSourceType = 'hrtim_master'
        self.hrtim = extSyncSourceBlock:getParameter('hrtim')
        fun = 'HRTIM%d_SCOUT' % {self.hrtim}
      elseif extSyncSourceBlock:blockMatches('hrtim_unit') then
        self.syncSourceType = 'hrtim_unit'
        self.hrtim = extSyncSourceBlock:getParameter('hrtim')
        local subtimer = extSyncSourceBlock:getParameter('subtimer')
        globals.target.checkExtSyncHrtimConnection(subtimer)

        fun = 'HRTIM%d_SCOUT' % {self.hrtim}
      else
        U.error("'Sync' inport must be connected to a HRTIM Master or HRTIM Timing unit block.")
      end

      -- check if configured port/pin combination is valid
      local port = self.port
      local pin = self.pin
      local output_type = self.output_type
      local errMsgPrefix = 'Invalid or unsupported external synchronization function (%s) for pin %s%d configured in External Sync block.'
         % {fun, port, pin}

      local af = globals.target.getAlternateFunctionOrError({
        func = fun,
        pad = '%s%d' % {port, pin},
        opt_errMsgPrefix = errMsgPrefix,
      })

      self.ext_sync['config'] = {
        port = port,
        pin = pin,
        af = af,
        outputType = output_type
      }

      -- make parent hrtim object aware that we need to generate a sync out signal
      local parent_hrtim = self:getBlockInstanceWithMatchingParameter('hrtim', 'hrtim')

      parent_hrtim:addSyncOut({
        src_type = self.syncSourceType,
        sync_behaviour = self.sync_type_out
      })
      globals.syscfg:addEntry('ext_sync_out', {
        pins = {self.ext_sync['config']},
        path = self.path
      })
    end

    return {
      Require = Require
    }
  end

  function ExtSync:registerSinkBID(sinkBID)
    assert(U.isNonNegativeIntScalar(sinkBID))

    table.insert(self.sinkBIDs, sinkBID)
  end

  function ExtSync:startsTimerOnSync()
    return (self.sync_type_in == 1 or self.sync_type_in == 2)
  end

  function ExtSync:resetsTimerOnSync()
    return (self.sync_type_in == 2)
  end
  
  function ExtSync:finalizeThis(c)
    --[[
      The External Sync block can only have either TIM-based blocks (i.e. PWM
      blocks) or HRTIM-based blocks (i.e. HRTIM Master and HRTIM Unit blocks) as
      sinks. An error is thrown if both types of blocks are connected.
    --]]
    local hasHrtimSinks = false
    local hasTimSinks = false
    for _, bid in pairs(self.sinkBIDs) do
      local sinkBlock = globals.instances[bid]

      if sinkBlock:getType() == 'pwm' then
        hasTimSinks = true
      elseif sinkBlock:getType() == 'hrtim_master'
      or sinkBlock:getType() == 'hrtim_unit' then
        hasHrtimSinks = true
      else
        error('An invalid type of block is registered as the sink block connected to the External Sync block.')
      end
    end

    if hasTimSinks and hasHrtimSinks then
      local errMsg = [[


        The External Sync block @3 must be connected to either PWM or HRTIM blocks, but not both.
        
        Connected blocks currently include both PWM and HRTIM blocks:
      ]]
      local errMsgBlockPaths = {self:getPath()} -- Note, this block is @3
      for i, bid in pairs(self.sinkBIDs) do
        errMsg = errMsg..'\n@%d' % {i + 3} -- add list of blocks to msg

        table.insert(errMsgBlockPaths, globals.instances[bid]:getPath())
      end

      U.error(errMsg, errMsgBlockPaths)
    end
  end

  function ExtSync:finalize(c)
    if static[self.cpu].finalized then
      return
    end
    
    for _, bid in pairs(static[self.cpu].instances) do
      local extsync = globals.instances[bid]
      if extsync:getCpu() == self.cpu then
        extsync:finalizeThis(c)
      end
    end
    
    static[self.cpu].finalized = true  
  end

  return ExtSync
end

return Module
