--[[
  Copyright (c) 2021 by Plexim GmbH
  All rights reserved.

  A free license is granted to anyone to use this software for any legal
  non safety-critical purpose, including commercial applications, provided
  that:
  1) IT IS NOT USED TO DIRECTLY OR INDIRECTLY COMPETE WITH PLEXIM, and
  2) THIS COPYRIGHT NOTICE IS PRESERVED in its entirety.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
  OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
  AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
  OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
  SOFTWARE.
--]]

local Module = {}
local U = require('common.utils')

local static = {}

function Module.getBlock(globals, cpu)

  local HrtimMaster = require('common.block').getBlock(globals, cpu)
  if static[cpu] == nil then
    static[cpu] = {
      numInstances = 0,
      instances = {},  
      finalized = false,
      phaseInputs = {}, -- Array of phase inputs supplied to HRTIM Timing Units
                        -- synchronized by this HRTIM Master. Phase inputs are
                        -- strings that we retrieve using Block.InputSignal
                        -- such as "0.1f" or "subsystem.SineWave".
    }
  end
  HrtimMaster["instance"] = static[cpu].numInstances
  static[cpu].numInstances = static[cpu].numInstances + 1
  
  function HrtimMaster:checkMaskParameters()
    if type(Block.Mask.CarrierFreq)  == 'table' or Block.Mask.CarrierFreq <= 0 then
      U.error('Carrier frequency must be a scalar value greater than zero.')
    end

    if Block.Mask.BurstMode == 2 then -- burst mode enabled
      if type(Block.Mask.BurstFreq) == 'table' or Block.Mask.BurstFreq <= 0 then
        U.error('Burst mode frequency must be a scaler value greater than zero.')
      end
    end

    if Block.Mask.AdcTrigger ~=1 or Block.Mask.TaskTrigger~=1 then
      if (Block.Mask.RepCtrPeriod > 0xFF) or not U.isPositiveIntScalar(Block.Mask.RepCtrPeriod) then
        U.error('Repetition counter period must be a positive integer value between [1, 255].')
      end
    end
    
  end

  function HrtimMaster:createImplicit(hrtim, params, req)
    -- should only be called by a hrtim instance 

    self.hrtim = hrtim
    req:add('HRTIM%i.Master' % {self.hrtim})
    table.insert(static[self.cpu].instances, self.bid)
    self.is_configured = false
    self.config = {}
    self.num_phase_shifted_channels = 0
    self.has_var_freq = false
    self:logLine('HRTIM%i Master timer implicitly created.' % {self.hrtim})
    self['parent_instance'] = params.instance
    if params['burst_mode'] ~= nil then
      self['burst_mode'] = params.burst_mode
    end

    -- master timer is synced to an external event
    self.ext_sync = {}
    self.ext_sync.start_tim_on_sync = false
    self.ext_sync.reset_tim_on_sync = false

  end

  function HrtimMaster:p_getDirectFeedthroughCode()
    local Require = ResourceList:new()
    local InitCode = U.CodeLines:new()
    local OutputCode = U.CodeLines:new()
    local OutputSignal = StringList:new()

    table.insert(static[self.cpu].instances, self.bid)
    self.num_phase_shifted_channels = 0
    self.registered_channels = {}

    self.is_configured = false
    if Block.Mask.VarFreq == 2 then
      self.has_var_freq = true
    else
      self.has_var_freq = false
    end
    local hrtimTiming = globals.target.getHrtimPrescaleAndPeriod(Block.Mask.CarrierFreq, {
          varFreq = self.has_var_freq,
          returnStringErrors = true
        })

    self.config = {
      frequency = hrtimTiming.freq,
      period_in_timer_counts = hrtimTiming.period,
      prescale = hrtimTiming.prescale,
      mode = 'single_shot'
    }

    local repetition_counter = 1  --dummy
    if Block.Mask.AdcTrigger ~= 1 or Block.Mask.TaskTrigger ~= 1 then
      repetition_counter = Block.Mask.RepCtrPeriod
      OutputSignal:append("{modtrig = {bid = %i}}" % {HrtimMaster:getId()})
      OutputSignal:append("{adctrig = {bid = %i}}" % {HrtimMaster:getId()})
    else
      OutputSignal:append('{}')
      OutputSignal:append('{}')
    end
    self.config['repetition'] = repetition_counter

    self.hrtim = Block.Mask.HrtimUnit

    Require:add('HRTIM%i.Master' % {self.hrtim})
    local sync_signal = {}
    sync_signal[1] = "{synco = {bid = %i, hrtim = '%i', freq = '%f', varFreq = '%i'}}" % {self.bid, self.hrtim, self['config'].frequency, self.has_var_freq}
    OutputSignal:append(sync_signal)

    -- burst mode controller
    if Block.Mask.BurstMode == 2 then
      local timing = globals.target.getHrtimBurstModePrescaleAndPeriod(Block.Mask.BurstFreq)
 
      self['burst_mode'] = {}
      self['burst_mode'].period = timing.period
      self['burst_mode'].prescaler = timing.prescale
      self['burst_mode'].clk_source = 'independent'

      Require:add('HRTIM%i.BMC' % {self.hrtim})
    end

    -- see if there is a fitting parent HRTIM block in the circuit
    local parent_hrtim = self:getBlockInstanceWithMatchingParameter('hrtim', 'hrtim')

    if parent_hrtim == nil then
      -- create an implicit parent hrtim
      parent_hrtim = self:makeBlock('hrtim', self.cpu)
      parent_hrtim:createImplicit(self.hrtim, {
        needs_master_timer = true
      }, Require)
    end
    
    self['parent_instance'] = parent_hrtim:getObjIndex()


    -- master timer is synced to an external event
    self.ext_sync = {}
    self.ext_sync.start_tim_on_sync = false
    self.ext_sync.reset_tim_on_sync = false
    -- timer is synced to an external event
    if Block.Mask.ExternalSync == 2 then
      self.ext_sync['enabled'] = true
      self.ext_sync['trigger_type'] = Block.Mask.SyncTriggerType
    end

    local output_enable = 0
    if Block.Mask.OutputEnable == 2 then
      output_enable = 1
    end
    if output_enable == 1 then
      OutputCode:append("PLXHAL_HRTIM_enablePwmOutputByMaster(0, %s);" % {Block.InputSignal[1][1]})
    else
      OutputCode:append("PLXHAL_HRTIM_enablePwmOutputByMaster(0, %s);" % {1})
    end

    if self.has_var_freq == true then
      OutputCode:append('PLXHAL_HRTIM_scalePeriod(0, %s);' % {Block.InputSignal[2][1]})
    end

    if Block.Mask.BurstMode == 2 then
      OutputCode:append("PLXHAL_HRTIM_setBurstModeCompare(0, %s);" % {Block.InputSignal[3][1]})
    end

    self.is_configured = true
    return {
      InitCode = InitCode,
      OutputCode = OutputCode,
      OutputSignal = OutputSignal,
      Require = Require,
      UserData  = {bid = HrtimMaster:getId()}
    }
  end

  function HrtimMaster:p_getNonDirectFeedthroughCode()
    local Require = ResourceList:new()
    local UpdateCode = U.CodeLines:new()

    self.synciTerminal = Block.InputSignal[4][1] -- must be set before call to getExtSyncBlock()

    -- check synci port
    if self.ext_sync['enabled'] then
      local extSyncBlock = self:getExtSyncBlock()

      extSyncBlock:registerSinkBID(self:getId())

      self.ext_sync.start_tim_on_sync = extSyncBlock:startsTimerOnSync()
      self.ext_sync.reset_tim_on_sync = extSyncBlock:resetsTimerOnSync()

      -- check if configured port/pin combination is valid
      local port = extSyncBlock.port
      local pin = extSyncBlock.pin
      local pull = extSyncBlock.input_type
      local func = 'HRTIM%d_SCIN' % {self.hrtim}
      local errMsgPrefix = 'Invalid or unsupported external synchronization function (%s) for pin %s%d configured in External Sync block.'
         % {func, port, pin}

      local af = globals.target.getAlternateFunctionOrError({
        func = func,
        pad = '%s%d' % {port, pin},
        opt_errMsgPrefix = errMsgPrefix,
      })

      -- make parent hrtim object aware that we need an external sync in
      local parent_hrtim = self:getBlockInstanceWithMatchingParameter('hrtim', 'hrtim')
      
      self.ext_sync['config'] = {
        port = port,
        pin = pin,
        af = af,
        pull = pull
      }
      parent_hrtim:addSyncIn(self.ext_sync['config'])
      globals.syscfg:addEntry('ext_sync', {
        unit = self.unit,
        pins = {self.ext_sync['config']},
        path = extSyncBlock.path
      })
    end
    
    return {
      Require = Require,
      UpdateCode = UpdateCode
    }
  end

  function HrtimMaster:usesExtSync()
    return self.ext_sync.enabled
  end

  function HrtimMaster:getExtSyncBlock()
    if not self.ext_sync.enabled then
      error('Function getExtSycBlock() called even though external synchronization is not enabled.')
    end

    local synciTerminal = self.synciTerminal:gsub("%s+", "") -- remove whitespace
    if synciTerminal:sub(1, #"{ext_synco") ~= "{ext_synco" then
      U.error("'Sync' inport must be connected to an External Sync block.")
    end
    local bid = eval(synciTerminal)["ext_synco"]["bid"]
    if not bid then
      error('External synchronization chain is broken.')
    end

    local extSyncBlock = globals.instances[bid]

    return extSyncBlock
  end

  function HrtimMaster:getInjAdcTrgSrcMacro()
    return 'LL_ADC_INJ_TRIG_EXT_HRTIM_TRG2'
  end

  function HrtimMaster:getRegAdcTrgSrcMacro()
    return 'LL_ADC_REG_TRIG_EXT_HRTIM_TRG1'
  end

  function HrtimMaster:getAdcPostScaler()
    return self.config.repetition
  end

  function HrtimMaster:getVarFreqCode()
    local varFreqCode
    if self.has_var_freq == true then
      varFreqCode = 'PLXHAL_HRTIM_applyPeriod(0); \n'
    end
    return varFreqCode
  end

  --[[
  Purpose
    HRTIM Timing Units synchronized by an HRTIM Master register themselves with
    the HRTIM Master block as channels. Each channel corresponds to a subtimer.
    
  Optional Phase Shift
    Optionally, HRTIM Timing Units can have a phase input. Phase shifts are
    inserted using compare values on the HRTIM Master. The HRTIM Master has
    four compare registers that can be used for this purpose. As a result, there
    can be up to four unique phase shifts per HRTIM Master. If HRTIM Timing
    Units have identical phase shifts, the same compare event can be used for
    multiple HRTIM Timing units.
    
  Return Values
    If the HRTIM Timing Unit registering itself using this function has a phase
    shift, this function will return the index of the compare register on the
    HRTIM Master that is used to create this phase shift. Otherwise, it will
    return nil.
  --]]
  function HrtimMaster:registerChannel(subtimer, params)
    assert(U.isChar(subtimer))
    U.enforceParamContract(
      params,
      {
        opt_phaseInput = U.isString,
      })

    table.insert(self.registered_channels, subtimer)

    if params.opt_phaseInput then
      -- Check if opt_phaseInput is identical to the phaseInput of a different
      -- HRTIM Timing Unit.
      for cmpIdx, phaseInput in ipairs(static[self.cpu].phaseInputs) do
        if phaseInput == params.opt_phaseInput then
          return cmpIdx
        end
      end

      -- If opt_phaseInput is new, we insert it in the phaseInputs array. The
      -- number of the array element is automatically incremented and
      -- corresponds to the index of the HRTIM Master compare register used for
      -- the phase shift.
      table.insert(static[self.cpu].phaseInputs, params.opt_phaseInput)
      local cmpIdx = #(static[self.cpu].phaseInputs)

      if cmpIdx > 4 then
        U.error([[
          A maximum of four phase shifts can be applied to HRTIM Timing Units synchronized by one HRTIM Master.
        ]])
      end
      return cmpIdx
    end
  end

  function HrtimMaster:setSinkForTriggerSource(sink)
    if sink ~= nil then
      self:logLine('My trigger is connected to %s' % {dump(sink)})
      if self[sink.type] == nil then
        self[sink.type] = {}
      end
      table.insert(self[sink.type], globals.instances[sink.bid])    
    end 
  end

  function HrtimMaster:propagateTriggerSampleTime(ts)
    local achievableTs
    if self.config['mode'] == 'pcc' then
      if self['base_trig_frequency'] ~= nil then
        achievableTs = 1/self['base_trig_frequency']
      end
      if achievableTs ~= nil then
        self:logLine('Propagating sample time %f s' % {achievableTs})
      end
    else
      achievableTs = 1/self.config['frequency']*self.config['repetition']
    end
    if self['modtrig'] ~= nil then
      for _, b in ipairs(self['modtrig']) do
        local f = b:propagateTriggerSampleTime(achievableTs)
      end
    end
    if self['adctrig'] ~= nil then
      for _, b in ipairs(self['adctrig']) do
        local f = b:propagateTriggerSampleTime(achievableTs)
      end
    end
  end

  function HrtimMaster:canBeImplicitTaskTriggerOriginBlock()
    return true
  end

  function HrtimMaster:requestImplicitTrigger(ts)
    local achievableTs
    if not self.is_configured then
        -- the master is available to generate any frequency trigger
        local hrtimTiming = globals.target.getHrtimPrescaleAndPeriod(1/ts, {
          varFreq = self.has_var_freq,
        })
        if type(hrtimTiming) == 'string' then
          return nil
        end
        self['base_trig_frequency'] = hrtimTiming.freq
        self.config['frequency'] = hrtimTiming.freq
        self.config['period_in_timer_counts'] = hrtimTiming.period
        self.config['prescale'] = hrtimTiming.prescale
        self.config['repetition'] = 1

        self:logLine('Offered trigger generator at %f Hz' % {hrtimTiming.freq})
        achievableTs = 1/hrtimTiming.freq
    else
      achievableTs = 1/self.config['frequency']*self.config['repetition']
      self:logLine('Offered trigger generator at %f Hz' % {1/achievableTs})
    end
    return achievableTs
  end

  function HrtimMaster:getPreInitCode()
    local isAdcTrig = false
    local adcTrigType
    if self.adctrig then
      local b = self.adctrig[1]
      if b:usesInjectedConversions() then
        adcTrigType = 'inj'
      end
      isAdcTrig = true
    end
    if isAdcTrig then
      if self:targetMatches({'f3', 'h7'}) then
        if self.config.repetition > 1 then
          U.error('Repetition counter period in HRTIM Master must be set to 1 on this target if the timing unit provides the ADC trigger.')
        end
      elseif not self:targetMatches('g4') then
        U.throwUnhandledTargetError()
      end
    end

    local timerSetupCode = self.globals.target.getHighResolutionMasterTimerSetupCode(self.hrtim, {
      period = self.config.period_in_timer_counts,
      prescale = self.config.prescale,
      repetition = self.config.repetition,
      reset_tim_on_sync = self.ext_sync.reset_tim_on_sync,
      start_tim_on_sync = self.ext_sync.start_tim_on_sync,
      mode = self.config.mode,
      adctrig = isAdcTrig,
      adctrigType = adcTrigType
    })

    local preInitCode = [[
      {
        %(timerSetupCode)s
        PLX_HRTIM_timerSetup(HrtimHandles[%(instance)d], PLX_HRTIM_MASTER);
      }
    ]] % {
      timerSetupCode = timerSetupCode,
      instance = self.parent_instance,
    }

    return preInitCode
  end

  function HrtimMaster:getBurstModeCode()
    local code = globals.target.getBurstModeSetupCode(self.hrtim, {
      period = self['burst_mode'].period,
      prescaler = self['burst_mode'].prescaler,
      clk_source = self['burst_mode'].clk_source
    })
    return code
  end
  
  function HrtimMaster:finalizeThis(c)
    local isModTrigger = false   
    if self['modtrig'] ~= nil then
      for _, b in ipairs(self['modtrig']) do
        if b:blockMatches('tasktrigger') then
          isModTrigger = true
          break
        end
      end
    end

    if isModTrigger == true then
      local itFunction = [[
      void HRTIM%i_Master_IRQHandler(void)
      {
          LL_HRTIM_ClearFlag_REP(HRTIM%i, LL_HRTIM_TIMER_MASTER);
          DISPR_dispatch();
      }
      ]]        
      c.Declarations:append("%s\n" % {itFunction % {self['hrtim'], self['hrtim']}})
      c.InterruptEnableCode:append('HAL_NVIC_SetPriority(HRTIM%i_Master_IRQn, configLIBRARY_MAX_SYSCALL_INTERRUPT_PRIORITY, 0);'  % {self['hrtim']})
      c.InterruptEnableCode:append('HAL_NVIC_EnableIRQ(HRTIM%i_Master_IRQn);' % {self['hrtim']})
      c.Declarations:append([[
        bool %(base_name)s_checkOverrun()
        {
          return HAL_NVIC_GetPendingIRQ(HRTIM%(hrtim)i_Master_IRQn);
        }
        ]] % {
          base_name = Target.Variables.BASE_NAME,
          hrtim = self.hrtim
      })
      c.PostInitCode:append('PLX_HRTIM_enableRepIT(HrtimHandles[%i], PLX_HRTIM_MASTER);' % {self['parent_instance']})
    end

    if self.has_var_freq == true then
      local subtimer_apply_code = ''
      for _, subtimer in pairs(self.registered_channels) do
        subtimer_apply_code = subtimer_apply_code .. [[
          PLX_HRTIM_applyPeriod(HrtimHandles[aHandle], PLX_HRTIM_T%s);
        ]] % {subtimer}
      end

      local varFreqApplyCode = [[
        extern PLX_HRTIM_Handle_t HrtimHandles[];
        void PLXHAL_HRTIM_applyPeriod(uint16_t aHandle)
        {
          PLX_HRTIM_applyPeriod(HrtimHandles[aHandle], PLX_HRTIM_MASTER);
          %(subtimerApplyCode)s
        }
      ]] % {
        hrtim = self.hrtim,
        subtimerApplyCode = subtimer_apply_code,
      }

      c.Declarations:append(varFreqApplyCode)
    end
    if self['burst_mode'] ~= nil then
      c.Declarations:append([[
        extern PLX_HRTIM_Handle_t HrtimHandles[];
        void PLXHAL_HRTIM_enableBurstMode(uint16_t aHandle, float aValue)
        {
          PLX_HRTIM_enableBurstMode(HrtimHandles[aHandle], aValue != 0.0f);
        }

        void PLXHAL_HRTIM_setBurstModePeriod(uint16_t aHandle, uint32_t aPeriod)
        {
          PLX_HRTIM_setBurstModePeriod(HrtimHandles[aHandle], aPeriod);
        }

        void PLXHAL_HRTIM_setBurstModeCompare(uint16_t aHandle, float aValue)
        {
          PLX_HRTIM_setBurstModeCompare(HrtimHandles[aHandle], aValue);
        }
      ]])
    end
  end

  function HrtimMaster:finalize(c)
    if static[self.cpu].finalized then
      return
    end

    for _, bid in pairs(static[self.cpu].instances) do
        local hrtim = globals.instances[bid]
        if hrtim:getCpu() == self.cpu then
          hrtim:finalizeThis(c)
        end
    end

    static[self.cpu].finalized = true  
  end
  
  return HrtimMaster
end

return Module
