--[[
  Copyright (c) 2021 by Plexim GmbH
  All rights reserved.

  A free license is granted to anyone to use this software for any legal
  non safety-critical purpose, including commercial applications, provided
  that:
  1) IT IS NOT USED TO DIRECTLY OR INDIRECTLY COMPETE WITH PLEXIM, and
  2) THIS COPYRIGHT NOTICE IS PRESERVED in its entirety.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
  OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
  AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
  OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
  SOFTWARE.
--]]

local Module = {}
local U = require('common.utils')

local static = {}

function Module.getBlock(globals, cpu)

  local HrtimPcc = require('common.block').getBlock(globals, cpu)
  if static[cpu] == nil then
    static[cpu] = {
      numInstances = 0,
      instances = {},
      finalized = false,  
    }
  end
  HrtimPcc["instance"] = static[cpu].numInstances
  static[cpu].numInstances = static[cpu].numInstances + 1
  
  function HrtimPcc:checkMaskParameters()
  end
    
  function HrtimPcc:p_getDirectFeedthroughCode()
    local Require = ResourceList:new()
    local InitCode = U.CodeLines:new()
    local OutputCode = U.CodeLines:new()
    local OutputSignal = StringList:new()
    
    self.hrtim = Block.Mask.HrtimUnit
    static[self.cpu].instances[self.hrtim] = self.bid
   
    OutputSignal:append("{}")
    OutputSignal:append("{}")

    local analogFltLineObj = self:getBlockInstance('analog_fault_line')
    -- check if analog protection signals are configured or not

    local flt_state = {1, 1}
    if U.arrayIsEmpty(analogFltLineObj:getParameter('analog_flt_signals')) then
      flt_state = {0, 0}
    end

    local sync_in = false
    if Block.Mask.ExternalSync == 2 then -- sync in enabled
      sync_in = true
      -- TODO: check if an external sync block is connected or not
    end

    self.has_var_freq = false

    -- burst mode controller
    if Block.Mask.BurstMode == 2 then -- burst mode enabled
      local timing = globals.target.getHrtimBurstModePrescaleAndPeriod(Block.Mask.BurstFreq)

      self['burst_mode'] = {}
      self['burst_mode'].period = timing.period
      self['burst_mode'].prescaler = timing.prescale
      self['burst_mode'].clk_source = 'independent'
      OutputCode:append("PLXHAL_HRTIM_enableBurstMode(0, 1);")
      OutputCode:append("PLXHAL_HRTIM_setBurstModeCompare(0, %s);" % {Block.InputSignal[2][1]})
    end
    
    -- create an implicit parent HRTIM
    self.parentHrtimObj = self:makeBlock('hrtim', self.cpu)
    self.parentHrtimObj:createImplicit(self.hrtim, {
      needs_master_timer = true,
      burst_mode = self['burst_mode']
    }, Require)
       
    local freqDesired = Block.Mask.CarrierFreq

    -- HRTIM timing
    local hrtimTiming = globals.target.getHrtimPrescaleAndPeriod(freqDesired, {
      varFreq = self.has_var_freq
    })

    if Block.Mask.CarrierFreqTol == 1 then  -- enforce exact value (no rounding)
      U.enforceExactFrequencyTolerance(
        {
          freqDesired = freqDesired, 
          freqAchievable = hrtimTiming.freq,
          descriptor = 'PWM frequency',
          

        })
    end
    
    -- common settings
    local leading_edge_blanking_time_s = Block.Mask.LeadingEdgeBlanking
    -- NOTE: we might want to make this configurable, but for now hard-code at 0.5% 
    local force_on_delay_s = 0.005/Block.Mask.CarrierFreq -- time at which output is forced on in case current was still above ramp at period
    
    -- process channel configurations
    self.channels = {}
    local subtimers_used = {}
    local comparators_used = {}
    local dac_in_n_used = {}
    local compParams = globals.target.getTargetParameters()['comps']
    local dacParams = globals.target.getTargetParameters()['dacs']
    
    local ch = 1
    local tab = 1
    while Block.Mask['Ch%iMode' % {tab}] ~= nil do
      if Block.Mask['Ch%iMode' % {tab}] ~= 1 then -- 1 = channel disabled
        local channel_mode
        if Block.Mask['Ch%iMode' % {tab}] == 2 then
          channel_mode = 'single'
        else
          channel_mode = 'complementary'
        end
      
        local dead_time = Block.Mask['Ch%iDeadTime' % {tab}]
        
        local polarity = 0
        if Block.Mask['Ch%iOutPol' % {tab}] ~= 1 then
          polarity = 1
        end
        
        local subtimer = string.char(Block.Mask['Ch%iTimingUnit' % {tab}] + 64)
        if subtimers_used[subtimer] ~= nil then
          U.error('Channels %i and %i are configured for the same timing unit %s.' % {subtimers_used[subtimer], tab, subtimer})
        end
        subtimers_used[subtimer] = tab
        
        -- find fitting comparator unit
        local sense_port = string.char(Block.Mask['Ch%iSensePort' % {tab}] + 64)
        local sense_pin = Block.Mask['Ch%iSensePin' % {tab}]
        local comparator_unit = 1
        local desired_gpio = '%s%i' % {sense_port, sense_pin}
        while compParams['COMP%i' % {comparator_unit}] ~= nil do
          local match_found = false
          if compParams['COMP%i' % {comparator_unit}].gpio_in_p ~= nil then
            for gpio, _ in pairs(compParams['COMP%i' % {comparator_unit}].gpio_in_p) do
              if gpio == desired_gpio then
                -- Only consider comparator units with hrtim_eev feature
                if compParams['COMP%i' % {comparator_unit}].hrtim_eev then
                  match_found = true
                  break
                end
              end
            end
          end
          if match_found == true then
            break
          end
          comparator_unit = comparator_unit + 1
        end
        if compParams['COMP%d' % {comparator_unit}] == nil then
          U.error('No comparator unit found with positive input %s.' % {desired_gpio})
        end
        if comparators_used[comparator_unit] ~= nil then
          U.error('Channels %i and %i are configured for the same comparator COMP%i.' % {comparators_used[comparator_unit], tab, comparator_unit})
        end
        comparators_used[comparator_unit] = tab
               
        -- compile list of suitable dacs
        local dac_in_n_options = {}
        for dac, params in pairs(dacParams) do
          local this_dac_unit = string.byte(dac, -1) - 48
          local match_found = false
          if params.num_channels ~= nil then
            for this_dac_channel = 1, params.num_channels do
              for dac_in_n, _ in pairs(compParams['COMP%i' % {comparator_unit}].dac_in_n) do
                if dac_in_n == 'DAC%i_CH%i' % {this_dac_unit, this_dac_channel} then
                  dac_in_n_options[dac_in_n] = {
                    unit = this_dac_unit,
                    channel = this_dac_channel
                  }
                end
              end
            end
          end
        end
        
        -- select fastest available dac
        local dac_unit
        local dac_channel
        local max_samples_per_second = 0
        for dac_in_n, params in pairs(dac_in_n_options) do
          if dac_in_n_used[dac_in_n] == nil then
            local this_max_samples_per_second = dacParams['DAC%i' % {params.unit}].max_samples_per_second
            if this_max_samples_per_second > max_samples_per_second then
              dac_unit = params.unit
              dac_channel = params.channel
              max_samples_per_second = this_max_samples_per_second
            end
          end
        end
        if dac_unit == nil then
          U.error("Unable to find a fitting DAC for channel %i" % {tab})
        end               
        dac_in_n_used['DAC%i_CH%i' % {dac_unit, dac_channel}] = tab
        
        local comparator_event = compParams['COMP%i' % {comparator_unit}].hrtim_eev[1]
        
        self.channels[ch] = {
          channel_mode = channel_mode,
          subtimer = subtimer,
          dac_unit = dac_unit,
          dac_channel = dac_channel,
          comparator_unit = comparator_unit,
          comparator_event = comparator_event,
          sense_port = sense_port,
          sense_pin = sense_pin,
          leading_edge_blanking_time_s = leading_edge_blanking_time_s,
          force_on_delay_s = force_on_delay_s, -- time at output is forced on in case current still above ramp at period
          dead_time = dead_time,
          polarity = polarity
        }
        ch = ch + 1
      end
      tab = tab + 1
    end

    -- generate configuration code
    for ch, config in pairs(self.channels) do
       OutputCode:append("PLXHAL_HRTIM_PCC_setPeakCurrent(%i, %i, %s);" % {self.instance, ch-1, Block.InputSignal[1][ch]})
      
      local event = {
        src = 'COMP%i_OUT' % {config.comparator_unit},
        polarity = 1,
        sensitivity = 'LEVEL',
        enable_fastmode = false
      }
      self.parentHrtimObj:addEvent(config.comparator_event, event)
      
      local dacRate = globals.target.getDacUpdateRate(config.dac_unit, Block.Mask.RampSlope, Block.Mask.SenseGain, hrtimTiming.clk, hrtimTiming.period, 0.01)
      
      -- add HRTIM timer unit
      local reset_tim_on_sync = false
      
      local hrtimTimerUnitParams = {
        f = freqDesired,
        carrier = 'sawtooth',
        mode = 'pcc',
        phase_shift = false,
        hasVarFreq = self.has_var_freq,
        op_mode = {
          channel_mode = config.channel_mode,
          force_on_delay_s = config.force_on_delay_s, 
          leading_edge_blanking = {
            time_s = config.leading_edge_blanking_time_s,
            event = config.comparator_event
          },
          reset_event = config.comparator_event,
          dac_step_rate = dacRate,
          dead_time = config.dead_time,
          polarity = config.polarity,
          reset_tim_on_sync = reset_tim_on_sync,
          idle_level = {0, 0},  -- TODO: make this in the mask dialog available?
          flt_state = flt_state -- TODO: make this in the mask dialog available?
        },
        repetition = {
          period = 1,
          m_rollover = 'BOTH'
        }
      }

      local hrtim_unit = self:makeBlock('hrtim_unit', self.cpu)
      hrtim_unit:createImplicit(self.hrtim, {
        subtimer = config.subtimer,
        params = hrtimTimerUnitParams
      }, Require)

      hrtim_unit:configureTimerUnit(config.subtimer, hrtimTimerUnitParams)

      if self['burst_mode'] ~= nil then
        hrtim_unit:enableBurstMode()
      end
     
      self.parentHrtimObj:addChannel(config.subtimer, {
        config = hrtim_unit['config'],
        bid = hrtim_unit.bid,
        is_synced_to_master = false
      }, Require)
      
      local slope = -math.abs(Block.Mask.RampSlope * Block.Mask.SenseGain / dacRate)
          
      -- create an implicit DAC for ramp-compensated setpoint
      local dacObj
      for _, b in ipairs(globals.instances) do
        if b:blockMatches('dac') then
          if b:getParameter('dac') == config.dac_unit then
            dacObj = b
            break
          end
        end
      end
      
      if dacObj == nil then
        dacObj = self:makeBlock('dac', self.cpu)
        dacObj:createImplicit(config.dac_unit, {
        }, Require)
      end
      
      local connect_gpio = false
      --if config.dac_unit == 1 then
      --  connect_gpio = true
      --end

      -- only use waveform generator if a slope != 0 is configured --> backwards compatibility with F3 series.
      local waveform_gen
      if Block.Mask.RampSlope ~= 0 then
        waveform_gen = {
            type = 'sawtooth',
            bias = math.floor(0x1000/Target.Variables.AdcVRef*Block.Mask.RampOffset * Block.Mask.SenseGain+0.5),
            increment = math.floor(0x10000/Target.Variables.AdcVRef*slope+0.5)
          }
      end
      
      dacObj:addChannel(config.dac_channel, {
        scale = Block.Mask.SenseGain,
        offset = Block.Mask.RampOffset * Block.Mask.SenseGain,
        ll_config = {
          trig_src = 'DAC_TRIGGER_HRTIM_RST_TRG%i' % {string.byte(config.subtimer)-64},
          trig_src2 = 'DAC_TRIGGER_HRTIM_STEP_TRG%i' % {string.byte(config.subtimer)-64},
          waveform_gen = waveform_gen,
          connect_gpio = connect_gpio
        },
        path = self:getName()
      }, Require)
     
      config.dac_obj = dacObj
      
      -- create implicit comparator
      local compObj = self:makeBlock('comp', self.cpu)
      compObj:createImplicit(config.comparator_unit, {
        inp_port = config.sense_port,
        inp_pin = config.sense_pin,
        inm = 'DAC%d_CH%d' % {config.dac_unit, config.dac_channel},
        path = self:getName()
      }, Require)
    end

    return {
      InitCode = InitCode,
      OutputCode = OutputCode,
      OutputSignal = OutputSignal,
      Require = Require,
      UserData  = {bid = HrtimPcc:getId()}
    }
  end
  
  function HrtimPcc:p_getNonDirectFeedthroughCode()
    local Require = ResourceList:new()
    local UpdateCode = U.CodeLines:new()
    --only last channel has to generate enable code
    local enableCode = self.parentHrtimObj:getEnableCode(#self.channels)
    if enableCode ~= nil then
      UpdateCode:append(enableCode)
    end
    
    return {
      Require = Require,
      UpdateCode = UpdateCode
    }
  end

  function HrtimPcc:getInjAdcTrgSrcMacro()
    return 'LL_ADC_INJ_TRIG_EXT_HRTIM_TRG2'
  end

  function HrtimPcc:getRegAdcTrgSrcMacro()
    return 'LL_ADC_REG_TRIG_EXT_HRTIM_TRG1'
  end

  function HrtimPcc:getAdcPostScaler()
    -- The HrtimPcc block does not provide an ADC post scaler.
  end
  
  function HrtimPcc:finalizeThis(c)
    -- important: generate into PostInitCode since this is a composite block and we are referencing other handles!
    for ch, config in pairs(self.channels) do
      c.PostInitCode:append('PLX_HRTIM_PCC_configureChannel(HrtimPccHandles[%i], %i, DacHandles[%i], %i, HrtimHandles[%i], PLX_HRTIM_T%s);' % 
        {self['instance'], ch-1, config.dac_obj:getObjIndex(), config.dac_channel-1, self.parentHrtimObj:getObjIndex(), config.subtimer})
    end
  end
  
  function HrtimPcc:finalize(c)
    if static[self.cpu].finalized then
      return
    end
  
    c.Include:append('plx_hrtimer_pcc.h')
    c.Declarations:append('PLX_HRTIM_PCC_Handle_t HrtimPccHandles[%i];' % {static[self.cpu].numInstances})
    c.Declarations:append('PLX_HRTIM_PCC_Obj_t HrtimPccObj[%i];' % {static[self.cpu].numInstances})
    
    c.Declarations:append('void PLXHAL_HRTIM_PCC_setPeakCurrent(uint16_t aHandle, uint16_t aChannel, float aCurrent){')
    c.Declarations:append('  PLX_HRTIM_PCC_setPeakCurrent(HrtimPccHandles[aHandle], aChannel, aCurrent);')
    c.Declarations:append('}')
    
    local code = [[
      for (int i = 0; i < %d; i++) {
        HrtimPccHandles[i] = PLX_HRTIM_PCC_init(&HrtimPccObj[i], sizeof(HrtimPccObj[i]));
      }
    ]]
    c.PreInitCode:append(code % {static[self.cpu].numInstances})
  
    for _, bid in pairs(static[self.cpu].instances) do
        local hrtim_pcc = globals.instances[bid]
        if hrtim_pcc:getCpu() == self.cpu then
          hrtim_pcc:finalizeThis(c)
        end
    end
       
    static[self.cpu].finalized = true  
  end
 
  return HrtimPcc
end

return Module
