--[[
  Copyright (c) 2021 by Plexim GmbH
  All rights reserved.

  A free license is granted to anyone to use this software for any legal
  non safety-critical purpose, including commercial applications, provided
  that:
  1) IT IS NOT USED TO DIRECTLY OR INDIRECTLY COMPETE WITH PLEXIM, and
  2) THIS COPYRIGHT NOTICE IS PRESERVED in its entirety.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
  OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
  AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
  OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
  SOFTWARE.
--]]

local U = require('common.utils')

local static = {}

local Module = {}

function Module.getBlock(globals, cpu)
  local HrtimUnit = require('common.block').getBlock(globals, cpu)
  if static[cpu] == nil then
    static[cpu] = {
      numInstances = 0,
      instances = {},
      subtimers_per_hrtim = {},
      powerstage = nil,
      finalized = false,
    }
  end
  HrtimUnit.instance = static[cpu].numInstances
  static[cpu].numInstances = static[cpu].numInstances + 1

  function HrtimUnit:checkMaskParameters()

    self.syncMode = U.enforceMask_newComboBox('ExternalSync',
                                              {'Self', 'Master', 'External'})

    if self.syncMode.equals('Self') then
      if type(Block.Mask.CarrierFreq) == 'table' or Block.Mask.CarrierFreq <= 0 then
        U.error('Carrier frequency must be a scalar value greater than zero.')
      end
    end
    -- Midpoint (for sawtooth carrier) trigger is not available if both channels run in independent mode
    if (Block.Mask.CarrierType == 1 and Block.Mask.TrigEvent == 2) then
      if Block.Mask.ChOutput == 3 then
        U.error(
          'Midpoint sampling not available for 2 independent output channels.')
      end
    end

    if Block.Mask.AdcTrigger ~= 1 or Block.Mask.TaskTrigger ~= 1 then
      if (Block.Mask.RepCtrPeriod > 0xFF) or not U.isPositiveIntScalar(Block.Mask.RepCtrPeriod) then
        U.error(
          'Repetition counter period must be a positive integer value between [1, 255].')
      end
    end
  end

  function HrtimUnit:createImplicit(hrtim, params, req)
    self.hrtim = hrtim
    table.insert(static[self.cpu].instances, self.bid)
    self.subtimer = params.subtimer
    self.subtimer_enum = string.byte(params.subtimer) - string.byte('A')
    self.is_synced_to_master = false
    self.syncMode = U.newComboBox('Self', {'Self', 'Master', 'External'})

    self:logLine('HRTIM%i %s timer implicitly created.' %
      {self.hrtim, self.subtimer})
    -- see if there is a fitting parent HRTIM block in the circuit
    local parent_hrtim = self:getBlockInstanceWithMatchingParameter('hrtim',
                                                                    'hrtim')

    if parent_hrtim == nil then
      -- create an implicit parent hrtim
      parent_hrtim = require('blocks.hrtim').getBlock({
        target = globals.target, instances = globals.instances
      })('hrtim')
      parent_hrtim:createImplicit(
        self.hrtim, {
          needs_master_timer = self.is_synced_to_master,
        }, req)
    end
    self.parent_instance = parent_hrtim:getObjIndex()
    self.parent_hrtim = parent_hrtim

    self.params = params.params
  end

  function HrtimUnit:p_getDirectFeedthroughCode()
    local Require = ResourceList:new()
    local InitCode = U.CodeLines:new()
    local OutputCode = U.CodeLines:new()
    local OutputSignal = StringList:new()

    table.insert(static[self.cpu].instances, self.bid)
    -- Main configuration
    self.hrtim = Block.Mask.HrtimUnit
    self.subtimer = string.char(65 + Block.Mask.TimingUnit - 1)  -- pulldown starts with 'A'
    self.subtimer_enum = Block.Mask.TimingUnit                   -- PLX_HRTIM_Timer_t starts with 'MASTER = 0'

    -- Channel configurations
    local output_channel = nil
    local channel_mode
    local idle_level = {}
    local flt_state = {}
    local polarity = {0, 0}
    if Block.Mask.ChMode == 1 then
      output_channel = Block.Mask.ChOutput
      channel_mode = 'independent'
      if Block.Mask.ChOutput == 1 then
        idle_level = {Block.Mask.IdleState1, 0}
        polarity = {Block.Mask.Ch1Polarity - 1, 0}
        flt_state = {Block.Mask.FaultState1, 0}
      elseif Block.Mask.ChOutput == 2 then
        idle_level = {0, Block.Mask.IdleState2}
        polarity = {0, Block.Mask.Ch2Polarity - 1}
        flt_state = {0, Block.Mask.FaultState2}
      else
        idle_level = {Block.Mask.IdleState1, Block.Mask.IdleState2}
        polarity = {Block.Mask.Ch1Polarity - 1, Block.Mask.Ch2Polarity - 1}
        flt_state = {Block.Mask.FaultState1, Block.Mask.FaultState2}
      end
    else
      channel_mode = 'complementary'
      idle_level = {Block.Mask.IdleState1, Block.Mask.IdleState2}
      flt_state = {Block.Mask.FaultState1, Block.Mask.FaultState2}
      if Block.Mask.Polarity ~= 1 then
        polarity = {1, 1}
      end
    end

    local analogFltLineObj = self:getBlockInstance('analog_fault_line')
    -- check if analog protection signals are configured or not

    if U.arrayIsEmpty(analogFltLineObj:getParameter('analog_flt_signals')) then
      -- override flt_state if no analog protection signals are configured in the Coder options
      flt_state = {0, 0}
    end

    if static[self.cpu].subtimers_per_hrtim[self.hrtim] == nil then
      static[self.cpu].subtimers_per_hrtim[self.hrtim] = 1
    else
      static[self.cpu].subtimers_per_hrtim[self.hrtim] = 
         static[self.cpu].subtimers_per_hrtim[self.hrtim] + 1
    end

    -- Sync configuration
    local carriers = {'sawtooth', 'symmetrical'}    -- must match mask dialog combo box
    local sampling_points = {'period', 'midpoint'}  -- must match mask dialog combo box
    local carrier = carriers[Block.Mask.CarrierType]
    local availableCarriers = 
       globals.target.getTargetParameters().hrtims.carrier
    if not U.arrayContainsValue(availableCarriers, carrier) then
      U.error('Carrier type "%s" not available on target %s.' %
        {carrier, globals.target.getChipName()})
    end
    local operation_mode
    local phase_shift = false
    local variable_period = false
    self.ext_sync = {}
    if self.syncMode.equals('Self') then
      self.is_synced_to_master = false
      if Block.Mask.VarFreq == 2 then
        variable_period = true
      end
      operation_mode = 'continuous'
    elseif self.syncMode.equals('Master') then
      self.is_synced_to_master = true
      if Block.Mask.PhaseShift == 2 then
        phase_shift = true
      end
      if Block.Mask.BurstMode == 2 then
        self.burst_mode = true
      end
      operation_mode = 'single_shot'
    elseif self.syncMode.equals('External') then
      self.is_synced_to_master = false
      operation_mode = 'continuous'
      self.ext_sync.enabled = true
    end
    local sampling_point
    if carrier == 'sawtooth' and Block.Mask.AdcTrigger == 2 then
      sampling_point = sampling_points[Block.Mask.TrigEvent]
    else
      sampling_point = sampling_points[1]  -- default
    end

    -- triggering
    local repetition_per
    local m_repetition_rollover
    if Block.Mask.AdcTrigger ~= 1 or Block.Mask.TaskTrigger ~= 1 then
      repetition_per = Block.Mask.RepCtrPeriod
      if 2 * math.floor(repetition_per / 2) == repetition_per then
        if Block.Mask.RepCtrEvent == 1 then
          m_repetition_rollover = 'RST'
        else
          m_repetition_rollover = 'PER'
        end
      else
        m_repetition_rollover = 'BOTH'
      end
      OutputSignal:append('{modtrig = {bid = %i}}' % {HrtimUnit:getId()})
      OutputSignal:append('{adctrig = {bid = %i}}' % {HrtimUnit:getId()})
    else
      OutputSignal:append('{}')
      OutputSignal:append('{}')
      repetition_per = 1  -- dummy
      m_repetition_rollover = 'BOTH'
    end

    local dead_time
    if channel_mode ~= 'independent' then
      dead_time = Block.Mask.DeadTime
    end

    self.params = {
      f = Block.Mask.CarrierFreq,
      carrier = carrier,
      mode = operation_mode,
      phase_shift = phase_shift,
      variable_period = variable_period,
      op_mode = {
        channel_mode = channel_mode,
        dead_time = dead_time,
        polarity = polarity,
        output_channel = output_channel,
        idle_level = idle_level,
        flt_state = flt_state
      },
      repetition = {
        period = repetition_per,
        m_rollover = m_repetition_rollover,
      },
      sampling_point = sampling_point
    }

    local sync_signal = {}
    sync_signal[1] = "{synco = {bid = %d, hrtim = '%d'}}" %
       {self.bid, self.hrtim}
    OutputSignal:append(sync_signal)

    return {
      InitCode = InitCode,
      OutputCode = OutputCode,
      OutputSignal = OutputSignal,
      Require = Require,
      UserData = {bid = HrtimUnit:getId()}
    }
  end

  function HrtimUnit:p_getNonDirectFeedthroughCode()
    local Require = ResourceList:new()
    local UpdateCode = U.CodeLines:new()

    local local_enable = Block.Mask.ShowEnable == 2

    local inIdx = U.enum({'m', 'en', 'synci', 'ph', 'f'})

    local syncOptions = {
      Master = {
        trigType = 'synco',
        blockName = 'HRTIM Master',
        blockType = 'hrtim_master',
      },
      External = {
        trigType = 'ext_synco',
        blockName = 'External Sync',
        blockType = 'extsync',
      },
    }

    function getSuggestedSyncSettingsForSourceBlock(blockType)
      for k, v in pairs(syncOptions) do
        if v.blockType == blockType then
          return k, v.trigType
        end
      end
    end

    -- see if there is a fitting parent HRTIM block in the circuit
    local parent_hrtim = self:getBlockInstanceWithMatchingParameter('hrtim',
                                                                    'hrtim')

    if parent_hrtim == nil then
      -- create an implicit parent hrtim
      parent_hrtim = self:makeBlock('hrtim', self.cpu)
      parent_hrtim:createImplicit(
        self.hrtim, {
          needs_master_timer = self.is_synced_to_master,
        }, Require)
    end

    self.parent_instance = parent_hrtim:getObjIndex()

    if not self.syncMode.equals('Self') then
      -- a synci must be appropriately connected for Master and External modes
      local expectedSource = syncOptions[self.syncMode.asString()]
      if not expectedSource then
        error('Developer needs to correct syncOptions table above.')
      end

      local synciTrig = U.portParseTriggerInput(inIdx.synci)
      local sourceBID
      if synciTrig then
        sourceBID = synciTrig.bid
        self.opt_sourceBlock = sourceBID and globals.instances[sourceBID]
      end
  
      if not self.opt_sourceBlock                                     -- No trigger connected
      or expectedSource.blockType ~= self.opt_sourceBlock:getType()   -- Synchronization setting is wrong
      or expectedSource.trigType ~= synciTrig.trigType then  -- Right block, wrong trigger output
        
        local selectedSyncModeVal = self.syncMode.asString()

        -- If there is a source block, check what kind and if valid suggest a change
        -- to the ExternalSync setting:
        local opt_sourceBlockText = ''
        if self.opt_sourceBlock then
          local suggestedSyncSetting, suggestedTrig = getSuggestedSyncSettingsForSourceBlock(self.opt_sourceBlock:getType())
          if  suggestedSyncSetting
          and not self.syncMode.equals(suggestedSyncSetting) then  -- Would be a valid block with change to Sync menu
            opt_sourceBlockText =
              "The currently connected block, @3, would be a valid synchronization signal source if @param:ExternalSync: is set to '%s'.\n\n" 
              % {suggestedSyncSetting}
          end
          if not suggestedSyncSetting                 -- not a valid blockType
          or suggestedTrig ~= synciTrig.trigType then -- or wrong trigger connected
            opt_sourceBlockText = opt_sourceBlockText..
            'The currently connected trigger of @3 cannot be used as a synchronization signal for the HRTIM Timing Unit.'
          end
        end
        U.error([[
          The 'Sync' inport of the HRTIM timing unit must be connected to the 'Sync' output of an %(expectedBlock)s block when the @param:ExternalSync: is set to '%(selectedSyncModeVal)s'.

          %(opt_sourceBlockText)s
          ]] % {
                  selectedSyncModeVal = selectedSyncModeVal,
                  expectedBlock = expectedSource.blockName,
                  opt_sourceBlockText = opt_sourceBlockText,
                }, self.opt_sourceBlock and {self.opt_sourceBlock:getPath()} or {})
      end

      -- for Master mode, confirm the HRTIM unit config is compatible:
      if self.syncMode.equals('Master') then
        -- Master output includes:
        -- {synco = {bid = %i, hrtim = '%i', freq = '%f', varFreq = '%i'}}
        local master_hrtim = synciTrig.synco.hrtim
        if not master_hrtim then
          error([[
            Source block is not configured correctly, 
            Master HRTIM blocks must provide an hrtim unit number.]])
        end

        if math.floor(master_hrtim) ~= self.hrtim then -- Weird, do we need to floor numbers from strings??
          -- This error is obsolete now that we only allow HRTIM1, 
          -- but can be triggered manually by incrementing the hrtim above.
          U.error([[
            The @param:HrtimUnit:3: of the HRTIM Master Block, @3, 
            must match the @param:HrtimUnit: of the synchronized HRTIM Timing unit, @1.
          ]], {self.opt_sourceBlock:getPath()})
        end

        -- specific master check
        local varFreqFromMaster = false
          
        local masterFreq = synciTrig.synco.freq -- FIX ME: Why not tonumber()
        -- set carrier frequency to master frequency if subtimer is synced to master
        self.params.f = masterFreq
        local masterHasVarFreq = tonumber(synciTrig.synco.varFreq)
        if masterHasVarFreq == 1 then
          varFreqFromMaster = true
        end
        self.params.masterHasVarFreq = varFreqFromMaster


      elseif self.syncMode.equals('External') then
        -- specific checks on External Sync block.
        local trigType = synciTrig.ext_synco.type
        if trigType == 1 then      -- Start on sync
          self.params.op_mode.start_tim_on_sync = true
        elseif trigType == 2 then  -- Start and reset on sync
          self.params.op_mode.start_tim_on_sync = true
          self.params.op_mode.reset_tim_on_sync = true
        end
  
        local extSyncBlock = globals.instances[sourceBID]

        extSyncBlock:registerSinkBID(self:getId())

        -- check if configured port/pin combination is valid
        local port = extSyncBlock.port
        local pin = extSyncBlock.pin
        local pull = extSyncBlock.input_type
        local fun = 'HRTIM%d_SCIN' % {self.hrtim}
        local errMsgPrefix =
            'Invalid @param:Port:3: and @param:Pin:3: combination (P%s%d) selected for @3.' % {port, pin}
  
        local af = globals.target.getAlternateFunctionOrError({
          func = fun,
          pad = '%s%d' % {port, pin},
          opt_errMsgPrefix = errMsgPrefix,
          opt_errArgs = {extSyncBlock:getPath()}
        })
  
        -- make parent hrtim object aware that we need an external sync in
        self.ext_sync.config = {
          port = port,
          pin = pin,
          af = af,
          pull = pull
        }
        parent_hrtim:addSyncIn(self.ext_sync.config)
        globals.syscfg:addEntry('ext_sync', {
          unit = self.unit,
          pins = {self.ext_sync.config},
          path = extSyncBlock.path
        })
      end

    end

    self:configureTimerUnit(self.subtimer, self.params)

    local freqDesired = self.params.f
    local freqAchievable = self.config.timing.frequency

    if Block.Mask.CarrierFreqTol == 1 then  -- enforce exact value (no rounding)
      U.enforceExactFrequencyTolerance(
        {
          freqDesired = self.params.f,
          freqAchievable = self.config.timing.frequency,
          descriptor = 'PWM frequency',
          opt_carrierFrequency = Block.Mask.CarrierFreq,
        }
      )
    end

    -- add channel to parent
    parent_hrtim:addChannel(self.subtimer, {
                              config = self.config,
                              bid = self.bid,
                              is_synced_to_master = self.is_synced_to_master,
                            }, Require)

    if self.is_synced_to_master then
      --[[
        If the HRTIM Timing Unit registering itself using this function has a
        phase shift, this function will return the index of the compare register
        on the HRTIM Master that is used to create this phase shift. Otherwise,
        it will return nil.
      ]]
      local opt_cmpIdx = parent_hrtim:registerChannelAtMaster(self.subtimer, {
        opt_phaseInput = self.params.phase_shift and Block.InputSignal[4][1] or nil
      })
      if self.params.phase_shift == true then
        self.opt_phaseShiftCmp = opt_cmpIdx
      end
      if self.params.masterHasVarFreq == true then
        local varFreqCode = parent_hrtim:getVarFreqCode()
        if varFreqCode ~= nil then
          UpdateCode:append(varFreqCode)
        end
      end
    end

    if self.params.phase_shift == true then
      local phaseShiftInput = Block.InputSignal[4][1]
      UpdateCode:append('PLXHAL_HRTIM_setPhase(0, %d, %s);' %
        {self.instance, phaseShiftInput})
      self.opt_constantPhaseShiftInput = U.portValueGetConstant(4)
    end

    if self.params.variable_period == true then
      UpdateCode:append('PLXHAL_HRTIM_scalePeriodHrtimUnit(0, %i, %s);' %
        {self.subtimer_enum, Block.InputSignal[5][1]})
    end

    if self.params.op_mode.channel_mode == 'independent' then
      local fcnName = 'PLXHAL_HRTIM_setDuty'
      if self.params.carrier == 'sawtooth' and self.params.sampling_point == 'midpoint' then
        fcnName = fcnName..'AndSampling'
      end
      local enableSignal1, enableSignal2
      if local_enable then
        enableSignal1 = Block.InputSignal[2][1]
        if self.params.op_mode.output_channel == 3 then
          enableSignal2 = Block.InputSignal[2][2]
        end
      else
        enableSignal1 = '1'
        enableSignal2 = '1'
      end
      if self.params.op_mode.output_channel == 1 then
        UpdateCode:append('%s(0, %i, %i, %s, %s);' %
          {fcnName, self.subtimer_enum, 0, Block.InputSignal[1][1], enableSignal1})
      elseif self.params.op_mode.output_channel == 2 then
        UpdateCode:append('%s(0, %i, %i, %s, %s);' %
          {fcnName, self.subtimer_enum, 1, Block.InputSignal[1][1], enableSignal1})
      else  -- two independent channels
        UpdateCode:append('%s(0, %i, %i, %s, %s);' %
          {fcnName, self.subtimer_enum, 0, Block.InputSignal[1][1], enableSignal1})
        UpdateCode:append('%s(0, %i, %i, %s, %s);' %
          {fcnName, self.subtimer_enum, 1, Block.InputSignal[1][2], enableSignal2})
      end
    else  -- complementary
      local fcnName = 'PLXHAL_HRTIM_setDutyComplementary'
      local enableSignal
      if local_enable then
        enableSignal = Block.InputSignal[2][1]
      else
        enableSignal = '1'
      end
      if self.params.carrier == 'sawtooth' and self.params.sampling_point == 'midpoint' then
        fcnName = fcnName..'AndSampling'
      end
      UpdateCode:append('%s(0, %i, %s, %s);' %
        {fcnName, self.subtimer_enum, Block.InputSignal[1][1], enableSignal})
    end
    local last_subtimer = static[self.cpu].subtimers_per_hrtim[self.hrtim]
    local enableCode = parent_hrtim:getEnableCode(last_subtimer)
    if enableCode ~= nil then
      UpdateCode:append(enableCode)
    end

    return {
      Require = Require,
      UpdateCode = UpdateCode
    }
  end

  function HrtimUnit:getInjAdcTrgSrcMacro()
    local adcTrg = globals.target.getAvailableInjAdcTrg(self.subtimer_enum)
    return 'LL_ADC_INJ_TRIG_EXT_HRTIM_TRG%d' % {adcTrg}
  end

  function HrtimUnit:getRegAdcTrgSrcMacro()
    local adcTrg = globals.target.getAvailableRegAdcTrg(self.subtimer_enum)
    return 'LL_ADC_REG_TRIG_EXT_HRTIM_TRG%d' % {adcTrg}
  end

  function HrtimUnit:getAdcPostScaler()
    return self.params.repetition.period
  end

  function HrtimUnit:setSinkForTriggerSource(sink)
    if sink ~= nil then
      if self[sink.type] == nil then
        self[sink.type] = {}
      end
      table.insert(self[sink.type], globals.instances[sink.bid])
    end
  end

  function HrtimUnit:propagateTriggerSampleTime(ts)
    local achievableTs = 1 / self.config.timing.frequency *
       self.params.repetition.period
    if self.params.carrier == 'symmetrical' then
      -- two update events per PWM period
      achievableTs = achievableTs / 2
    end
    if self.modtrig ~= nil then
      for _, b in ipairs(self.modtrig) do
        local f = b:propagateTriggerSampleTime(achievableTs)
      end
    end
    if self.adctrig ~= nil then
      for _, b in ipairs(self.adctrig) do
        local f = b:propagateTriggerSampleTime(achievableTs)
      end
    end
  end

  function HrtimUnit:canBeImplicitTaskTriggerOriginBlock()
    return true
  end

  function HrtimUnit:requestImplicitTrigger(ts)
    local achievableTs = 1 / self.config.timing.frequency *
       self.params.repetition.period
    if self.params.carrier == 'symmetrical' then
      -- two update events per PWM period
      achievableTs = achievableTs / 2
    end
    self:logLine('Offered trigger generator at %f Hz' % {1 / achievableTs})
    return achievableTs
  end

  function HrtimUnit:configureTimerUnit(unit, params)
    local timing
    local cmp_init
    local events = {}

    if params.mode == 'pcc' then
      timing = globals.target.getHrtimPrescaleAndPeriod(params.f, {
        varFreq = params.masterHasVarFreq
      })
      -- configure force-on delay (needed if current still above ramp at period)
      local comp1_init = math.floor(
        params.op_mode.force_on_delay_s * timing.freq * timing.period + 0.5)
      if comp1_init == 0 then
        comp1_init = 1
      end
      -- configure leading edge blanking time
      local comp3_init = math.floor(
        params.op_mode.leading_edge_blanking.time_s * timing.freq * timing.period + 0.5)

      -- configure cmp2 value for providing DAC step
      local comp2_init = timing.clk / params.op_mode.dac_step_rate
      -- configure leading edge blanking event
      if comp3_init > comp1_init then
        events[params.op_mode.leading_edge_blanking.event] = {
          filter = 'BLANKINGCMP3',
          latch_enabled = false,
        }
      end
      cmp_init = {comp1_init, comp2_init, comp3_init, 0}
    elseif (params.mode == 'continuous') or (params.mode == 'single_shot') then
      if params.carrier == 'symmetrical' then
        timing = globals.target.getHrtimPrescaleAndPeriod(2 * params.f, {
          varFreq = params.masterHasVarFreq
        })
      else
        timing = globals.target.getHrtimPrescaleAndPeriod(params.f, {
          varFreq = params.masterHasVarFreq
        })
      end
    else
      U.error('Operating mode not supported.')
    end

    local config = {
      timing = {
        frequency = timing.freq,
        period_in_timer_counts = timing.period,
        prescale = timing.prescale,
      },
      cmp_init = cmp_init,
      events = events,
      reset_trigger = 1,
      dead_time = {}
    }
    if params.carrier == 'symmetrical' then
      config.timing.frequency = timing.freq / 2
    end
    if params.op_mode.dead_time ~= nil then
      local deadTimeTiming = globals.target.getHrtimDeadtimePrescaleAndCounts(
        params.op_mode.dead_time)

      config.dead_time.counts = deadTimeTiming.counts
      config.dead_time.prescale = deadTimeTiming.prescale
    end

    local output_channel = 1
    if params.op_mode.channel_mode == 'independent' then
      if params.op_mode.output_channel == 1 then
        output_channel = 1
      elseif params.op_mode.output_channel == 2 then
        output_channel = 2
      else
        output_channel = 3
      end
    end

    if output_channel == 1 or output_channel == 3 then
      local pad = globals.target.getHrtimGpio(self.hrtim, unit, 1)
      if pad == nil then
        U.error('Variant %s does not provide resource "HRTIM%d.%s"' %
          {globals.target.getFullChipName(), self.hrtim, unit})
      end
      local port1 = pad.port
      local pin1 = pad.pin

      local af1 = globals.target.getAlternateFunctionOrError({
        func = 'HRTIM%d_CH%s1' % {self.hrtim, unit},
        pad = '%s%d' % {port1, pin1},
      })

      local set1, reset1
      if params.mode == 'pcc' then
        if params.op_mode.reset_tim_on_sync == true then
          set1 = {'TIMPER', 'TIMCMP1', 'RESYNC'}
        else
          set1 = {'TIMPER', 'TIMCMP1'}
        end
        reset1 = {'EEV_%i' % {params.op_mode.reset_event}}
      elseif params.mode == 'continuous' then -- continuous mode implies external synchronization
        if params.carrier == 'sawtooth' then
          if params.op_mode.reset_tim_on_sync == true then
            set1 = {'TIMPER', 'RESYNC'}
          else
            set1 = {'TIMPER'}
          end
        end
        reset1 = {'TIMCMP1'}
      elseif params.mode == 'single_shot' then -- single-shot mode implies synchronization by a master HRTIM
        if params.carrier == 'sawtooth' then
          if params.phase_shift == true then
            set1 = {'MASTERCMP'}  -- if phase-shift is enabled, the set1 must be set to MASTERCMP1 ...MASTERCMP4
          else
            set1 = {'MASTERPER'}
          end
        end
        reset1 = {'TIMCMP1'}
      else
        U.error('Operating mode not supported.')
      end
      config.ll_ch1_config = {
        port = port1,
        pin = pin1,
        af = af1,
        set = set1,
        reset = reset1,
        polarity = params.op_mode.polarity[1],
        idle_level = params.op_mode.idle_level[1],
        flt_state = params.op_mode.flt_state[1],
      }
    end

    if params.op_mode.channel_mode == 'complementary' or output_channel == 2 or output_channel == 3 then
      local pad = globals.target.getHrtimGpio(self.hrtim, unit, 2)
      if pad == nil then
        U.error('Variant %s does not provide resource "HRTIM%d.%s"' %
          {globals.target.getFullChipName(), self.hrtim, unit})
      end
      local port2 = pad.port
      local pin2 = pad.pin

      local af2 = globals.target.getAlternateFunctionOrError({
        func = 'HRTIM%d_CH%s2' % {self.hrtim, unit},
        pad = '%s%d' % {port2, pin2},
      })

      local set2, reset2
      if params.mode == 'pcc' then
        if params.op_mode.reset_tim_on_sync == true then
          set2 = {'TIMPER', 'RESYNC'}
          reset2 = {'TIMCMP3'}
        else
          set2 = {'TIMPER'}
          reset2 = {'TIMCMP3'}
        end
      elseif params.mode == 'continuous' then -- continuous mode implies external synchronization
        if params.carrier == 'sawtooth' then
          if params.op_mode.reset_tim_on_sync == true then
            set2 = {'TIMPER', 'RESYNC'}
          else
            set2 = {'TIMPER'}
          end
        end
        reset2 = {'TIMCMP3'}
      elseif params.mode == 'single_shot' then -- single-shot mode implies synchronization by a master HRTIM
        if params.carrier == 'sawtooth' then
          if params.phase_shift == true then
            set2 = {'MASTERCMP'}  -- if phase-shift is enabled, the set1 must be set to MASTERCMP1 ...MASTERCMP4
          else
            set2 = {'MASTERPER'}
          end
        end
        reset2 = {'TIMCMP3'}
      else
        U.error('Operating mode not supported.')
      end
      config.ll_ch2_config = {
        port = port2,
        pin = pin2,
        af = af2,
        set = set2,
        reset = reset2,
        polarity = params.op_mode.polarity[2],
        idle_level = params.op_mode.idle_level[2],
        flt_state = params.op_mode.flt_state[2],
      }
    end

    local pinconf = {}
    for ch = 1, 2 do
      local llconf = config['ll_ch%d_config' % ch]
      if llconf ~= nil then
        table.insert(pinconf, {
          port = llconf.port,
          pin = llconf.pin,
          af = llconf.af,
        })
      end
    end

    globals.syscfg:addEntry('hrtim', {
      unit = self.hrtim,
      pins = pinconf,
      path = self:getName()
    })

    -- apply configuration to subtimer object
    self.config = config
  end

  function HrtimUnit:enableBurstMode()
    self.burst_mode = true
  end

  function HrtimUnit:getPreInitCode()
    -- see if there is a powerstage protection block in the circuit
    if static[self.cpu].powerstage == nil then
      static[self.cpu].powerstage = self:getBlockInstance('powerstage')
    end

    local analog_flt_signals = {}
    if static[self.cpu].powerstage ~= nil then
      local flt_signals = static[self.cpu].powerstage:getAnalogProtectionSignals()
      if not U.arrayIsEmpty(flt_signals) then
        for _, flt in pairs(flt_signals) do
          local flt_line = 
             globals.target.getTargetParameters().comps
             ['COMP%d' % {flt.comp_unit}].hrtim_flt
          local eev_lines = 
             globals.target.getTargetParameters().comps
             ['COMP%d' % {flt.comp_unit}].hrtim_eev
          if flt_line ~= nil then
            table.insert(analog_flt_signals, flt_line)
          elseif eev_lines ~= nil then  -- HRTIM has no compatible fault line available --> try using an external event channel
            for _, eev in pairs(eev_lines) do
              local eev_flt_line = 
                 globals.target.getTargetParameters().hrtims.fault_inputs
                 ['eev%d' % {eev}]
              if eev_flt_line ~= nil then
                table.insert(analog_flt_signals, eev_flt_line)
                break
              end
            end
          end
        end
      end
    end

    local channelSetupCode = ''
    for ch = 1, 2 do
      local llconf = self.config['ll_ch%d_config' % ch]
      if llconf then
        channelSetupCode = channelSetupCode..
           self.globals.target.getHighResolutionTimerChannelSetupCode(
             self.hrtim, {
               set = llconf.set,
               reset = llconf.reset,
               polarity = llconf.polarity,
               subtim = self.subtimer,
               channel = ch,
               idle_level = llconf.idle_level,
               burst_mode = self.burst_mode,
               cmp = self.opt_phaseShiftCmp,
               carrier = self.params.carrier,
               flt_state = llconf.flt_state,
               parentInstance = self.parent_instance,
               isSyncedToMaster = self.is_synced_to_master,
             })
      end
    end

    local subtimFltEnableCode = ''
    if #analog_flt_signals > 0 then
      subtimFltEnableCode = globals.target.getHrtimSubtimerFaultEnableCode(self.hrtim, {
        subtimer = self.subtimer,
        flt_lines = analog_flt_signals
      })
    end

    local isAdcTrig = false
    local adcUnit
    local adcTrigType
    if self.adctrig then
      local b = self.adctrig[1]
      adcUnit = b:getParameter('adcUnit')
      if b:usesInjectedConversions() then
        adcTrigType = 'inj'
      end
      isAdcTrig = true
    end

    if isAdcTrig then
      if self:targetMatches({'f3', 'h7'}) then
        if self.params.repetition.period > 1 then
          U.error(
            'Repetition counter period in HRTIM Timing Unit %s must be set to 1 on this target if the timing unit provides the ADC trigger.' %
            {self.subtimer})
        end
      elseif not self:targetMatches('g4') then
        U.throwUnhandledTargetError()
      end
    end

    local subtimSetupCode = self.globals.target.getHighResolutionSubTimerSetupCode(
      self.hrtim,
      {
        subtimer = self.subtimer,
        subtimer_enum = self.subtimer_enum,
        mode = self.params.mode,
        carrier = self.params.carrier,
        timing = self.config.timing,
        dead_time = self.config.dead_time,
        cmp_init = self.config.cmp_init,
        trigger = {
          is_adc_trigger = isAdcTrig,
          adc_trigger_unit = adcUnit,
          sampling_point = self.params.sampling_point,
          adc_trig_type = adcTrigType,
        },
        repetition = self.params.repetition,
        events = self.config.events,
        phase_shift = self.params.phase_shift,
        cmp = self.opt_phaseShiftCmp,
        reset_trigger = self.config.reset_trigger,
        reset_tim_on_sync = self.params.op_mode.reset_tim_on_sync,
        start_tim_on_sync = self.params.op_mode.start_tim_on_sync,
      })

    local code = [[
      {
        %(subtimSetupCode)s
        PLX_HRTIM_timerSetup(HrtimHandles[%(instance)d], PLX_HRTIM_T%(m_subtim)s);
        %(channelSetupCode)s
        %(subtimFltEnableCode)s
      }
    ]] % {
      subtimSetupCode = subtimSetupCode,
      instance = self.parent_instance,
      m_subtim = self.subtimer,
      channelSetupCode = channelSetupCode,
      subtimFltEnableCode = subtimFltEnableCode,
    }

    return code
  end

  function HrtimUnit:checkExtSyncChainIsValid()
    if self.syncMode.equals('Self') then
      -- No external sync chain exists in this case
      return
    end

    if self.syncMode.equals('External') then
      local extSyncBlock = self.opt_sourceBlock

      if not extSyncBlock:resetsTimerOnSync() then
        -- No external sync chain verification is done in this case
        return
      end

      local errorLinkBlockPaths = {
        self:getPath(),             -- 3
        extSyncBlock:getPath(),     -- 4
      }

      if self.params.carrier == 'sawtooth' then
        U.warning([[
          If @param:CarrierType:3: is set to 'Sawtooth', the HRTIM Timing Unit can be synchronized to an external synchronization signal.

          To accomplish this, the @param:CarrierFreq:3:, f_HRTIM, must be lower than the frequency of the external synchronization signal supplied to @4, f_EXT.

          Additionally, the duty cycle must be scaled by f_HRTIM / f_EXT.
        ]], errorLinkBlockPaths)
      elseif self.params.carrier == 'symmetrical' then
        U.warning([[
          If @param:CarrierType:3: is set to 'Symmetrical', the @param:CarrierFreq:3:, f_HRTIM, must be an integer multiple of the frequency of the external synchronization signal supplied to @4, f_EXT.
        ]], errorLinkBlockPaths)
      else
        error('Invalid carrier configured.')
      end
    elseif self.syncMode.equals('Master') then
      local hrtimMasterBlock = self.opt_sourceBlock

      if not hrtimMasterBlock:usesExtSync() then
        -- No external sync chain exists in this case
        return
      end

      local extSyncBlock = hrtimMasterBlock:getExtSyncBlock()

      if not extSyncBlock:resetsTimerOnSync() then
        -- No external sync chain verification is done in this case
        return
      end

      local errorLinkBlockPaths = {
        self:getPath(),             -- 3
        hrtimMasterBlock:getPath(), -- 4
        extSyncBlock:getPath(),     -- 5
      }

      if not self.params.phase_shift then -- no phase shift hrtim unit
        U.error([[
          Only HRTIM Timing Units with @param:PhaseShift:3: enabled can be synchronized by an HRTIM Master with @param:ExternalSync:4: set to 'External' and itself synchronized by an External Sync block with @param:SyncBehaviourIn:5: set to 'Start and reset on sync'.
        ]], errorLinkBlockPaths)
      else
        if self.params.carrier == 'sawtooth' then
          U.warning([[
            If @param:CarrierType:3: is set to 'Sawtooth' and @param:PhaseShift:3: is enabled, an HRTIM Timing Unit can be synchronized to an external signal under certain conditions.
            
            Let f_HRTIM be the @param:CarrierFreq:4: and f_EXT be the frequency of the external synchronization signal. Then, the following must hold:
            
            1 - f_EXT / f_HRTIM < phase < f_HRTIM / f_EXT.

            Additionally, the phase and the duty cycle must be scaled by f_HRTIM / f_EXT.
          ]], errorLinkBlockPaths)
        elseif self.params.carrier == 'symmetrical' then
          if self.opt_constantPhaseShiftInput == 0 then
            U.warning([[
              If @param:CarrierType:3: is set to 'Symmetrical' and @param:PhaseShift:3: is enabled and set to 0, the @param:CarrierFreq:4:, f_HRTIM, must be an integer multiple of the frequency of the external synchronization signal supplied to @5, f_EXT.
            ]], errorLinkBlockPaths)
          else -- phase shift other than 0
            U.error([[
              If @param:CarrierType:3: is set to 'Symmetrical', only HRTIM Timing Units with @param:PhaseShift:3: enabled and set to 0 can be synchronized by an HRTIM Master with @param:ExternalSync:4: set to 'External' and itself synchronized by an External Sync block with @param:SyncBehaviourIn:5: set to 'Start and reset on sync'.
            ]], errorLinkBlockPaths)
          end
        else
          error('Invalid carrier configured.')
        end
      end
    end
  end


  function HrtimUnit:finalizeThis(c)
    -- Check if the synchronization chain is invalid
    self:checkExtSyncChainIsValid()

    local isModTrigger = false
    if self.modtrig ~= nil then
      for _, b in ipairs(self.modtrig) do
        if b:blockMatches('tasktrigger') then
          isModTrigger = true
          break;
        end
      end
    end
    if isModTrigger == true then

      c.Declarations:append([[
        void HRTIM%(hrtim)d_TIM%(subtimer)s_IRQHandler(void) {
          LL_HRTIM_ClearFlag_REP(HRTIM%(hrtim)d, LL_HRTIM_TIMER_%(subtimer)s);
          DISPR_dispatch();
        }

        bool %(base_name)s_checkOverrun() {
          return HAL_NVIC_GetPendingIRQ(HRTIM%(hrtim)d_TIM%(subtimer)s_IRQn);
        }
        ]] % {
        base_name = Target.Variables.BASE_NAME,
        hrtim = self.hrtim,
        subtimer = self.subtimer,
      })

      c.InterruptEnableCode:append(
        [[
          HAL_NVIC_SetPriority(HRTIM%(hrtim)d_TIM%(subtimer)s_IRQn, configLIBRARY_MAX_SYSCALL_INTERRUPT_PRIORITY, 0);
          HAL_NVIC_EnableIRQ(HRTIM%(hrtim)d_TIM%(subtimer)s_IRQn);
        ]] % {
          hrtim = self.hrtim, 
          subtimer = self.subtimer})
      

      c.PostInitCode:append(
        'PLX_HRTIM_enableRepIT(HrtimHandles[%(instance)d], PLX_HRTIM_T%(subtimer)s);' %
        {instance = self.parent_instance, subtimer = self.subtimer})
    end
  end

  function HrtimUnit:finalize(c)
    if static[self.cpu].finalized then
      return
    end

    for _, bid in pairs(static[self.cpu].instances) do
      local hrtim = globals.instances[bid]
      if hrtim:getCpu() == self.cpu then
        hrtim:finalizeThis(c)
      end
    end

    static[self.cpu].finalized = true
  end

  return HrtimUnit
end

return Module
