--[[
  Copyright (c) 2022 by Plexim GmbH
  All rights reserved.

  A free license is granted to anyone to use this software for any legal
  non safety-critical purpose, including commercial applications, provided
  that:
  1) IT IS NOT USED TO DIRECTLY OR INDIRECTLY COMPETE WITH PLEXIM, and
  2) THIS COPYRIGHT NOTICE IS PRESERVED in its entirety.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
  OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
  AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
  OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
  SOFTWARE.
--]]--
local Module = {}
local U = require('common.utils')
local Coder = require('Coder')

local static = {numInstances = 0}

local function getHtmlPinMapTableRow(params)
  local left_odd = ""
  local left_even = ""
  local right_odd = ""
  local right_even = ""

  local classR1 = 'right'
  local classR2 = 'right'
  local classL1 = 'left'
  local classL2 = 'left'

  if params.left_odd ~= nil then
    if params.left_odd:sub(1, #"static:") == "static:" then
      left_odd = params.left_odd:sub(#"static:"+1, -1) -- remove whitespace
      classR1 = 'rightStatic'
    else
      left_odd = params.left_odd
    end
  else
    classR1 = 'rightStatic'
  end

  if params.left_even ~= nil then
    if params.left_even:sub(1, #"static:") == "static:" then
      left_even = params.left_even:sub(#"static:"+1, -1) -- remove whitespace
      classL1 = 'leftStatic'
    else
      left_even = params.left_even
    end
  else
    classL1 = 'leftStatic'
  end

  if params.right_odd ~= nil then
    if params.right_odd:sub(1, #"static:") == "static:" then
      right_odd = params.right_odd:sub(#"static:"+1, -1) -- remove whitespace
      classR2 = 'rightStatic'
    else
      right_odd = params.right_odd
    end
  else
    classR2 = 'rightStatic'
  end

  if params.right_even ~= nil then
    if params.right_even:sub(1, #"static:") == "static:" then
      right_even = params.right_even:sub(#"static:"+1, -1) -- remove whitespace
      classL2 = 'leftStatic'
    else
      right_even = params.right_even
    end
  else
    classL2 = 'leftStatic'
  end

  local row = [[
        <tr>
          <th class="%(classR1)s"> <div style="overflow: scroll; width: 150px; white-space: nowrap;"> %(left_odd)s </div></th>
          <th class="padding"> </th>
          <th class="%(classL1)s"> <div style="overflow: scroll; width: 150px; white-space: nowrap;"> %(left_even)s </div></th>
          <th class="%(classR2)s"> <div style="overflow: scroll; -ms-overflow-style: none; scrollbar-width: none; width: 150px; white-space: nowrap;">  %(right_odd)s </div></th>
          <th class="padding"> </th>
          <th class="%(classL2)s"> <div style="overflow: scroll; width: 150px; white-space: nowrap;">  %(right_even)s </div></th>
        </tr>
  ]] % {
    classR1 = classR1,
    classR2 = classR2,
    classL1 = classL1,
    classL2 = classL2,
    left_odd = left_odd,
    left_even = left_even,
    right_odd = right_odd,
    right_even = right_even
  }
  return row
end

local function generateGenericHtmlTable(pinmap)
  local htmlCode = [[
    <style type="text/css">
    .tg  {border-collapse:collapse;border-spacing:0;}
    .tg td{border-color:black;border-style:solid;border-width:1px;font-family:Arial, sans-serif;font-size:14px;
      overflow:hidden;padding:10px 16px;word-break:normal;}
    .tg th{border-color:black;border-style:solid;border-width:1px;font-family:Arial, sans-serif;font-size:14px;
      font-weight:normal;overflow:hidden;padding:10px 16px;word-break:normal;}
    .tg .tg-uu6m{background-color:#dae8fc;font-family:Arial, Helvetica, sans-serif !important;font-size:12px;font-weight:bold;
      text-align:left;vertical-align:middle}
    .tg .tg-z56b{font-family:Arial, Helvetica, sans-serif !important;font-size:12px;text-align:left;vertical-align:middle}
    </style>
    <table class="tg">
    <thead>
      <tr>
        <th class="tg-uu6m">Pin name</th>
        <th class="tg-uu6m">Signal</th>
        <th class="tg-uu6m">Target Block</th>
      </tr>
    </thead>
    <tbody>
  ]]
  for pad, gpio in pairs(pinmap) do
    local row = [[
      <tr>
        <th class="tg-zu9e">%(pad)s</th>
        <th class="tg-z56b">%(description)s</th>
        <th class="tg-z56b">%(path)s</th>
      </tr>
    ]] % {
      pad = pad, 
      description = gpio[1].description,
      path = gpio[1].path
    }
    htmlCode = htmlCode .. row
  end

  local code = [[
    </tbody>
    </table>
  ]]
  htmlCode = htmlCode .. code

  return htmlCode
end

local function generateNuceloPinviewHtmlCode(pinmap, env)
  local htmlCode = ''
  local code = [[
    <!DOCTYPE html PUBLIC "-//W3C//DTD HTML 4.01//EN" "http://www.w3.org/TR/html4/strict.dtd">
    <html>
    <head>
      <meta http-equiv="Content-Type" content="text/html; charset=utf-8">
      <title>Nucleo %(famPrefix)s pin map</title>
      <link href="%(path)s/pinmap_style.css" rel="stylesheet" type="text/css">
    </head>
    <body>
    <div id="header">
      <h2>%(title)s</h2>
    </div>
    <div id="content">
      <div class="demo-container">
        <table>
          <tr>
              <th class="header"> </td>
          </tr>
  ]] % {
    famPrefix = env.target.getFamilyPrefix():upper(),
    path = Target.Variables.BUILD_ROOT,
    title = 'Pin map for: %s' % {Target.Variables.BASE_NAME},
  }
  htmlCode = htmlCode .. code


  local fullPinMap = {}
  local nbrOfRows 
  if env.target.getFamilyPrefix() == 'h7' then
    nbrOfRows = 35
  else
    nbrOfRows = 19
  end
  for i = 1, nbrOfRows do
    fullPinMap[i] = {}
  end
  env.target.addStaticInformationToPinmap(fullPinMap)
  for pad, gpio in pairs(pinmap) do
    local loc = env.target.getGpioLocation(pad)
    if gpio[1].description == nil then
      gpio[1].description = ''
    end
    if loc ~= nil and loc.pin ~= nil then
      if loc.con == 'CN7' or loc.con == 'CN11' then
        if loc.pin % 2 == 0 then
          fullPinMap[loc.pin/2][2] = 'P%s: %s (%s)' % {pad, gpio[1].description, gpio[1].path}
        else
          fullPinMap[(loc.pin+1)/2][1] = 'P%s: %s (%s)' % {pad, gpio[1].description, gpio[1].path}
        end
      else -- CN10 or CN12
        if loc.pin % 2 == 0 then
          fullPinMap[loc.pin/2][4] = 'P%s: %s (%s)' % {pad, gpio[1].description, gpio[1].path}
        else
          fullPinMap[(loc.pin+1)/2][3] = 'P%s: %s (%s)' % {pad, gpio[1].description, gpio[1].path}
        end
      end
    else
      U.warning('P%s cannot be represented in seleted pinout view.' % {pad})
    end
  end
  for _, row in ipairs(fullPinMap) do
    local rowCode = getHtmlPinMapTableRow({
      left_odd = row[1],
      left_even = row[2],
      right_odd = row[3],
      right_even = row[4]
    })
    htmlCode = htmlCode .. rowCode
  end


  local code = [[
          </table>
        </div>
      </div>
    </body>
    </html>
  ]]
  htmlCode = htmlCode .. code

  return htmlCode
end

function Module.getBlock(globals, cpu)
  -- only one pinmap instance per target
  local PinMap = require('common.block').getBlock(globals, cpu)
  PinMap["instance"] = static.numInstances
  static.numInstances = static.numInstances + 1

  function PinMap:p_getDirectFeedthroughCode()
    U.error("Explicit use of PinMap:p_getDirectFeedthroughCode via target block not supported.")
  end

  function PinMap:finalize(f)
    if static.numInstances ~= 1 then
      U.error('There should be only one (implicit) instance of the PinMap block.')
    end
    local pinMap =  globals.pinmap:get()
    if Target.Variables.GeneratePinMap ~= 1 then
      local installDir = Coder.GetInstallDir()
      local fileName = "%s/%s_pinout.html" % {installDir, Target.Variables.BASE_NAME}
      local file, e = io.open(fileName, "w")
      if file == nil then
        U.error(e)
      end
      io.output(file)
      local htmlCode
      if Target.Variables.GeneratePinMap == 3 then -- nucleo-64 or nucleo-144
        htmlCode = generateNuceloPinviewHtmlCode(pinMap, globals)
        io.write(htmlCode)
      else
        htmlCode = generateGenericHtmlTable(pinMap)
        io.write(htmlCode)
      end
      file:close()
    end
  end

  return PinMap
end

return Module

