--[[
  Copyright (c) 2021 by Plexim GmbH
  All rights reserved.

  A free license is granted to anyone to use this software for any legal
  non safety-critical purpose, including commercial applications, provided
  that:
  1) IT IS NOT USED TO DIRECTLY OR INDIRECTLY COMPETE WITH PLEXIM, and
  2) THIS COPYRIGHT NOTICE IS PRESERVED in its entirety.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
  OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
  AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
  OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
  SOFTWARE.
--]]

local Module = {}
local U = require('common.utils')

local static = {}

local outputType = {
  "PUSHPULL",
  "OPENDRAIN"
}

function Module.getBlock(globals, cpu)

  local Powerstage = require('common.block').getBlock(globals, cpu)
  if static[cpu] == nil then
    static[cpu] = {
      numInstances = 0,
      numChannels = 0,
      instances = {},
      finalized = false, 
    }
  end
  Powerstage["instance"] = static[cpu].numInstances
  static[cpu].numInstances = static[cpu].numInstances + 1
  
  function Powerstage:getEnableCode()
    -- TODO: maybe we need to issue this statement once for each task?
    if self.enable_code_generated == nil then
      -- only do this once
      self.enable_code_generated = true
      return 'PLXHAL_PWR_syncdPwmEnable();'
    end
  end

  function Powerstage:p_getDirectFeedthroughCode()
    local Require = ResourceList:new()
    local InitCode = U.CodeLines:new()
    local OutputSignal = StringList:new()
    local OutputCode = U.CodeLines:new()
    
    -- there can only be one powerstage protection block per CPU
    if self.cpu == 0 then
      Require:add("Powerstage Control")
    elseif self.cpu == 1 then
      Require:add("Powerstage Control CPU2")
    end
    
    table.insert(static[self.cpu].instances, self.bid)

    self.analogFltLineObj = self:getGlobalBlockInstance('analog_fault_line')

    if self.analogFltLineObj == nil then
      error('TSP exception: Analog fault line object not found.')
    end
      
    local ts = Block.Task["SampleTime"]
    if ts[1] == 0 then
      U.error("Invalid sample time.")
    end
    self['sample_time'] = ts[1]
    self['task_name'] = Block.Task["Name"]
    
    if Block.Mask.EnableSignal == 1 then
      self['enable_port'] = string.char(65+Block.Mask.EnablePort-1)
      self['enable_pin'] = Block.Mask.EnablePin
      self['enable_pol'] = (Block.Mask.EnablePolarity == 2)
      self['outputType'] = outputType[Block.Mask.OutputType]
      Require:add('P%s'% {self['enable_port']}, self['enable_pin'])
      globals.syscfg:addEntry('powerstage', {
        port = self['enable_port'],
        pin = self['enable_pin'],
        outputType = self['outputType'],
        path = self:getName()
      })
    end

    -- analog protection signal activation
    self.analog_protection_signals = {}
    local s = 1
    while Block.Mask['AnalogFltSig%dReact' % {s}] ~= nil do
      if Block.Mask['AnalogFltSig%dReact' % {s}] == 2 then
        if not self.analogFltLineObj:isAnalogProtectionSignalConfigured(s) then
          U.error('Please enable and configure "analog protection signal %d" under "Coder Options -> Target -> Protections".' % {s})
        end
        local config = self.analogFltLineObj:getAnalogProtectionSignalConfig(s)
        table.insert(self.analog_protection_signals, {
          reaction = 'one_shot',
          comp_unit = config.comp_unit,
          threshold = config.threshold,
          polarity = config.polarity,
          analog_fault_signal = config.analog_fault_signal,
          signal = s,
        })
      end
      s = s + 1
    end
 
    OutputCode:append("{")
    OutputCode:append("  if((%s) > 0){" % {Block.InputSignal[1][1]})
    OutputCode:append("   PLXHAL_PWR_setEnableRequest(true);") 
    OutputCode:append("  } else {")
    OutputCode:append("   PLXHAL_PWR_setEnableRequest(false);") 
    OutputCode:append("  }")  
    OutputCode:append("}")
    
    OutputSignal:append("PLXHAL_PWR_isEnabled()")
 
    return {
      InitCode = InitCode,
      OutputCode = OutputCode,
      OutputSignal = { OutputSignal },
      Require = Require,
      UserData  = {bid = Powerstage:getId()}
    }  
  end

  function Powerstage:getAnalogProtectionSignals()
    return self.analog_protection_signals
  end
  
  function Powerstage:finalizeThis(c)
    local ps_rate = math.floor(1/self['sample_time']+0.5)
    
    c.PreInitCode:append("{")
    c.PreInitCode:append('PLX_PWR_sinit();')
    if self['enable_port'] ~= nil then
      c.PreInitCode:append('static PLX_DIO_Obj_t doutObj;')
      c.PreInitCode:append('PLX_DIO_Handle_t doutHandle = PLX_DIO_init(&doutObj, sizeof(doutObj));')
      
      c.PreInitCode:append("PLX_DIO_configureOut(doutHandle, PLX_PORT%s, %i, %s);" % {self.enable_port, self.enable_pin, tostring(self.enable_pol)})
      c.PreInitCode:append("PLX_PWR_configure(doutHandle, %i);" % {ps_rate}) 
    else
      c.PreInitCode:append("PLX_PWR_configure(0, %i);" % {ps_rate}) 
    end   
    c.PreInitCode:append("}") 
  end

  function Powerstage:finalize(c)
    if static[self.cpu].finalized then
      return
    end
    
    c.Include:append('plx_power.h')
        
    c.Declarations:append('void PLXHAL_PWR_setEnableRequest(bool aEnable){');
    c.Declarations:append('PLX_PWR_setEnableRequest(aEnable);');
    c.Declarations:append('PLX_PWR_runFsm();');
    c.Declarations:append('}'); 
    
    c.Declarations:append('bool PLXHAL_PWR_isEnabled(){');
    c.Declarations:append('  return PLX_PWR_isEnabled();');
    c.Declarations:append('}');   
     
    c.Declarations:append('void PLXHAL_PWR_syncdPwmEnable(){');
    c.Declarations:append('  PLX_PWR_syncdSwitchingEnable();');
    c.Declarations:append('}');  
    
    for _, bid in pairs(static[self.cpu].instances) do
        local powerstage = globals.instances[bid]
        if powerstage:getCpu() == self.cpu then
          powerstage:finalizeThis(c)
        end
    end

    static[self.cpu].finalized = true   
  end
  
  return Powerstage
end

return Module
