--[[
  Copyright (c) 2021 by Plexim GmbH
  All rights reserved.

  A free license is granted to anyone to use this software for any legal
  non safety-critical purpose, including commercial applications, provided
  that:
  1) IT IS NOT USED TO DIRECTLY OR INDIRECTLY COMPETE WITH PLEXIM, and
  2) THIS COPYRIGHT NOTICE IS PRESERVED in its entirety.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
  OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
  AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
  OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
  SOFTWARE.
--]]

local Module = {}
local U = require('common.utils')

local static = {}

function Module.getBlock(globals, cpu)

  local Spi = require('common.block').getBlock(globals, cpu)
  if static[cpu] == nil then
    static[cpu] = {
      numInstances = 0,
      instances = {},
      finalized = false,
    }
  end
  Spi["instance"] = static[cpu].numInstances
  static[cpu].numInstances = static[cpu].numInstances + 1

  function Spi:checkMaskParameters()
  end

  function Spi:createImplicit(spi, params, req)
    self.spi = math.floor(spi) -- cast float to int
    static[self.cpu].instances[self.spi] = self.bid

    req:add('SPI%s' % {self.spi})

    self:logLine('SPI %s implicitly created.' % {self.spi})

    self.config = {
      dataWidth = params.charlen,
      polarity = params.pol,
      phase = params.phase,
      baudrate = params.baudrate,
      isMaster = (params.baudrate > 0),
      sckport = params.sckport,
      sckpin = params.sckpin,
      misoport = params.misoport,
      misopin = params.misopin,
      mosiport = params.mosiport,
      mosipin = params.mosipin,
      csport = params.csport,
      cspin = params.cspin,
      apb = params.apb
    }

    self.config.sckgpio_af = globals.target.getAlternateFunctionOrError({
      func = 'SPI%d_SCK' % self.spi,
      pad = '%s%d' % {self.config.sckport, self.config.sckpin},
      opt_errMsgPrefix = '%s%d not available as SCK pin for SPI %d.'
         % {self.config.sckport, self.config.sckpin, self.spi}
    })

    self.config.misogpio_af = globals.target.getAlternateFunctionOrError({
      func = 'SPI%d_MISO' % self.spi,
      pad = '%s%d' % {self.config.misoport, self.config.misopin},
      opt_errMsgPrefix = 'P%s%d not available as MISO pin for SPI %d.'
         % {self.config.misoport, self.config.misopin, self.spi}
    })

    self.config.mosigpio_af = globals.target.getAlternateFunctionOrError({
      func = 'SPI%d_MOSI' % self.spi,
      pad = '%s%d' % {self.config.mosiport, self.config.mosipin},
      opt_errMsgPrefix = 'P%s%d not available as MOSI pin for SPI %d.'
         % {self.config.mosiport, self.config.mosipin, self.spi}
    })

    -- for slave also initialize slave select signal
    if not self.config.isMaster then
      self.config.csgpio_af = globals.target.getAlternateFunctionOrError({
        func = 'SPI%d_NSS' % self.spi,
        pad = '%s%d' % {self.config.csport, self.config.cspin},
        opt_errMsgPrefix = 'P%s%d not available as NSS pin for SPI %d.'
           % {self.config.csport, self.config.cspin, self.spi}
      })
    end

    -- for master, also check if clock is within allowable range
    if self.config['isMaster'] then
      self.config['prescaler'] = globals.target.getSpiClockPrescaler(self.spi, self.config['baudrate'])
      globals.target.checkSpiClockIsAchievable(self.spi, {
        baudrate = self.config['baudrate'], 
        prescaler = self.config['prescaler']
      })

      local spiClock = globals.target.getActualSpiClockFrequency(self.spi, {
        baudrate = self.config['baudrate'], 
        prescaler = self.config['prescaler']
      })
      local relError = math.abs(spiClock/self.config['baudrate']-1)*100
      if relError > 10 then
            local msg = [[

            Unable to accurately achieve the desired SPI clock:
            - desired value: %.0f Hz
            - closest achievable value: %.0f Hz

            The SPI master will be configured to the closest achievable value. 
            ]] % {self.config['baudrate'], spiClock}
        U.warning(msg)
      end
    else
      self.config['prescaler'] = 2
    end
    req:add('P%s' % {self.config['sckport']}, self.config['sckpin'])
    req:add('P%s' % {self.config['misoport']}, self.config['misopin'])
    req:add('P%s' % {self.config['mosiport']}, self.config['mosipin'])
    if not self.config['isMaster'] then
      req:add('P%s' % {self.config['csport']}, self.config['cspin'])
    end
    local pinconf = {}
    table.insert(pinconf,{
      port = self.config['sckport'],
      pin = self.config['sckpin'],
      af = self.config['sckgpio_af']
    })
    table.insert(pinconf,{
      port = self.config['misoport'],
      pin = self.config['misopin'],
      af = self.config['misogpio_af']
    })
    table.insert(pinconf,{
      port = self.config['mosiport'],
      pin = self.config['mosipin'],
      af = self.config['mosigpio_af']
    })
    if not self.config['isMaster'] then
      table.insert(pinconf,{
        port = self.config['csport'],
        pin = self.config['cspin'],
        af = self.config['csgpio_af']
      })
    end
    globals.syscfg:addEntry('spi', {
      unit = self.spi,
      pins = pinconf,
      path = params.path
    })
  end

  function Spi:p_getDirectFeedthroughCode()
    U.error("Explicit use of SPI via target block not supported.")
  end

  function Spi:p_getNonDirectFeedthroughCode()
    U.error("Explicit use of SPI via target block not supported.")
  end

  function Spi:finalizeThis(c)
    
    local comment
    if (self.config['isMaster']) then
      comment = 'master'
    else
      comment = 'slave'
    end
    c.PreInitCode:append('// configure SPI%s as an SPI %s' % {self.spi, comment})
    c.PreInitCode:append('{\n')
    c.PreInitCode:append('PLX_SPI_setup(SpiHandles[%i], PLX_SPI_SPI%i, %i);\n' % {self.instance, self.spi, self.config['dataWidth']})
    c.PreInitCode:append(self.globals.target.getSpiConfigurationCode(self.spi, self.config))
    c.PreInitCode:append("}")
    c.PostInitCode:append('// Start SPI%s' % {self.spi})
    c.PostInitCode:append('PLX_SPI_start(SpiHandles[%i]);\n' % {self.instance})
  end

  function Spi:finalize(c)
    if static[self.cpu].finalized then
      return
    end

    c.Include:append('plx_spi.h')
    c.Declarations:append('PLX_SPI_Handle_t SpiHandles[%i];' %
                              {static[self.cpu].numInstances})
    c.Declarations:append('PLX_SPI_Obj_t SpiObj[%i];' % {static[self.cpu].numInstances})
    c.Declarations:append(
        'uint16_t PLXHAL_SPI_isRxFifoEmpty(uint16_t aHandle){')
    c.Declarations:append(
        '  return PLX_SPI_isRxFifoEmpty(SpiHandles[aHandle]);')
    c.Declarations:append('}')

    c.Declarations:append(
        'bool PLXHAL_SPI_putWords(uint16_t aHandle, uint16_t *aData, uint16_t aLen){')
    c.Declarations:append(
        '  return PLX_SPI_putWords(SpiHandles[aHandle], aData, aLen);')
    c.Declarations:append('}')

    c.Declarations:append(
        'bool PLXHAL_SPI_getWords(uint16_t aHandle, uint16_t *aData, uint16_t aLen){')
    c.Declarations:append(
        '  return PLX_SPI_getWords(SpiHandles[aHandle], aData, aLen);')
    c.Declarations:append('}')

    c.Declarations:append(
        'bool PLXHAL_SPI_getAndResetRxOverrunFlag(uint16_t aHandle){')
    c.Declarations:append(
        '  return PLX_SPI_getAndResetRxOverrunFlag(SpiHandles[aHandle]);')
    c.Declarations:append('}')

    c.Declarations:append(
        'bool PLXHAL_SPI_isTxBusy(uint16_t aHandle){')
    c.Declarations:append(
        '  return PLX_SPI_isTxBusy(SpiHandles[aHandle]);')
    c.Declarations:append('}')


    local code = [[
      for (int i = 0; i < %d; i++) {
        SpiHandles[i] = PLX_SPI_init(&SpiObj[i], sizeof(SpiObj[i]));
      }
    ]]
    c.PreInitCode:append(code % {static[self.cpu].numInstances})

    for _, bid in pairs(static[self.cpu].instances) do
      local spi = globals.instances[bid]
      if spi:getCpu() == self.cpu then
        spi:finalizeThis(c)
      end
    end

    static[self.cpu].finalized = true
  end

  return Spi
end

return Module