--[[
  Copyright (c) 2021 by Plexim GmbH
  All rights reserved.

  A free license is granted to anyone to use this software for any legal
  non safety-critical purpose, including commercial applications, provided
  that:
  1) IT IS NOT USED TO DIRECTLY OR INDIRECTLY COMPETE WITH PLEXIM, and
  2) THIS COPYRIGHT NOTICE IS PRESERVED in its entirety.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
  OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
  AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
  OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
  SOFTWARE.
--]]

local Module = {}
local U = require('common.utils')

local static = {}

function Module.getBlock(globals, cpu)

  local SpiSlave = require('common.block').getBlock(globals, cpu)
  if static[cpu] == nil then
    static[cpu] = {
      numInstances = 0,
      instances = {},
      finalized = false,
    }
  end
  SpiSlave["instance"] = static[cpu].numInstances
  static[cpu].numInstances = static[cpu].numInstances + 1

  function SpiSlave:checkMaskParameters()
    if Block.Mask.CharLength > 16 or Block.Mask.CharLength < 4 or not U.isPositiveIntScalar(Block.Mask.CharLength) then
      U.error("The number of 'Bits per word' has to be an integer between 4 and 16")
    end

    if #Block.Mask.SckPin ~= 1 then
      U.error("Invalid SCK Pin configuration.")
    end

    if #Block.Mask.MisoPin ~= 1 then
      U.error("Invalid MISO Pin configuration.")
    end

    if #Block.Mask.MosiPin ~= 1 then
      U.error("Invalid MOSI Pin configuration.")
    end

    if #Block.Mask.CsPin ~= 1 then
      U.error("Invalid CS Pin configuration.")
    end
  end

  function SpiSlave:p_getDirectFeedthroughCode()
    local Require = ResourceList:new()
    local OutputCode = U.CodeLines:new()

    table.insert(static[self.cpu].instances, self.bid)

    self.spi = Block.Mask.Spi
    local mode = Block.Mask.Mode - 1

    local sckgport = string.char(65+Block.Mask.SckPort - 1)
    local sckpin = Block.Mask.SckPin
    local misoport = string.char(65+Block.Mask.MisoPort - 1)
    local misopin = Block.Mask.MisoPin
    local mosiport = string.char(65+Block.Mask.MosiPort - 1)
    local mosipin = Block.Mask.MosiPin
    local csport = string.char(65+Block.Mask.CsPort - 1)
    local cspin = Block.Mask.CsPin

    local apb = globals.target.getTargetParameters()['spis']['SPI%d' % {self.spi}]['apb']

    self.spi_obj = self:makeBlock('spi', self.cpu)
    self.spi_obj:createImplicit(self.spi, {
      path = self:getName(),
      charlen = Block.Mask.CharLength,
      pol = (mode >= 2),
      phase = (mode == 1) or (mode == 3),
      baudrate = 0,     -- slave gets its clock signal from external
      sckport = sckgport,
      sckpin = sckpin,
      misoport = misoport,
      misopin = misopin,
      mosiport = mosiport,
      mosipin = mosipin,
      csport = csport,
      cspin = cspin,
      apb = apb
    }, Require)

    self.spi_instance = self.spi_obj:getObjIndex()

    local dim = Block.Mask.Dim
    local spi_fifo_depth =
        globals.target.getTargetParameters()['spis']['SPI%d' % {self.spi}]['fifo_depth']
    if (Block.Mask.CharLength <= 8) then
      spi_fifo_depth = spi_fifo_depth * 2 - 1 -- fifo can only hold 3 data frames
    end
    if dim > spi_fifo_depth then
      U.error("Maximum number of words per transmission for this target equals %i." %
              {spi_fifo_depth})
    end

    local outSignal = Block:OutputSignal()
    OutputCode:append('{')
    -- setup I/O buffers
    OutputCode:append("static uint16_t SpiSlaveRxBuffer[%i] = {" % {dim})
    for i = 1, dim do
      if i > 1 then
        OutputCode:append(", 0")
      else
        OutputCode:append("0")
      end
    end
    OutputCode:append("};\n")
    OutputCode:append("static uint16_t SpiSlaveTxBuffer[%i];\n" % {dim})
    OutputCode:append("static bool SpiSlaveDataReady = false;\n")
    OutputCode:append("{\n")
    for i = 1, dim do
      OutputCode:append("    SpiSlaveTxBuffer[%i] = %s;\n" % {i - 1, Block.InputSignal[1][i]})
    end

    -- service task
    OutputCode:append("\n")
    OutputCode:append("SpiSlaveDataReady =  PLXHAL_SPI_getWords(%i, &SpiSlaveRxBuffer[0], %i);\n" % {self.spi_instance, dim})
    OutputCode:append("if (SpiSlaveDataReady)\n")
    OutputCode:append("  {\n")
    OutputCode:append(
        "      PLXHAL_SPI_putWords(%i, &SpiSlaveTxBuffer[0], %i);\n" % {self.spi_instance, dim})
    OutputCode:append(" }\n")
    OutputCode:append("}\n")

    -- output signals
    local readLookUp = {0, 1, 2}
    if (dim == spi_fifo_depth) and (Block.Mask.CharLength <= 8) then
      readLookUp = {1, 2, 0}
    end
    for i = 1, dim do
      OutputCode:append('%s = SpiSlaveRxBuffer[%i];' % {outSignal[1][i], readLookUp[i]})
    end
    OutputCode:append('%s = SpiSlaveDataReady;' % {outSignal[2][1]})
    OutputCode:append('%s = PLXHAL_SPI_getAndResetRxOverrunFlag(%i); \n' % {outSignal[3][1], self.spi_instance}) 
    OutputCode:append('}')

    return {
      OutputCode = OutputCode,
      Require = Require,
      UserData = {bid = self:getId()}
    }
  end

  function SpiSlave:p_getNonDirectFeedthroughCode()
    return {}
  end

  function SpiSlave:finalizeThis(c)
  end

  function SpiSlave:finalize(c)
    if static[self.cpu].finalized then
      return
    end

    for _, bid in pairs(static[self.cpu].instances) do
      local spislave = globals.instances[bid]
      if spislave:getCpu() == self.cpu then
        spislave:finalizeThis(c)
      end
    end

    static[self.cpu].finalized = true
  end

  return SpiSlave
end

return Module
