/*
   Copyright (c) 2021 by Plexim GmbH
   All rights reserved.

   A free license is granted to anyone to use this software for any legal
   non safety-critical purpose, including commercial applications, provided
   that:
   1) IT IS NOT USED TO DIRECTLY OR INDIRECTLY COMPETE WITH PLEXIM, and
   2) THIS COPYRIGHT NOTICE IS PRESERVED in its entirety.

   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
   OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
   SOFTWARE.
 */

#ifndef PLX_ADC_IMPL_H_
#define PLX_ADC_IMPL_H_

#include "stm32f3xx_ll_gpio.h"
#include "stm32f3xx_ll_dma.h"
#include "stm32f3xx_ll_adc.h"

/* Delay between ADC end of calibration and ADC enable.                     */
/* Delay estimation in CPU cycles: Case of ADC enable done                  */
/* immediately after ADC calibration, ADC clock setting slow                */
/* (LL_ADC_CLOCK_ASYNC_DIV32). Use a higher delay if ratio                  */
/* (CPU clock / ADC clock) is above 32.                                     */
#define PLX_ADC_DELAY_CALIB_ENABLE_CPU_CYCLES  (LL_ADC_DELAY_CALIB_ENABLE_ADC_CYCLES * 32)

typedef enum PLX_ADC_UNIT {
   PLX_ADC1 = 0,
   PLX_ADC2,
   PLX_ADC3,
   PLX_ADC4
} PLX_ADC_Unit_t;

typedef struct PLX_ADC_OBJ
{
   ADC_TypeDef *adcInstance;
   uint16_t numRegularConversions;
   uint16_t oversamplingRatio;
   uint16_t numRegularChannels;
   uint16_t numInjectedConversions;
   uint16_t injResults[4];
   float injScale[4];
   float injOffset[4];
   uint16_t* results;
   float* scale;
   float* offset;
   volatile uint16_t doubleBufferOffset;
} PLX_ADC_Obj_t;

typedef PLX_ADC_Obj_t *PLX_ADC_Handle_t;

__STATIC_INLINE void PLX_ADC_updateInjected(PLX_ADC_Handle_t aHandle)
{
	PLX_ADC_Obj_t *obj = (PLX_ADC_Obj_t *)aHandle;

   obj->injResults[0] = LL_ADC_INJ_ReadConversionData12(obj->adcInstance, LL_ADC_INJ_RANK_1);
   obj->injResults[1] = LL_ADC_INJ_ReadConversionData12(obj->adcInstance, LL_ADC_INJ_RANK_2);
   obj->injResults[2] = LL_ADC_INJ_ReadConversionData12(obj->adcInstance, LL_ADC_INJ_RANK_3);
   obj->injResults[3] = LL_ADC_INJ_ReadConversionData12(obj->adcInstance, LL_ADC_INJ_RANK_4);
}

__STATIC_INLINE float PLX_ADC_getIn(PLX_ADC_Handle_t aHandle, uint16_t aChannel, bool aIsInjChannel)
{
   PLX_ADC_Obj_t *obj = (PLX_ADC_Obj_t *)aHandle;
   if (aIsInjChannel)
   {
      return ((float)obj->injResults[aChannel] * obj->injScale[aChannel] + obj->injOffset[aChannel]);
   }
   else
   {
      return ((float)obj->results[aChannel + obj->doubleBufferOffset ] * obj->scale[aChannel] + obj->offset[aChannel]);
   }
}

__STATIC_INLINE float PLX_ADC_getMean(PLX_ADC_Handle_t aHandle, uint8_t aOffset)
{
   PLX_ADC_Obj_t *obj = (PLX_ADC_Obj_t *)aHandle;

   uint32_t processedValue = 0;
   for (int i = 0; i < obj->oversamplingRatio; i++) {
      processedValue += obj->results[(i * obj->numRegularChannels + aOffset) + obj->doubleBufferOffset];
   }
   processedValue = processedValue / obj->oversamplingRatio;

   return ((float)processedValue * obj->scale[aOffset] + obj->offset[aOffset]);
}

__STATIC_INLINE float PLX_ADC_getMax(PLX_ADC_Handle_t aHandle, uint8_t aOffset)
{
   PLX_ADC_Obj_t *obj = (PLX_ADC_Obj_t *)aHandle;

   uint16_t processedValue = 0;
   for (int i = 0; i < obj->oversamplingRatio; i++) {
      uint16_t result = obj->results[(i * obj->numRegularChannels + aOffset) + obj->doubleBufferOffset];
      if (result > processedValue)
      {
         processedValue = result;
      }
   }
   return ((float)processedValue * obj->scale[aOffset] + obj->offset[aOffset]);
}

__STATIC_INLINE float PLX_ADC_getMin(PLX_ADC_Handle_t aHandle, uint8_t aOffset)
{
   PLX_ADC_Obj_t *obj = (PLX_ADC_Obj_t *)aHandle;

   uint16_t processedValue = 4096;
   for (int i = 0; i < obj->oversamplingRatio; i++) {
      uint16_t result = obj->results[(i * obj->numRegularChannels + aOffset) + obj->doubleBufferOffset];
      if (result < processedValue)
      {
         processedValue = result;
      }
   }

   return ((float)processedValue * obj->scale[aOffset] + obj->offset[aOffset]);
}


#endif /* PLX_ADC_IMPL_H_ */
