/*
   Copyright (c) 2021 by Plexim GmbH
   All rights reserved.

   A free license is granted to anyone to use this software for any legal
   non safety-critical purpose, including commercial applications, provided
   that:
   1) IT IS NOT USED TO DIRECTLY OR INDIRECTLY COMPETE WITH PLEXIM, and
   2) THIS COPYRIGHT NOTICE IS PRESERVED in its entirety.

   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
   OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
   SOFTWARE.
 */

#ifndef PLX_CANBUS_IMPL_H_
#define PLX_CANBUS_IMPL_H_

#include "stm32f3xx_hal_can.h"
#include <string.h>

typedef enum PLX_CANBUS_UNIT {
   PLX_CAN_CAN1 = 0
} PLX_CANBUS_Unit_t;

typedef enum PLX_CANBUS_RxFIFO {
   PLX_CAN_RX_FIFO_0 = 0,
   PLX_CAN_RX_FIFO_1
} PLX_CANBUS_RxFifo_t;

typedef enum PLX_CAN_PROTOCOL {
   PLX_CAN_2_0 = 0,
   PLX_CAN_FD
} PLX_CANBUS_Protocol_t;

typedef struct PLX_CANBUS_OBJ
{
   PLX_CANBUS_Unit_t    unit;
   CAN_HandleTypeDef    canHandle;
   uint8_t              lastErrorCode;
} PLX_CANBUS_Obj_t;

typedef PLX_CANBUS_Obj_t *PLX_CANBUS_Handle_t;

typedef struct PLX_CANBUS_TxHEADER_OBJ
{
  CAN_TxHeaderTypeDef header;
} PLX_CANBUS_TxHeader_Obj_t;

typedef PLX_CANBUS_TxHeader_Obj_t *PLX_CANBUS_TxHeader_t;

typedef struct PLX_CAN_MESSAGE_OBJ
{
   uint32_t    id;
   uint8_t     data[8];
   uint8_t     length;
   uint8_t     valid;
   uint8_t     overwritten;
} PLX_CAN_Message_Obj_t;

typedef struct PLX_CAN_TXMESSAGE_OBJ
{
   uint8_t     data[8];
   uint16_t    txHandleIndex;
}
PLX_CAN_TxMessage_Obj_t;

typedef struct PLX_CAN_LOOKUPTABLEENTRY_OBJ
{
   uint32_t canID;
   uint8_t index;
} PLX_CAN_LookupTableEntry_Obj_t;

typedef union {
   PLX_CAN_TxMessage_Obj_t canMsg;
} PLX_CAN_Message_Union;

typedef struct PLX_CAN_MSG_QUEUE
{
   volatile uint16_t head;
   volatile uint16_t tail;
   volatile uint16_t length;
   uint16_t bufferSize;
   PLX_CANBUS_Protocol_t protocol;
   PLX_CAN_Message_Union* msgBuffer;
} PLX_CAN_MSG_QUEUE_t;

typedef PLX_CAN_MSG_QUEUE_t* PLX_CAN_MSG_QUEUE_Handle_t;

typedef PLX_CANBUS_TxHeader_Obj_t *PLX_CANBUS_TxHeader_t;

inline static uint32_t PLX_CAN_dlcToBytes(uint32_t aDlc) {
    switch(aDlc) {
        case 0x00000000U: return 0;
        case 0x00010000U: return 1;
        case 0x00020000U: return 2;
        case 0x00030000U: return 3;
        case 0x00040000U: return 4;
        case 0x00050000U: return 5;
        case 0x00060000U: return 6;
        case 0x00070000U: return 7;
        case 0x00080000U: return 8;
        default: return 0; // Return 0 for invalid DLC values
    }
}

__STATIC_INLINE bool PLX_CANBUS_msgQueueIsEmpty(PLX_CAN_MSG_QUEUE_Handle_t aQueue)
{  
   PLX_CAN_MSG_QUEUE_t* obj = (PLX_CAN_MSG_QUEUE_t*)aQueue;
   return (obj->length == 0);
}

__STATIC_INLINE bool PLX_CANBUS_enqueMsg(PLX_CAN_MSG_QUEUE_Handle_t aQueue, PLX_CAN_TxMessage_Obj_t* aMsg)
{
   PLX_CAN_MSG_QUEUE_t* obj = (PLX_CAN_MSG_QUEUE_t*)aQueue;
   if (obj->length < obj->bufferSize)
   {
      uint32_t interruptState;
      interruptState = enterCriticalSection();
      if (obj->tail == obj->bufferSize)
      {
         obj->tail = 0;
      }
      obj->msgBuffer[obj->tail].canMsg = *(PLX_CAN_TxMessage_Obj_t*)aMsg;
      obj->tail++;
      obj->length++;
      exitCriticalSection(interruptState);
      return true;
   }
   else
   {
      return false;
   }
}

__STATIC_INLINE bool PLX_CANBUS_dequeueMsg(PLX_CAN_MSG_QUEUE_Handle_t aQueue, PLX_CAN_TxMessage_Obj_t* aMsg)
{
   PLX_CAN_MSG_QUEUE_t* obj = (PLX_CAN_MSG_QUEUE_t*)aQueue;
   if (obj->length > 0)
   {
      uint32_t interruptState;
      interruptState = enterCriticalSection();
      *(PLX_CAN_TxMessage_Obj_t*)aMsg = obj->msgBuffer[obj->head].canMsg;
      memset(&obj->msgBuffer[obj->head].canMsg, 0, sizeof(PLX_CAN_TxMessage_Obj_t));
      obj->head++;
      if (obj->head == obj->bufferSize)
      {
         obj->head = 0;
      }
      obj->length--;
      exitCriticalSection(interruptState);
      return true;
   }
   else
   {
      return false;
   }
}

#endif /* PLX_CANBUS_IMPL_H_ */
