--[[
  Copyright (c) 2021 by Plexim GmbH
  All rights reserved.

  A free license is granted to anyone to use this software for any legal
  non safety-critical purpose, including commercial applications, provided
  that:
  1) IT IS NOT USED TO DIRECTLY OR INDIRECTLY COMPETE WITH PLEXIM, and
  2) THIS COPYRIGHT NOTICE IS PRESERVED in its entirety.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
  OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
  AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
  OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
  SOFTWARE.
--]]

local debug = require('debug')
local HAL = require('common.coderGenHalCode')
local T = require(Target.Variables.TargetName)
local U = require('common.utils')
local VersionNumber = require('common.version')
local ExtTools = require('common.external_tools')

local Coder = {}

local Coder = require('common.coder_common')
local Registry = Coder.Registry

local FactoryBlock = Coder.CreateFactoryBlock(T)

function Coder.getTargetFamilyForOldCreateTargetBlockCheck()
  return 'STM32'
end

function Coder.GetInstallDir()
  if Target.Variables.BuildType == 1 then
    return U.enforceValidCoderOptionsDirectory(
      'InstallDir',
      U.getDisallowedPathCharsForInstallDir()
    )
  else
    return U.normalizePath(Target.Variables.BUILD_ROOT)
  end
end

function Coder.GetInstructionOverrides()
  return require('InstructionOverrides')
end

function Coder.p_Initialize()
  local ci_modelHooksRecipe = {
    -- This table is populated in Coder.Initialize and holds all values
    -- returned by this function.
    Resources = ResourceList:new(),
    Require = ResourceList:new(),
    Declarations = U.CodeLines:new(),
  }

  -- do this before using any T methods
  T.configure(ci_modelHooksRecipe.Resources)
  Target:SetVariantName(T.getFullChipName())

  -- (implicit) clock module - MUST COME FIRST!
  local clockBlock = FactoryBlock:makeBlock('clock', 0)

  -- (implicit) system configuration
  local sysCfg = FactoryBlock:makeBlock('syscfg', 0)

  -- (implicit) pin map
  local pinMap = FactoryBlock:makeBlock('pinmap', 0)

  -- (implicit) external mode module
  if type(Target.Variables.EXTERNAL_MODE) == 'number' then
    local extModeBlock = FactoryBlock:makeBlock('extmode', 0)
    extModeBlock:createImplicit(ci_modelHooksRecipe.Require)
  end

  Coder.setupPILSimulation(ci_modelHooksRecipe)
  
  -- (implicit) analog protection management
  local flt = FactoryBlock:makeBlock('analog_fault_line', 0)
  flt:createImplicit(ci_modelHooksRecipe.Require, ci_modelHooksRecipe.Resources)

  return ci_modelHooksRecipe
end

function Coder.p_Finalize()
  -- This table is returned by this function and used by plecs to generate
  -- the <model_name>.c file. Note code added here must be terminated by newlines
  local cf_modelHooksRecipe = {
    HeaderDeclarations = U.CodeLines:new(),
    Include = StringList:new(),
    Require = ResourceList:new(),
    Declarations = U.CodeLines:new(),
    PreInitCode = U.CodeLines:new(),
    PostInitCode = U.CodeLines:new(),
    TerminateCode = U.CodeLines:new(),
  }

  cf_modelHooksRecipe.Include:append('plx_hal.h')

  local serialNumber
  if Target.Variables.SpecifySerialNumber == 1 then
    serialNumber = tostring(Target.Variables.SerialNumber)
  end
  
  -- check jtag tool paths
  local progInterface
  if Target.Variables.BuildType == 3 then
    progInterface = Target.Variables.ProgIf
  end
  local debugInterface
  if Target.Variables.EXTERNAL_MODE == 1 then
    if Target.Variables.BuildType == 3 then
      debugInterface = Target.Variables.ProgIf
    else
      debugInterface = Target.Variables.DebugIf
    end
  end
  if (progInterface == 1) or (debugInterface == 1) then
    ExtTools.OpenOcd.assertOpenOcdFound()
  end
  if (progInterface == 2) or (debugInterface == 2) then
    ExtTools.JLink.assertJLinkFound()
  end

  if Target.Variables.FLOAT_TYPE == 'float' then
    cf_modelHooksRecipe.Declarations:append('#define sin sinf\n')
    cf_modelHooksRecipe.Declarations:append('#define cos cosf\n')
  end

  -- categorize block instances
  local blockInstancesByType = {}
  for _, b in ipairs(Registry.BlockInstances) do
    if blockInstancesByType[b:getType()] == nil then
      blockInstancesByType[b:getType()] = {}
    end
    table.insert(blockInstancesByType[b:getType()], b)
  end

  -- first see if an adequate base-task trigger timer is present
  local triggerOriginBlock
  local triggerOriginAbsFreqError
  if Target.Variables.TaskFreqTol == 1 then
    local achievableTs = T.getAchievableTimSampleTime(Target.Variables
      .SAMPLE_TIME)
    triggerOriginAbsFreqError = math.abs(1 / Target.Variables.SAMPLE_TIME -
      1 / achievableTs)
    if triggerOriginAbsFreqError ~= 0.0 then
      local msg = [[
      Unable to accurately meet the desired step size:
      - desired step size: %(ts_desired)e
      - closest achievable step size: %(ts_actual)e

      You may want to modify the "Step size tolerance" parameter or adjust the system clock frequency under Coder Options->Target->General.
      ]] % {
        ts_desired = Target.Variables.SAMPLE_TIME,
        ts_actual = achievableTs,
      }
      U.error(msg)
    end
  else
    triggerOriginAbsFreqError = (1 / Target.Variables.SAMPLE_TIME) *
       (Target.Variables.SampleTimeRelTol / 100)
  end

  for _, b in ipairs(Registry.BlockInstances) do
    if b:canBeImplicitTaskTriggerOriginBlock() then
      local ts = b:requestImplicitTrigger(Target.Variables.SAMPLE_TIME)
      local achievableTs = T.getAchievableTimSampleTime(Target.Variables
        .SAMPLE_TIME)
      -- if the sample time of a providing block is rounded to the closest achievable value, compare against the sample time that can be achieved.
      if (ts ~= nil) and (ts > 0) then
        local freqError = math.abs(1 / ts - 1 / achievableTs)
        if (freqError <= triggerOriginAbsFreqError) then
          -- new best fit
          triggerOriginBlock = b
          triggerOriginAbsFreqError = freqError
          U.log(
            'New best fit for default triggering block: bid=%d, Ts=%e, Ferror=%e\n' %
            {b:getId(), ts, freqError})
        end
      end
    end
  end

  if triggerOriginBlock == nil then
    -- no suitable block found - create invisible timer block
    U.log('- No suitable triggering block found. Creating implicit timer.\n')
    triggerOriginBlock = FactoryBlock:makeBlock('timer', 0)
    if blockInstancesByType[triggerOriginBlock:getType()] == nil then
      blockInstancesByType[triggerOriginBlock:getType()] = {}
    end
    table.insert(blockInstancesByType[triggerOriginBlock:getType()],
                 triggerOriginBlock)
    triggerOriginBlock:createImplicit({
      f = 1 / Target.Variables.SAMPLE_TIME, triggerOrigin = true
    })
  end

  -- provide implicit base trigger to blocks that might need it
  for _, b in ipairs(Registry.BlockInstances) do
    b:setImplicitTriggerSource(triggerOriginBlock:getId())
  end

  -- inform all trigger sources about the sinks attached to them
  for _, b in ipairs(Registry.BlockInstances) do
    b:setSinkForTriggerSource()
  end

  -- propagate all trigger connections
  for _, b in ipairs(Registry.BlockInstances) do
    b:propagateTriggerSampleTime()
  end

  -- make sure model has a task trigger, if not create implicit
  if blockInstancesByType['tasktrigger'] == nil then
    -- create an implicit trigger
    local taskTrigger = FactoryBlock:makeBlock('tasktrigger', 0)

    blockInstancesByType['tasktrigger'] = {}
    table.insert(blockInstancesByType['tasktrigger'], taskTrigger)

    -- connect to trigger block
    local blockForTaskTrigger
    -- connect to triggered ADC, if possible
    local triggeredAdcInstances = {}
    if blockInstancesByType['adc'] ~= nil then
      for _, adc in ipairs(blockInstancesByType['adc']) do
        -- only add triggered adc instances
        if adc:isTriggered() then
          table.insert(triggeredAdcInstances, adc)
        end
      end
    end

    if not U.arrayIsEmpty(triggeredAdcInstances) then
      local candidates = {}
      -- find ADCs at correct sample time
      for _, adc in ipairs(triggeredAdcInstances) do
        if adc:getTriggerSampleTime() == nil then
          U.error('ADC %i has undefined trigger time' % {adc:getId()})
        end
        U.log('- ADC %i detected at ts=%f.\n' %
          {adc:getId(), adc:getTriggerSampleTime()})
        if adc:getTriggerSampleTime() == Target.Variables.SAMPLE_TIME then
          table.insert(candidates, adc)
        end
      end
      if #candidates ~= 0 then
        -- find ADC that takes the longest to complete conversions
        local maxConversionTime = 0
        for _, adc in ipairs(candidates) do
          if maxConversionTime < adc:getTotalConversionClkCycles() then
            maxConversionTime = adc:getTotalConversionClkCycles()
            blockForTaskTrigger = adc
          end
        end
      end
    end

    if blockForTaskTrigger == nil then
      -- no suitable ADC found, use timer
      blockForTaskTrigger = triggerOriginBlock
    end

    taskTrigger:setImplicitTriggerSource(blockForTaskTrigger:getId())
    local sink = {
      type = 'modtrig',
      bid = taskTrigger:getId()
    }
    blockForTaskTrigger:setSinkForTriggerSource(sink)
    for _, b in ipairs(Registry.BlockInstances) do
      b:propagateTriggerSampleTime()
    end
  end

  -- final task trigger check
  if blockInstancesByType['tasktrigger'] == nil then
    U.error('Exception: Model does not have a task trigger.')
  else
    local ts = blockInstancesByType['tasktrigger'][1]:getTriggerSampleTime(
      'modtrig')
    if ts == nil then
      U.error('Exception: Task trigger does not have a defined sample time.')
    else
      local relTol = 1e-6
      if Target.Variables.TaskFreqTol == 2 then
        relTol = Target.Variables.SampleTimeRelTol / 100
      end

      local absTsError = math.abs(ts - Target.Variables.SAMPLE_TIME)
      local absTsTol = relTol * Target.Variables.SAMPLE_TIME

      if absTsError > absTsTol then
        U.error([[
        Unable to meet the allowable step size tolerance:
        - desired step size: %(ts_desired)e
        - closest achievable step size: %(ts_actual)e
        - relative error: %(rerror)0.1f %%

        You may want to modify the trigger chain or adjust the "Step size tolerance" parameter under Coder Options->Target->General.
        ]] % {
          ts_desired = Target.Variables.SAMPLE_TIME,
          ts_actual = ts,
          rerror = 100 * math.abs((ts - Target.Variables.SAMPLE_TIME) / ts)
        })
      end
    end
  end

  -- finalize all blocks

  -- This table will be populated with data/code to generate
  -- the <model_name>_hal.c (and cla) files. Note code added here is
  -- automatically terminated with newlines.
  local halCodeGenRecipe = {
    HeaderDeclarations = U.CodeLines:new(),
    Include = StringList:new(),
    Require = ResourceList:new(),
    Declarations = U.CodeLines:new(),
    PreInitCode = U.CodeLines:new(),
    PostInitCode = U.CodeLines:new(),
    TerminateCode = U.CodeLines:new(),

    InterruptEnableCode = U.CodeLines:new(),
    TimerSyncCode = U.CodeLines:new(),
    BackgroundTaskCodeBlocks = U.CodeLines:new(),
    PilHeaderDeclarations = U.CodeLines:new(),
  }

  halCodeGenRecipe.Include:append('plx_hal.h')
  halCodeGenRecipe.Include:append('FreeRTOSConfig.h')
  halCodeGenRecipe.Include:append('plx_dispatcher.h')

  -- generate pre-init code for external libraries
  for name, par in pairs(Registry.ExternalLibraries) do
    if par.PreInitCode ~= nil then
      halCodeGenRecipe.PreInitCode:append(
        '\n// initialization of %s' % {par.LibFileName})
      for _, v in ipairs(par.PreInitCode) do
        halCodeGenRecipe.PreInitCode:append(v)
      end
    end
  end

  -- first process analog protection signals
  if blockInstancesByType['analog_fault_line'] ~= nil then
    for _, flt in ipairs(blockInstancesByType['analog_fault_line']) do
      flt:processConfiguration(halCodeGenRecipe)
    end
  end

  for _, b in ipairs(Registry.BlockInstances) do
    -- skip syscfg and pinmap from beeing finalized. They need to finalized at the
    -- very end since other blocks require inputs and/or outputs in their finalize
    -- function.
    if  not b:blockMatches('syscfg')
    and not b:blockMatches('pinmap') then
      b:finalize(halCodeGenRecipe)
    end
  end
  -- Finalize syscfg before pinmap, as the pinmap gets filled in syscfg.
  if blockInstancesByType['syscfg'] ~= nil then
    for _, b in ipairs(blockInstancesByType['syscfg']) do
      b:finalize(halCodeGenRecipe)
    end
  end
  -- Finalize pinmap at the very end as then all inputs/outputs are fixed.
  if blockInstancesByType['pinmap'] ~= nil then
    for _, b in ipairs(blockInstancesByType['pinmap']) do
      b:finalize(halCodeGenRecipe)
    end
  end


  -- create PIL structure
  for _, v in ipairs(halCodeGenRecipe.PilHeaderDeclarations) do
    cf_modelHooksRecipe.HeaderDeclarations:append(v..'\n')
  end

  U.log('\nBlock coding complete.\n\n')
  U.log('Blocks in model: %s\n' % {dump(blockInstancesByType)})
  U.log('\n\n')
  U.log('Target settings: %s\n' % {dump(Target.Variables)})

  U.dumpLog()

  -- create sandboxed low-level code
  cf_modelHooksRecipe.Declarations:append(
    'extern void %s_initHal();\n' % {Target.Variables.BASE_NAME})
  cf_modelHooksRecipe.PreInitCode:append(
    '%s_initHal();\n' % {Target.Variables.BASE_NAME})
  HAL.generateHalCode(
    '%s/%s_hal.c' % {Target.Variables.BUILD_ROOT, Target.Variables.BASE_NAME},
    halCodeGenRecipe)

  -- version ID
  local tspVerDef = '#undef TSP_VER'
  local tspVersion = VersionNumber.new(Target.Version)
  if tspVersion then
    tspVerDef = '#define TSP_VER 0x%s' % {tspVersion:majorMinorAs4DigitString()}
  end

  -- process templates
  local templateDict = {}

  templateDict['|>TSP_VERSION<|'] = Target.Version
  templateDict['|>BASE_NAME<|'] = Target.Variables.BASE_NAME
  templateDict['|>TARGET_ROOT<|'] = Target.Variables.TARGET_ROOT
  templateDict['|>SHARED_MAKE_FUNCTIONS<|'] = ExtTools.SHARED_MAKE_FUNCTIONS
  templateDict['|>TSP_VER_DEF<|'] = tspVerDef
  templateDict['|>DATA_TYPE<|'] = Target.Variables.FLOAT_TYPE

  local familySrcDir =
     ExtTools.assertValidTARGET_ROOT()..'/gcc/'..T.getFamilyPrefix()

  local codegenDir = ''
  if Target.Variables.BuildType > 1 then
    --[[
        "One-click" build from PLECS, flash only if BuildType == 3 --> handled in BuildSteps
    --]]
    codegenDir = ExtTools.GccArm.assertGccArmNoneEabiFound()

    Coder.assertFLOAT_TYPEIsFloat()

    local progExePath = ''
    local progArgs = ''

    if Target.Variables.BuildType == 3 then
      local progDir = ExtTools.C2pGdb.assertProgrammerFound()
    
      progExePath = progDir..'/c2p-gdb'

      local debugServerCmd = ''
      if (progInterface == 1) then
        local path
        if Target.Variables.OpenOcdConfig == 2 then
          path = U.enforceCO(U.fileExists, 'OpenOcdScriptPath')
        end
        debugServerCmd = ''
           ..ExtTools.OpenOcd.getBaseCommand()
           ..' -s '..U.shellQuote(Target.Variables.TARGET_ROOT..'/templates/openocd')
           ..' '..T.getOpenOcdOptions(path, serialNumber)


      elseif (progInterface == 2) then
        debugServerCmd = '%(q_gdbServerExe)s %(options)s' % {
          q_gdbServerExe = U.shellQuote(ExtTools.JLink.getGdbServerExe()),
          options = T.getJLinkOptions(serialNumber),
        }
      end

      progArgs = ''
         ..'load '
         ..'--server-cmd '..U.shellQuote(debugServerCmd)
         ..' --server-start-delay 1000 --port 3333'
         ..' --nvic '..T.getNvicAddress(Target.Variables.Cpu)
    end

    -- configure user defined flags for compiler and linker
    local compilerFlags = ''
    local linkerFlags = ''

    for i, v in ipairs(Registry.CompilerFlags) do
      if i == 1 then
        compilerFlags = compilerFlags..v..' \\'
      else
        compilerFlags = compilerFlags..'\n'..v..' \\'
      end
    end

    for i, v in ipairs(Registry.LinkerFlags) do
      if i == 1 then
        linkerFlags = linkerFlags..v..' \\'
      else
        linkerFlags = linkerFlags..'\n'..v..' \\'
      end
    end

    -- add external libraries
    local i = 1
    for _, par in pairs(Registry.ExternalLibraries) do
      if par.IncludePath ~= nil then
        if compilerFlags == '' then
          compilerFlags = compilerFlags..'-I"%s" \\' % {par.IncludePath}
        else
          compilerFlags = compilerFlags..'\n'..'-I"%s" \\' % {par.IncludePath}
        end
      end
      if par.LibFilePath ~= nil then
        if linkerFlags == '' then
          linkerFlags = linkerFlags..'-L"%s" \\' % {par.LibFilePath}
        else
          linkerFlags = linkerFlags..'\n'..'-L"%s" \\' % {par.LibFilePath}
        end
      end
      if par.LibFileName ~= nil then
        if linkerFlags == '' then
          linkerFlags = linkerFlags..'-l%s \\' % {par.LibFileName}
        else
          linkerFlags = linkerFlags..'\n'..'-l%s \\' % {par.LibFileName}
        end
      end
      i = i + 1
    end

    templateDict['|>LFLAGS<|'] = linkerFlags
    templateDict['|>CFLAGS<|'] = compilerFlags


    templateDict['|>INSTALL_DIR<|'] = './'
    templateDict['|>BIN_DIR<|'] = './output_STM32%s' % {T.getFullChipName()}
    templateDict['|>CG_PATH<|'] = codegenDir
    templateDict['|>PROGRAMMER_EXE_PATH<|'] = progExePath
    templateDict['|>PROGRAMMER_ARGS<|'] = progArgs
    templateDict['|>SRC_ROOT<|'] = familySrcDir

    if T.targetMatches('h7') then
      templateDict['|>TARGET_CPU<|'] = T.getCpuName(Target.Variables.Cpu)
                   
    end

    local hse_value = ''
    if Target.Variables.UseIntOsc ~= 1 then
      hse_value = '-DHSE_VALUE=%sUL' % {tostring(math.floor(T.getCleanExtClkHz()))}
    end
    templateDict['|>DEFINE_HSE_VALUE<|'] = hse_value

    local task_scheduler = ''
    if (Target.Variables.TaskScheduler == 2) or (#Model.Tasks == 1) then
      task_scheduler = '-DPLX_BARE_METAL_SCHEDULER'
    end
    templateDict['|>DEFINE_BARE_METAL_SCHEDULER<|'] = task_scheduler

    U.copyTemplateFile(familySrcDir..'/templates/main.mk',
                       Target.Variables.BUILD_ROOT..
                       '/'..Target.Variables.BASE_NAME..'.mk', templateDict)
    U.copyTemplateFile(familySrcDir..'/../shrd/templates/main.c',
                       Target.Variables.BUILD_ROOT..
                       '/'..Target.Variables.BASE_NAME..'_main.c', templateDict)
    U.copyTemplateFile(
      familySrcDir..
      '/templates/options_'..T.getMemorySizeChipName():lower()..'.mk',
      Target.Variables.BUILD_ROOT..
      '/'..Target.Variables.BASE_NAME..'_options.mk', templateDict)

    if Target.Variables.GeneratePinMap == 3 then
      U.copyTemplateFile(
        Target.Variables.TARGET_ROOT..
        '/templates/pinmap_style%d.css' % {T.getNucleoBoardPinCount()},
        Target.Variables.BUILD_ROOT..'/pinmap_style.css', templateDict)
    end
    if Target.Variables.AutoprobeDevice == 1 then
      local serialNumberString = ''
      if Target.Variables.SpecifySerialNumber == 1 then
        serialNumberString = 'st-link serial '..
           tostring(Target.Variables.SerialNumber)
      end
      templateDict['|>SERIAL_NUMBER<|'] = serialNumberString
      U.copyTemplateFile(
        Target.Variables.TARGET_ROOT..'/templates/autoprobe_config.cfg',
        Target.Variables.BUILD_ROOT..'/autoprobe_config.cfg', templateDict)
    end
  else
    --[[
        Generate files into Eclipse project and build/debug from there
    --]]
 
    local installDir = Coder.GetInstallDir() -- final destination for generated files

    templateDict['|>INSTALL_DIR<|'] = installDir
    templateDict['|>BUILD_ROOT<|'] = Target.Variables.BUILD_ROOT

    local hse_value = ''
    if Target.Variables.UseIntOsc ~= 1 then
      hse_value = '#define HSE_VALUE %sUL' % {
        tostring(math.floor(T.getCleanExtClkHz()))}
    end
    templateDict['|>DEFINE_HSE_VALUE<|'] = hse_value
    local task_scheduler = ''
    if (Target.Variables.TaskScheduler == 2) or (#Model.Tasks == 1) then
      task_scheduler = '#define PLX_BARE_METAL_SCHEDULER'
    end
    templateDict['|>DEFINE_BARE_METAL_SCHEDULER<|'] = task_scheduler

    U.copyTemplateFile(Target.Variables.TARGET_ROOT..'/templates/install.mk',
                       Target.Variables.BUILD_ROOT..
                       '/'..Target.Variables.BASE_NAME..'.mk', templateDict)
    U.copyTemplateFile(Target.Variables.TARGET_ROOT..'/templates/cg.mk',
                       installDir..'/cg.mk', templateDict)
    U.copyTemplateFile(familySrcDir..'/../shrd/templates/main.c',
                       installDir..'/'..Target.Variables.BASE_NAME..'_main.c',
                       templateDict)
    U.copyTemplateFile(familySrcDir..'/../shrd/templates/plx_defines.h',
                       installDir..'/plx_defines.h', templateDict)
  end

  return cf_modelHooksRecipe
end

return Coder
